/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/spana.c,v $
 * $Id: spana.c,v 3.10 1999/07/20 21:09:06 heiner Exp $
 *
 *	solve special (sub-) problems
 *
 * Ideas:
 * - when A has K+F, defender strategy: move just K:
 *   try to show K+F cannot cover K-escapes [normal only]
 */

#include "types.h"
#include "board.h"
#include "output.h"
#include "job.h"
#include <stdio.h>
#include "stats.h"
#include "spana.h"

#if PROD_LEV
# define SPANA_STATS	0
#else	/* !PROD_LEV */
# ifndef SPANA_STATS
#  define SPANA_STATS	1		/* CF: local statistics */
# endif
#endif	/* !PROD_LEV */

#undef  THIS_STATS
#define THIS_STATS	SPANA_STATS
#include "statsupp.h"

#if SPANA_STATS
					/* MDB = min depth, B only */
static Counter	 sc_mdb;
static Counter	 sc_mdb_done;
static Counter	 sc_mdb_base[ ANA_MANY + 1 ];
static Counter	 sc_mdb_resu[ ANA_MANY + 1 ];
#endif	/* SPANA_STATS */


/*
 * Optionally print, then initialize analysis statistics gathered here.
 */
    Eximpl void			/*ARGSUSED*/
spana_stats( Bool withprint )
{
#if SPANA_STATS
    register int	i;

    if( withprint && (f_stats > 0) ) {
	if( sc_mdb ) {
	    printf("mdb:");
	    show_scs(1,9, sc_mdb     , " calls,");
	    show_scs(1,9, sc_mdb_done, " done");
	    printf(" [%5.1f%%]\n", percent(sc_mdb_done, sc_mdb));
	    for( i=0 ; i<(ANA_MANY+1) ; ++i ) {
		if( sc_mdb_base[i] || sc_mdb_resu[i] ) {
		    printf("mdb in/out %2d:", i);
		    show_nz(0,10, '-', sc_mdb_base[i]);
		    show_nz(1,10, '-', sc_mdb_resu[i]);
		    printf(" [%6.2f%%]\n", percent(sc_mdb_resu[i], sc_mdb));
		}
	    }
	}
    }
    sc_mdb      = 0;
    sc_mdb_done = 0;
    for( i=0 ; i<(ANA_MANY+1) ; ++i ) {
	sc_mdb_base[i] = 0;
	sc_mdb_resu[i] = 0;
    }
#endif	/* SPANA_STATS */
}


/* ============================================================================
 * Special chess knowledge for KXK jobs.
 */

#if 1
/*
 * We have an odd-ply-depth job, where the colour to move has to mate
 * or stalemate the other side.  The side to move has one piece left,
 * and the other side has just the bare king.
 * While we could have endgame tables for this, we still can make
 * some analysis much cheaper than reading from a file.  Also,
 * we can say something about help-jobs, self-jobs and stalemate jobs,
 * for which we normally have no endgame tables.
 * We may return at most ANA_MANY+1, indicating "job impossible"
 */
#define KXK_INF	(1<<0)		/* =Inf */
#define KXK_R	(1<<1)
#define KXK_E	(1<<2)
#define KXK_H	(1<<3)
#define KXK_h	(1<<4)
#define KXK_1MX	(1<<5)
#define KXK_U	(1<<6)

#define KXK_RE	(KXK_R  | KXK_E)
#define KXK_RH	(KXK_R  | KXK_H)
#define KXK_Rh	(KXK_R  | KXK_h)
#define KXK_1Rh	(KXK_Rh | KXK_1MX)
#define KXK_1RE	(KXK_RE | KXK_1MX)
#define KXK_RHU	(KXK_RH | KXK_U)
#define KXK_REH	(KXK_RE | KXK_H)

#if 0
/*
E  = D-K must be in a corner (implies R)
R  = D-K must be at a border
H  = A-K must help (be near D-K)
U  = A-B must promote, first
Rh = either RH or RE (take min of both)

Mat	#	h#	s#	Patt	h-Patt	s-Patt
KBK	RHU	RHU	RHU	Rh	Rh	Rh
KSK	=Inf	=Inf	=Inf	RE<=1	RE	RE
KLK	=Inf	=Inf	=Inf	Rh<=1	Rh	Rh
KTK	RH	RH	RH	REH	REH	REH
KDK	RH	RH	RH	Rh	Rh	Rh
*/
#endif
		/* [BSLTD]            B        S        L        T        D */
static const uint8	a_m [5] ={ KXK_RHU, KXK_INF, KXK_INF, KXK_RH , KXK_RH };
static const uint8	a_hm[5] ={ KXK_RHU, KXK_INF, KXK_INF, KXK_RH , KXK_RH };
static const uint8	a_sm[5] ={ KXK_RHU, KXK_INF, KXK_INF, KXK_RH , KXK_RH };
static const uint8	a_p [5] ={ KXK_Rh , KXK_1RE, KXK_1Rh, KXK_REH, KXK_Rh };
static const uint8	a_hp[5] ={ KXK_Rh , KXK_RE , KXK_Rh , KXK_REH, KXK_Rh };
static const uint8	a_sp[5] ={ KXK_Rh , KXK_RE , KXK_Rh , KXK_REH, KXK_Rh };

#if 0
  static const uint8	brd__dist[8] ={ 0,1,2,3,3,2,1,0 };
# define border_dist(x)	brd__dist[x]
#else
    /* Implementing small table of 2-bit numbers in a 16-bit constant.
     * 00 01 10 11 11 10 01 00
     *   1     b     e     4
     */
# define border_dist(x)	((0x1be4U >> (((unsigned)(x))<<1)) & 03)
#endif


    static int			/* distance to border */
kd_R( unsigned pos64 )
{
    int		dx;
    int		dy;

    dx = border_dist(COL64(pos64));
    dy = border_dist(LIN64(pos64));
    return (dx < dy) ? dx : dy;		/* min(dx,dy) */
}


    static int			/* distance to corner */
kd_E( unsigned pos64 )
{
    int		dx;
    int		dy;

    dx = border_dist(COL64(pos64));
    dy = border_dist(LIN64(pos64));
    return (dx < dy) ? dy : dx;		/* max(dx,dy) */
}


    Eximpl int			/* 0 | -maxdepth | +mindepth */
kxk_mindep( register Xconst Board* bp )
{
    int		tom;
    int		inx;
    uint8	attr;
    unsigned	ak64;		/* to move, shall (stale)mate */
    unsigned	dk64;		/* shall be (stale)mated */

    tom = bp->b_tomove;
    {
	    register const int8*	cntp;
	cntp = bp->b_fig_cnt[tom];
	if(      cntp[bauer   ] ) inx = 0;
	else if( cntp[springer] ) inx = 1;
	else if( cntp[laeufer ] ) inx = 2;
	else if( cntp[turm    ] ) inx = 3;
	else if( cntp[dame    ] ) inx = 4;
	else goto dunno;			/* huh? sorry */
    }
    switch( f_jobtype ) {
     case JT_normal:		attr = a_m [inx]; break;
     case JT_stalemate:		attr = a_p [inx]; break;
     case JT_selfmate:		attr = a_sm[inx]; break;
     case JT_selfstalemate:	attr = a_sp[inx]; break;
     case JT_helpmate:		attr = a_hm[inx]; break;
     case JT_helpstalemate:	attr = a_hp[inx]; break;
     default:			goto dunno;	/* sorry */
    }
    if( attr & KXK_INF ) {
	goto nosolve;		/* cannot solve at all */
    }
    ak64 = K_FP(bp,            tom )->f_pos64;
    dk64 = K_FP(bp, opp_colour(tom))->f_pos64;

    if( attr & KXK_1MX ) {
	if( attr & KXK_E ) {
	    if( kd_E(dk64) > 0 ) goto nosolve;
	}else if( attr & KXK_R ) {
	    if( kd_R(dk64) > 0 ) goto nosolve;
	}
	return -1;		/* at most depth 1 */
    }

    if( (attr & (KXK_R | KXK_E)) && (attr & (KXK_H | KXK_h)) ) {
	/*
	 * need border, may be even corner.
	 * need support either unconditionally, or alternately
	 * to corner.  Hence we want to compute RH.  See "MEMO.trivEG".
	 */
	    int	akx;
	    int	aky;
	    int	dkx;
	    int	dky;
	    int	vxlo;
	    int	vxhi;
	    int	vylo;
	    int	vyhi;
	    int	dx;
	    int	dy;
	    int	d;
	    int	e;
	dkx = COL64(dk64); dky = LIN64(dk64);
	akx = COL64(ak64); aky = LIN64(ak64);
	/*
	 * n==1: 1 + ...
	 *  max(
	 *    min(
	 *	vxlo =     max(aK.x-3, dK.x)
	 *	vxhi = 7 - min(aK.x+3, dK.x)
	 *	vylo =     max(aK.y-3, dK.y)
	 *	vyhi = 7 - min(aK.y+3, dK.y)
	 *    )
	 *    |aK.x - dK.x|/2 - 1
	 *    |aK.y - dK.y|/2 - 1
	 *  )
	 */
	dx = akx - dkx;
	if( dx < 0 ) dx = -dx;
	dx >>= 1; dx -= 1;
	dy = aky - dky;
	if( dy < 0 ) dy = -dy;
	dy >>= 1; dy -= 1;
	if( dx < dy ) dx = dy;
	if( dx < 0  ) dx = 0;
	vxlo = akx-3; if( dkx > vxlo ) vxlo = dkx;
	vxhi = akx+3; if( dkx < vxhi ) vxhi = dkx; vxhi = 7-vxhi;
	vylo = aky-3; if( dky > vylo ) vylo = dky;
	vyhi = aky+3; if( dky < vyhi ) vyhi = dky; vyhi = 7-vyhi;
	if( vxhi < vxlo ) vxlo = vxhi;
	if( vyhi < vylo ) vylo = vyhi;
	if( attr & KXK_E ) {	/* unconditional corner */
	    d = ((vxlo > vylo) ? vxlo : vylo);		/* REH */
	}else {
	    d = ((vxlo < vylo) ? vxlo : vylo);		/* RH */
	    if( (attr & KXK_h) && (d > dx) ) {
		e = kd_E(dk64);		/* alternatively */
		if( e < d ) d = e;	/* get minimum */
	    }
	}
	if( d < dx ) d = dx;
#if 0
	e = kd_R(dk64);
	if( e > d ) {
	    put_packed(&(bp->b_packed), "kxk: ");
	    printf("Is %d, should be %d\n", d, e);
	}
#endif
#if 0
	put_packed(&(bp->b_packed), "kxk: ");
	printf("--> %d\n", 1+d);
#endif
	return 1 + d;
    }
    if( attr & KXK_E ) {
	return 1 + kd_E(dk64);
    }
dunno:;
    return 0;			/* no knowledge implemented */
nosolve:;
    return ANA_MANY+1;		/* cannot solve at all */
}
#endif

/* ============================================================================
 * Special chess knowledge for normal mate and attacker with K and Bs, only.
 */


#if PROD_LEV
# define DBG_K4	0
# define IF_V(v,x)	{/*empty*/;}
#else
# define DBG_K4	0	/* CF: makes min_d_b_k4mate verbose */
# define IF_V(v,x)	if( (v) ) { x }
#endif


/* When a defender K has a 2x2 escape block, we want it via directions ...
 */
static int8	sq_dirs[4][4];
static Flag	sq_dirs_inited	= FALSE;

    static int
step2damdir( register int delta )
{
    register int	dir;

    for( dir=MIN_D_DIR ; dir<MAX_D_DIR ; ++dir ) {
	if( dam_mov[dir] == delta ) {
	    return dir;
	}
    }
    return NO_DIR;
}

    static void
init_1_sq_dirs( int inx, int d1, int d2 )
{
    sq_dirs[inx][0] = d1;
    sq_dirs[inx][1] = d2;
    sq_dirs[inx][2] = step2damdir(dam_mov[d1]+dam_mov[d2]);
    sq_dirs[inx][3] = ZERO_DIR;
}

    static void
init_sq_dirs(void)
{
    if( ! sq_dirs_inited ) {
	/*
	 * There are 4 2x2 squares constructable from a position.
	 * We choose a T-direction, and both of its orthogonal
	 * directions.  Summing both steps yields the 4th position.
	 * With the opposite of the chosen T-direction we do the same.
	 */
	init_1_sq_dirs(0,         MIN_T_DIR ,         ort_dir(MIN_T_DIR) );
	init_1_sq_dirs(1,         MIN_T_DIR , opp_dir(ort_dir(MIN_T_DIR)));
	init_1_sq_dirs(2, opp_dir(MIN_T_DIR),         ort_dir(MIN_T_DIR) );
	init_1_sq_dirs(3, opp_dir(MIN_T_DIR), opp_dir(ort_dir(MIN_T_DIR)));
	sq_dirs_inited = TRUE;
    }
}

#define CHK_SQ_DIRS()	if( sq_dirs_inited );else init_sq_dirs()


    static int			/* minimum number of steps needed by */
min_d_k(
    register const Field*	kp,	/* some K at this field, to step */
    register const Field*	tp)	/* to this field */
{
    register int		dx;
    register int		dy;

    dx = COL64(kp->f_pos64) - COL64(tp->f_pos64);	/* col delta */
    dy = LIN64(kp->f_pos64) - LIN64(tp->f_pos64);	/* lin delta */
    if( dx < 0 ) dx = -dx;				/* absolute value */
    if( dy < 0 ) dy = -dy;				/* absolute value */
    return (dx > dy) ? dx : dy;				/* maximum */
}


    static int				/* minimum, fastest way for att */
min_d_b_k4mate(
    register const Board*	bp,
    register rColour		att,
    register const Field*	fp,		/* assume attB here */
    int				nescs,		/* #(non-NIL in escs), 2 | 4 */
    register Xconst Field*	escs[4],	/* defK at one of these */
    register const Field*	akp,		/* NIL | assume attK here */
    const Field*		ignattb,	/* NIL, or considered beaten */
    register int		mindist,	/* already achieved minimum */
    int				v)		/* verbose: debugging */
{
    register const Field*	tp;
    register int		dist;		/* collected att-steps */
    register int		lin;
    register int		i;
    register int		d;
    register rColour		def;
    int				promlin;
    int				baslin;
    Bool			lastfrombaslin;

#if DBG_K4
    IF_V(v, printf("{"); )
#endif
    dist           = 0;			/* att did not yet move */
    promlin        = PROM_LIN(att);	/* there he may promote */
    baslin         = BAS_LIN(att);	/* from there a double step is ok */
    def            = opp_colour(att);	/* defender */
    lastfrombaslin = FALSE;
    /*
     * Now we enumerate the chances of the att to move with a B on "fp".
     * Thus we increase "dist", until we find a real value,
     * which then is used to minimize "mindist".
     */
    while( dist < mindist ) {		/* may improve minimum */
#if DBG_K4
	IF_V(v, 
	    printf("["); put_position(fp-bp->b_f); printf(":%d]", dist);
	)
#endif
	lin = LIN64(fp->f_pos64);	/* att-B currently at this line */
					/* promotion ? */
	if( lin == promlin ) {
	    /*
	     * With "fp" we have already reached the promotion line.
	     * Hence we can instantly become an S or D, which may check
	     * or even mate.
	     * In order to be a mate, all the escapes must possibly
	     * be covered, now, or else we may add a move.
	     * If we still know, where the att-K is, he cannot yet
	     * attack any of the escapes, else he can halve them.
	     *
	     * If none of the escapes is attackable by S or D,
	     * we can add 1 move.
	     */
		int	dirs[4];
	    dirs[0] = att_dir(fp, escs[0]);
	    dirs[1] = att_dir(fp, escs[1]);
	    if( nescs == 2 ) {
		if( akp ) {
		    if( no_dir(dirs[0])
		     || no_dir(dirs[1])
		     || (dam_dir(dirs[0]) != dam_dir(dirs[1])) ) {
			dist += 1;	/* cannot cover both with attack */
		    }
		}else {			/* att-K may attack 1, already */
		    if( !dam_dir(dirs[0])
		     && !dam_dir(dirs[1]) ) {
			dist += 1;	/* cannot check immediately */
		    }
		}
	    }else {			/* nescs == 4 */
		dirs[2] = att_dir(fp, escs[2]);
		dirs[3] = att_dir(fp, escs[3]);
		if( akp ) {
		    /*
		     * There are still 4 fields to be attacked.
		     * An S cannot immediately.  A D may.
		     */
		    if( !dam_dir(dirs[0])
		     || !dam_dir(dirs[1])
		     || !dam_dir(dirs[2])
		     || !dam_dir(dirs[3])
		      ) {
			dist += 1;	/* cannot mate immediately */
		    }
		}else {			/* att-K may take up to 2 */
		    /*
		     * There are always still two fields of opposite
		     * field colour to be attacked by the promoting B.
		     * Hence only a D or T promotion may be able to do it
		     * immediately, not S.  Else, one more move is added.
		     */
		    if( (  dam_dir(dirs[0])
			 + dam_dir(dirs[1])
			 + dam_dir(dirs[2])
			 + dam_dir(dirs[3])
			) < 2
		      ) {
			dist += 1;	/* cannot mate immediately */
		    }
		} /* ! akp */
	    } /* nescs == 4 */
	    if( dist < mindist ) mindist = dist;
	    goto rdy;			/* this B does not exist anymore */
	} /* att-B promotes at fp */
					/* checking ? beating ? */
	for( i=0 ; i<2 ; ++i ) {
	    tp = fp + (i ? bau_right : bau_left)[att];
	    /*
	     * Currently there is a B-attack to "tp".
	     * It may attack one of the escapes, or may be used
	     * to beat a def-piece.
	     */
	    if( (tp == escs[0]) || (tp == escs[1])
	     || (tp == escs[2]) || (tp == escs[3]) ) {	/* may check */
		if( akp ) {	/* cannot yet disturb */
		    dist += 1;	/* defK forced out, or may do its move */
		}
		if( dist < mindist ) mindist = dist;
		goto rdy;
	    }
	    if( (tp->f_c == def) && (tp->f_f != koenig) ) {
		/*
		 * With the next step our att-B may beat some def-piece.
		 * We handle the beat by a recursive call.
		 */
		    int	basedist	= dist+1;
		if( basedist < mindist ) {	/* may contribute to min */
		    d = basedist + min_d_b_k4mate(bp, att, tp, nescs, escs,
						  akp, ignattb,
						  mindist - basedist, v);
		    if( d < mindist ) mindist = d;
		}
	    }
	}
	/*
	 * Promotion, checking and beating is handled, leaving single
	 * and double steps of the att-B.
	 */
	tp = fp + bau_mov[att];		/* single B-step */
	if( (tp->f_c != empty) && (tp != ignattb) ) {	/* blocked, somehow */
	    if( tp->f_c == att ) {	/* own figure blocking */
		if( tp->f_f == koenig ) {
		    if( akp ) {
			dist += 1;	/* K makes free: 1 more move */
			akp = 0;	/* somewhere else, now */
		    }
		}else {			/* must be B, which is faster */
		    goto rdy;		/* so stop this thread */
		}
	    }else if( tp->f_c == def ) {
		if( tp->f_f != koenig ) {	/* def-K moves around */
		    /*
		     * Some defender, but not the K, is blocking the att-B.
		     * Either another att-B must beat it, but then that one
		     * will be faster, and our thread is not minimal,
		     * and need not be completed in that version.
		     * Or, the att-K is needed to put away the def-piece.
		     * But, a K will never beat a D, which just sits there.
		     */
		    if( tp->f_f == dame ) {	/* cannot be approached by K */
			goto rdy;		/* so this thread is dead */
		    }
		    if( akp ) {		/* att-K still known to be there */
			dist += min_d_k(akp, tp);	/* att-K -> tp */
			dist += 1;	/* att-K leaving tp, then */
			akp = 0;	/* somewhere else, now */
		    }
		}
	    }else {		/* should not happen */
		goto rdy;
	    }
	}
	/*
	 * Now, when the att-B moves onto one of the two or four def-K
	 * escapes, and we still know the att-K on its original place,
	 * then we can conclude, that the att-B will be beaten,
	 * except it is supported, somehow.
	 * Support may be by att-K, or by another att-B, possibly promoting.
	 * If we do not know the att-K place, it may already support, here.
	 */
#if DBG_K4
	IF_V(v, if( akp ) printf("A"); )
#endif
	if( akp				/* else may support, already */
	 && (   (tp == escs[0]) || (tp == escs[1])
	     || (tp == escs[2]) || (tp == escs[3])
	    )
	  ) {				/* B may vanish */
	    d = min_d_k(akp, tp) - 1;	/* att-K moves to support */
	    if( d > 0 ) {		/* att-K not yet supporting at tp */
		if( (bp->b_fig_cnt[att][bauer] - !!ignattb) > 1 ) { /* more B */
		    dist += 1;		/* somehow supporting the att-B */
		}else {			/* there is no B-support possible */
		    dist += d;		/* att-K support */
		}
		akp = 0;		/* cannot guarantee him anymore */
	    }
	}
	fp = tp;			/* move the att-B the single step */
	if( lastfrombaslin ) {		/* could have done a double step */
	    lastfrombaslin = FALSE;	/* next time it is not true anymore */
	}else {
	    dist += 1;			/* account for the single step */
	    if( lin == baslin ) {
		lastfrombaslin = TRUE;	/* remember possible double step */
	    }
	}
    }
rdy:;
#if DBG_K4
    IF_V(v, printf("%d}", mindist); )
#endif
    return mindist;
}


    static int			/* minimum (fastest) over all att B-lines */
mdb_minim_att_b(
    register const Board*	bp,		/* attacker to move and mate */
    register rColour		att,
    int				nescs,		/* #(non-NIL in escs), 2 or 4 */
    register Xconst Field*	escs[4],	/* defK at one of these */
    register const Field*	akp,		/* attK is here */
    const Field*		ignattb,	/* NIL, or considered beaten */
    register int		mindist,	/* already achieved minimum */
    int				maxdist,	/* maximum of those minima */
    int				v)		/* verbose: debugging */
{
    register const Field*	tp;
    register int		ifig;
    register int		minifig;
    register int		pos;
    register int		d;

    IF_V(v, 
	printf("\n\t");
	for( d=0 ; d<nescs ; ++d ) {
	    put_position(escs[d]-bp->b_f);
	}
	if( ignattb ) {
	    printf(" -"); put_position(ignattb-bp->b_f);
	}
	printf(":");
    )
    minifig = COLOUR_IDX(att);
    for( ifig = minifig + bp->b_max_piece[att] - 1
       ; ifig >= minifig
       ; --ifig ) {
	pos = bp->b_piece[ifig];
	if( no_pos(pos) )
	    continue;	/* skip empty slot (beaten piece) */
	tp = &(bp->b_f[pos]);
	if( (tp->f_f != bauer)		/* consider Bs, only */
	 || (tp == ignattb)		/* that one is considered beaten */
	  ) {
	    continue;
	}
	d = min_d_b_k4mate(bp, att, tp, nescs, escs, akp, ignattb,
								mindist, v/2);
	IF_V(v, 
	    printf(" "); put_position(tp-bp->b_f); printf("=%d", d);
	)
	if( d < mindist ) {
	    mindist = d;
	    if( mindist <= maxdist ) {	/* will not contribute to maximum */
		break;
	    }
	}
    }
    return mindist;		/* reduced minimum */
}


#if 1
    static Bool			/* whether just now an B */
mdb_attb_can1reach(
    register rColour		att,	/* of this attacker (to move) */
    register const Field*	tp)	/* can move/beat to this field */
{
    register const Field*	fp;	/* proposed from-field */

    if( tp->f_c == empty ) {			/* test move */
	/* NOTE: e.p. ruled out globally */
	fp = tp - bau_mov[att];
	if( fp->f_c == empty ) {
	    fp = tp - bau_mov[att];
	    if( (fp->f_c == att) && (fp->f_f == bauer)
	     && (LIN64(fp->f_pos64) == BAS_LIN(att)) ) {
		return TRUE;	/* found double step move */
	    }
	}else {
	    if( (fp->f_c == att) && (fp->f_f == bauer) ) {
		return TRUE;	/* found single step move */
	    }
	}
    }else if( tp->f_c == opp_colour(att) ) {	/* test beat */
	fp = tp - bau_left[att];
	if( (fp->f_c == att) && (fp->f_f == bauer) ) {
	    return TRUE;	/* found beat move */
	}
	fp = tp - bau_right[att];
	if( (fp->f_c == att) && (fp->f_f == bauer) ) {
	    return TRUE;	/* found beat move */
	}
    }
    return FALSE;		/* no such move found */
}


    static Bool			/* whether we guarantee, that */
mdb_defk_canbeat(
    Xconst Board*		bp,	/* within this board */
    const Field*		dkp,	/* this def-K will beat within 1 move */
    register Xconst Field*	fp,	/* this att-B (party to move) */
    const Field*		akp)	/* where the att-K is here */
{
    register rColour		att;
    register rColour		def;
    register Xconst Field*	tp;
    register rPieceSet		amask;

    att   = akp->f_c;
    def   = opp_colour(att);
    amask = COLOUR_MASK(att);
    if( F_DATT(fp) & amask ) {
	return FALSE;		/* already supported */
    }
    if( ! (F_DATT(fp) & SET1(dkp->f_idx)) ) {
	return FALSE;		/* not yet reached by def-K */
    }
    if( min_d_k(akp, fp) <= 2 ) {
	/*FFS*/
	return FALSE;		/* may be supported by att-K in 1 move */
    }
    if( (fp[bau_left [att]].f_c == def)
     || (fp[bau_right[att]].f_c == def) ) {
	return FALSE;		/* att-B may beat: out of reach */
    }
    /*
     * Check single and double step of att-B:
     */
    tp = fp + bau_mov[att];
    if( tp->f_c == empty ) {	/* needed for both of them */
	if( ! (F_DATT(tp) & SET1(dkp->f_idx)) ) {
	    return FALSE;	/* def-K cannot beat back */
	}
	if( F_DATT(tp) & amask ) {
	    return FALSE;	/* def-K is not allowed to beat back */
	}
	if( LIN64(fp->f_pos64) == BAS_LIN(att) ) {
	    tp += bau_mov[att];
	    if( tp->f_c == empty ) {
		return FALSE;	/* att-B has double step: out of reach */
	    }
	}
    }
    /*
     * Check support by other att-B in one move:
     */
    if( bp->b_fig_cnt[att][bauer] > 1 ) {	/* exist other att-B */
				/* check promotions */
	for( tp = &(bp->b_f[ MK_POS(0,BAS_LIN(def)) ])
	   ; tp->f_c != border
	   ; tp += MK_DELTA(1,0)
	   ) {
	    if( (tp->f_c == att) && (tp->f_f == bauer) ) {
		    register const Field*	p;
		p = tp + bau_mov[att];
		if( (p->f_c == empty) && ! no_dir(att_dir(p,fp)) ) {
		    return FALSE;	/* supported by promotion */
		}
		p = tp + bau_left[att];
		if( (p->f_c == def) && ! no_dir(att_dir(p,fp)) ) {
		    return FALSE;	/* supported by promotion */
		}
		p = tp + bau_right[att];
		if( (p->f_c == def) && ! no_dir(att_dir(p,fp)) ) {
		    return FALSE;	/* supported by promotion */
		}
	    }
	}
	if( mdb_attb_can1reach(att, fp-bau_left [att])
	 || mdb_attb_can1reach(att, fp-bau_right[att]) ) {
	    return FALSE;		/* supported by another att-B */
	}
    }
    return TRUE;		/* no objections found */
}


    static int		/* (possibly) improved maximum */
mdb_try_defstrat(
    Xconst Board*	bp,
    int			nescs,		/* #(non-NIL in 'escs'): 2 or 4 */
    Xconst Field*	escs[4],	/* last filled one is 'dkp' */
    PieceSet		attbset,	/* those att-B attack at the escs (<2)*/
    const Field*	akp,		/* att-K currently here */
    int			maxdist,	/* old maximum, to be improved */
    int			v)
{
    int			att;
    int			mindist;
    Xconst Field*	ignattb;
    const Field*	dkp;
    Bool		done;
    int			d;

    /*
     * We know, already, that none of the proposed escapes is
     * a border, already attacked by att-K, or blocked by a def-piece.
     * Some may be occupied by att-B, and there may be some
     * with attacks of some att-B, which we then check, whether
     * we can guarantee him to be beaten in the next move by def-K.
     */
    att = akp->f_c;
    dkp = escs[nescs-1];	/* last is always origin */
    if( attbset ) {		/* must be beaten (else not really escapes) */
	    register int	idx;
	    register int	zeroes;
	idx = 0;
	while( attbset & 01 ) {
	    zeroes = MIN_LOW_ZERO(attbset);
	    attbset >>= zeroes; idx += zeroes;
	}
	ignattb = &(bp->b_f[ bp->b_piece[idx] ]);
	if( ! mdb_defk_canbeat(bp, dkp, ignattb, akp) ) {
	    goto out;		/* must beat, but cannot */
	}
    }else {
	ignattb = (Field*)0;
    }
    /*
     * Ok, 'escs' are true escapes.
     * Check, whether the att-K can force the def-K out of his strategy:
     */
    if( nescs == 2 ) {		/* att-K may force def-K out */
	    int	kkd0;
	    int	kkd1;
	kkd0 = min_d_k(akp, escs[0]);
	kkd1 = min_d_k(akp, escs[1]);	/* current K-K distance */
	/*
	 * As both are not yet attacked by att-K, the above distances
	 * must be at least 2.
	 * With (dist-2) steps of att-K, he comes 2 near to a field.
	 * One further move may attack it, if not currently occupied
	 * by the def-K.  One further move may say check/mate, as
	 * the def-K is within his escapes, anymore.
	 * Thus, (dist) att-moves are necessary.
	 * After an even # of moves (e.g. 0), the def-K is at escs[1].
	 * After an odd  # of moves (e.g. 0), the def-K is at escs[0].
	 * This may force another tempo.
	 */
	if( kkd0 < kkd1 ) {
	    mindist = kkd0;	/* occupied after odd number of moves */
	    if( mindist & 01 ) {
		mindist += 1;	/* needs one tempo more */
	    }
	}else if( kkd0 > kkd1 ) {
	    mindist = kkd1;	/* occupied after even number of moves */
	    if( ! (mindist & 01) ) {
		mindist += 1;	/* needs one tempo more */
	    }
	}else {			/* both equally far away: att-K can choose */
	    mindist = kkd0;
	    /*
	     * FFS: the att-K may have no fast path, as he would need to
	     *      walk through the def-K.
	     */
	}
	if( mindist <= maxdist ) {
	    goto out;		/* will not contribute to maximum */
	}
    }else {			/* nescs == 4 */
	mindist = ANA_MANY;	/* att-K cannot force def-K */
    }
    /*
     * Now, that we have the basic value for mindist,
     * we go on to minimize it according to the att-B capabilities.
     * These may be reduced, if we not yet have an ignattb,
     * but there is one to be beaten immediately:
     */
    done = FALSE;
    if( ! ignattb ) {
	    register Xconst Field*	tp;
	    register int		einx;
	for( einx = nescs-2 ; einx>=0 ; --einx ) {
	    tp = escs[einx];
	    if( (tp->f_c == akp->f_c) && (tp->f_f == bauer) ) {
		IF_V(v, 
		    printf(" ["); put_position(dkp-bp->b_f);
		    printf("x"); put_position(tp -bp->b_f);
		    printf("=");
		)
		if( mdb_defk_canbeat(bp, dkp, tp, akp) ) {
		    IF_V(v, printf("y]"); )
		    done = TRUE;
		    d = mdb_minim_att_b(bp, att, nescs, escs, akp, tp,
						mindist, maxdist, v);
		    if( d > maxdist ) {	/* improves maximum */
			maxdist = d;
			IF_V(v, printf("!"); )
		    }
		}else {
		    IF_V(v, printf("n]"); )
		}
	    }
	}
    }
    if( ! done ) {
	d = mdb_minim_att_b(bp, att, nescs, escs, akp, ignattb,
						mindist, maxdist, v);
	if( d > maxdist ) {		/* improves maximum */
	    maxdist = d;
	    IF_V(v, printf("!"); )
	}
    }
out:;
    return maxdist;
}
#endif


    Eximpl int			/* possibly improved 'omindep' */
mate_mindep_bbb(
    register Xconst Board*	bp,		/* attacker to move and mate */
    int				omindep,	/* caller would try this */
    int				v)		/* verbose: debugging */
{
    register int		att;
    register Xconst Field*	dkp;
    register const Field*	akp;
    register int		maxmindep = 0;	/* not yet known anything */

    scinc(sc_mdb);
    att = bp->b_tomove;
    if( bp->b_fig_cnt[att][bauer] != (bp->b_cur_piece[att] - 1) ) {
	goto out;
    }
    /*
     * Guard against e.p. (just now) and castling of attacker.
     * As we try to move only the K of the defender, no more e.p. will
     * be possible then (within the scope of this analysis).
     * If the attacker cannot castle, his K cannot be faster then
     * a single step per move.
     */
    if( ! no_pos(bp->b_ep) || bp->b_castle[att] ) {
	goto out;
    }
    /*
     * The attacker (who is to move) has B, only (except his K).
     * We try to count, how many moves must be done by the attacker,
     * in order to say check.
     *
     * For this, we fix the moves of the defender in such a way,
     * that we can predict the attacker's possibilities to advance
     * his pawns, to say check directly or promote.
     *
     * (A) Fix the def-K switching between his current position,
     *     and one beneath it, which is currently free.
     *     Now we know all the time we look forward, exactly where
     *     the def-K is.
     * (B) Fix the def-K on four positions, forming a 2x2 square,
     *     containing the current position, such that all four
     *     positions currently are not attacked.
     *     Now, the def-K cannot be forced out of this square
     *     just by the att-K.  The attacker needs other means.
     *     This case is especially important, in case the both K
     *     are near, already, as (A) does not say anything valuable.
     *
     * The attacker will not be able to say check, before one of the
     * following conditions occur:
     * - a B attacks def-K (check)
     * - a B promotes
     * - a B attacks one of the fixed K-positions, forcing def-K out
     *   of them (now the premise fails and analysis stops)
     * - a K attacks one of the fixed K-positions, such that he
     *   forces def-K out.
     *
     * General sketch of algorithm:
     *  L = List of all def-K moving strategies
     *  return maximum(kms in L: min_k_moves(kms))
     *  min_k_moves(kms) =
     *    min( min_to_promote_and_check(kms),
     *         min_to_check_directly(kms),
     *         min_to_stop_def_strat_by_B(kms),
     *         min_to_stop_def_strat_by_K(kms) )
     *
     * (A)
     * We try to find two positions for the defender K, on which he may
     * stay (we assume he does so).
     * In order to say check, either some moveable B must do so,
     * or some B must be promoted, at least.
     *
     * (S) Special consideration, to catch some annoying cases:
     *     When there is one att-B within the planned 2 or 4 escape
     *     positions of the def-K, sometimes we can guarantee, that
     *     it will be beaten with the next def-K move:
     *     (Sa) the att-B is not yet defended by some other att-piece,
     *     (Sb) the att-B cannot be defended by the att move to come
     *          (neither by att-K, nor by another att-B, nor by a promotion).
     *     (Sc) the att-B cannot beat any def-piece, now,
     *     and either
     *     (Sd) the att-B currently cannot step forward,
     *      or
     *     (Se) if the att-B does a (single) step, it can immediately
     *          be beaten by def-K within its escape fields.
     *     In such a case, the attacks by that att-B are not stopping the
     *     the def-K.  Also, that att-B will not be able to contribute
     *     to any further check or mate move.
     *
     * Further ideas (FFS):
     *	- when all attB are blocked by defender, and cannot beat due
     *	  to missing beat target, it is not enough, to force the defK
     *	  out of his position.  Instead, the attK must beat a blocker,
     *	  or force the defK to beat an attB.  Then, things may change.
     *	- When all attB are blocked as above, and the attK is (or can be)
     *	  boxed (e.g. by a defD) into a part of the board, from where
     *	  he cannot reach a beat move, nor force the defK to beat ...
     *	  then the attK cannot succeed to find a check in any depth.
     *	- When a promotion is guarded by the defK (i.e. he blocks or beats)
     *	  then either support by the attK is needed, or another attB must
     *	  be used to threaten something.
     *
     * Find K-positions, then scan the attacker B-pieces ...
     */
    scinc(sc_mdb_done);
    akp = K_FP(bp,            att );
    dkp = K_FP(bp, opp_colour(att));
    maxmindep = omindep;	/* maximum to be improved */
    {
	    register Xconst Field*	tp;
	    register int		dir;
	    register rPieceSet		amask;
	    register rPieceSet		aatts;
	    register int		dinx;
	    register int		sq;
	    Xconst Field*		escs[4];
	amask = COLOUR_MASK(att);
	aatts = 0;
	escs[1] = dkp;
	escs[2] = 0;
	escs[3] = 0;
	for( dir=MIN_D_DIR ; dir<MAX_D_DIR ; ++dir ) {
	    tp = dkp + dam_mov[dir];
	    if( (tp->f_c == border)
	     || (tp->f_c == opp_colour(att))
	     || (F_DATT(tp) & amask)
	      ) {
		continue;	/* unusable escape */
	    }
	    escs[0] = tp;
	    maxmindep = mdb_try_defstrat(bp, 2, escs, aatts, akp, maxmindep, v);
	}
	CHK_SQ_DIRS();
	escs[3] = dkp;			/* origin is always the last one */
	for( sq=0 ; sq<4 ; ++sq ) {	/* 4 2x2 squares */
	    aatts = 0;			/* collects bad att-B */
	    for( dinx=0 ; dinx<3 ; ++dinx ) {
		tp = dkp + dam_mov[ sq_dirs[sq][dinx] ];
		if( (tp->f_c != empty) && (tp->f_c != att) ) {	/* blocked */
		    goto next_sq;
		}
		if( F_DATT(tp) & amask & BOTH_KINGS ) {	/* attacked */
		    goto next_sq;
		}
		aatts |= F_DATT(tp) & amask;
		escs[dinx] = tp;
	    }
	    /* We found a possible 2x2 square, where def-K cannot
	     * be forced out by att-K.  But, there still may be bad att-B.
	     */
	    if( min2elems(aatts) ) {
		goto next_sq;
	    }
	    maxmindep = mdb_try_defstrat(bp, 4, escs, aatts, akp, maxmindep, v);
    next_sq:;
	}
    }
out:;
    if( maxmindep > ANA_MANY ) {
	maxmindep = ANA_MANY;
    }
    scinc(sc_mdb_base[omindep]);
    if( maxmindep > omindep ) {	/* improves lower bound: will be used */
	scinc(sc_mdb_resu[maxmindep]);
    }
    IF_V(v, printf(" -> %d\n", maxmindep); )
    return maxmindep;
}


#if 0
/* ============================================================================
 * Special chess knowledge for normal mate and attacker with K and L, only.
 */
    static int
mate_mindep_KL( bp, v )
    register Board*	bp;
{
    /*
     * Provided that the Attacker (A) has just one K and one L,
     * we find a minimum number of moves necessary to say check.
     * Idea: show that the defending K (D.K) has 2 fields
     * of the opposite color as A.L, on which he can stay,
     * until forced away by the A.K (by attacking one of them).
     *
     *	(a) find move for D.K onto oppcolor (not blocked) (field f1)
     *	(b) find further move on same color (field f2)
     *	(c) show both fields to be far away from A.K:
     *	    then A.K cannot hinder D.K to do the moves.
     *	If f1 and f2 are found, such that A not yet attacks them,
     *	    d1 := d(f1, A.K)	[>=2]
     *	    d2 := d(f2, A.K)	[>=2]
     *	White can say check not faster than in
     *	    min( d1, d2 ) - 1	[attack f1 or f2]
     *	    + 1			[say check]
     *	==  min( d1, d2 )
     *	Furthermore, A.K cannot attack where the D.K just is.
     *	A.K first attacks f1 in d1-1 moves.
     *	If D.K before that (after d1-2 moves) is on f1,
     *	A.K needs one more move.
     *	D.K is on f1 after odd #(moves).
     *	D.K is on f2 after even #moves (> 0 except on f2, already).
     *	    D1 := d1 + (D.K on f1 after d1-2 moves)
     *	       == d1 + odd(d1)
     *	    D2 := d2 + (D.K on f2 after d2-2 moves)
     *	       == d2 + (even(d2) && (D.K@f2 || (d2-2)>0))
     *	==  min( D1, D2 )
     *	Compute max(all f1,f2: min(D1,D2)): this is lower limit
     *	for forcable check and hence forcable mate.
     *
     * FFS: it is sufficient for the above, that A has only L of same color.
     */
    register int	mindep	= 0;	/* not yet known anything */
    register int	att;

    att = bp->b_tomove;
    if( bp->b_cur_piece[att] != 2 ) {
	goto out;	/* A has more than two pieces */
    }
    if( bp->b_fig_cnt[att][laeufer] != (bp->b_cur_piece[att] - 1) ) {
	goto out;	/* A has not only L */
    }
    /*
     * Find the A.L and determine the color of its field ...
     */
    FFS
out:;
    if( mindep > ANA_MANY ) {
	mindep = ANA_MANY;
    }
    /* scinc(sc_mdb_res[mindep]); */
    IF_V(v, printf(" -> %d\n", mindep); )
    return mindep;
}
#endif
