;;########################################################################
;; tranobj2.lsp
;; contains code to implement prototype transformation objects
;; Copyright (c) 1991-95 by Forrest W. Young
;;########################################################################

;;------------------------------------------------------------------------
;;absolute value object proto
;;------------------------------------------------------------------------

(defproto absval-transf-object-proto '() () transf-object-proto)

(defmeth absval-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth absval-transf-object-proto :options () t)

(defmeth absval-transf-object-proto :analysis ()
  (let* (
         (data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (absval-data (mapcar #'abs (column-list data-matrix)))
         (absval-data-matrix 
               (transpose (matrix size (combine absval-data))))
         )
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :title     (concatenate 'string "Abs-" (send self :title))
          :labels    (send current-data :active-labels) 
          :data      (combine absval-data-matrix)
          :variables (send self :active-variables '(numeric))
          :types     (send self :active-types     '(numeric)))))

(defun absolute-value 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Abs")
   (title      nil)
   )
  (send absval-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;normal scores object proto
;;------------------------------------------------------------------------

(defproto nscores-transf-object-proto '() () transf-object-proto)

(defmeth nscores-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth nscores-transf-object-proto :options () t)

(defmeth nscores-transf-object-proto :analysis ()
  (let* (
         (data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (nscores-data (mapcar #'nscores (column-list data-matrix)))
         (nscores-data-matrix 
               (transpose (matrix size (combine nscores-data))))
         )
    (data (send self :name)
          :created   (- (send *desktop* :num-icons) 1)
          :title     (concatenate 'string "NScores-" (send self :title))
          :labels    (send current-data :active-labels) 
          :data      (combine nscores-data-matrix)
          :variables (send self :active-variables '(numeric))
          :types     (send self :active-types     '(numeric)))))

(defun normal-scores 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "NScrs")
   (title      nil)
   )
  (send nscores-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;exponential object proto
;;------------------------------------------------------------------------

(defproto exponential-transf-object-proto '(choice) () transf-object-proto)

(defmeth exponential-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth exponential-transf-object-proto :choice (&optional (value nil set))
    (if set (setf (slot-value  'choice) value))
    (slot-value 'choice))

(defmeth exponential-transf-object-proto :options () 
  (send self :choice
        (choose-item-dialog "Choose Exponent(ial) Function" 
                            '("Natural Exponential (base e)" 
                              "Exponential - base x"
                              "Square Root (power 1/2)" "Square (power 2)"
                              "R'th Root (power 1/r)" "R'th Power (power r)"))))

(defmeth exponential-transf-object-proto :analysis ()
  (let* (
         (data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (ncol (first size))
         (choice (send self :choice))
         (param nil)
         (string nil)
         (exp-data nil)
         (exp-data-matrix nil))
    (when choice 
          (cond
            ((= choice 0) 
             (setf exp-data (mapcar #'exp (column-list data-matrix)))
             (setf string "NatExp"))
            ((= choice 1) 
             (setf param (get-value-dialog "Specify Base of Exponential"
                                           :initial 10))
             (setf exp-data (mapcar #'log  (column-list data-matrix) 
                                    (repeat param ncol)))
             (setf string "XExp"))
            ((= choice 2)
             (setf exp-data (mapcar #'sqrt (column-list data-matrix)))
             (setf string "Sqrt"))
            ((= choice 3)
             (setf exp-data (mapcar #'^ (column-list data-matrix)
                                    (repeat 2 ncol)))
             (setf string "Sqre"))
            ((= choice 5)
             (setf param (get-value-dialog "Specify Power" :initial 3))
             (setf exp-data (mapcar #'^ (column-list data-matrix)
                                    (repeat param ncol)))
             (setf string "PowR"))
            ((= choice 4)
             (setf param (get-value-dialog "Specify Root" :initial 3))
             (setf exp-data (mapcar #'^ (column-list data-matrix)
                                    (repeat (/ param) ncol)))
             (setf string "RootR"))
            )
          (setf exp-data-matrix 
                (transpose (matrix size (combine exp-data))))
          (data (strcat string "-" (send current-data :name))
                    :created   (- (send *desktop* :num-icons) 1)
                    :title     (strcat string "-" (send self :title))
                    :labels    (send current-data :active-labels) 
                    :data      (combine exp-data-matrix)
                    :variables (send self :active-variables '(numeric))
                    :types     (send self :active-types     '(numeric))))))

(defun exponential 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Exp")
   (title      nil)
   )
  (send exponential-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;logarithm object proto
;;------------------------------------------------------------------------

(defproto logarithm-transf-object-proto '(choice) ()
  transf-object-proto)

(defmeth logarithm-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth logarithm-transf-object-proto :choice (&optional (value nil set))
    (if set (setf (slot-value  'choice) value))
    (slot-value 'choice))

(defmeth logarithm-transf-object-proto :options () 
  (send self :choice
        (choose-item-dialog "Choose Logarithm Function" 
                            '("Natural (base e)" "Log-x (base x)"))))

(defmeth logarithm-transf-object-proto :analysis ()
  (if (<= (min (combine (send self :active-data-matrix '(numeric)))) 0)
      (error-message "Cannot take logarithms of non-positive numbers.")
      (let* ((data-matrix (send self :active-data-matrix '(numeric)))
             (size (reverse (array-dimensions data-matrix)))
             (ncol (first size))
             (choice (send self :choice))
             (base nil)
             (string nil)
             (log-data nil)
             (log-data-matrix nil))
        (when choice 
              (cond
                ((= choice 0) 
                 (setf log-data (mapcar #'log (column-list data-matrix)))
                 (setf string "NatLog"))
                ((= choice 1) 
                 (setf base (get-value-dialog "Specify Base of Logs"
                                              :initial 10))
                 (setf log-data 
                       (mapcar #'log  (column-list data-matrix) 
                               (repeat base ncol)))
                 (setf string "XLog")))
              (setf log-data-matrix 
                    (transpose (matrix size (combine log-data))))
              (data (strcat string "-" (send current-data :name))
                    :created   (- (send *desktop* :num-icons) 1)
                    :title     (strcat string "-" (send self :title))
                    :labels    (send current-data :active-labels) 
                    :data      (combine log-data-matrix)
                    :variables (send self :active-variables '(numeric))
                    :types     (send self :active-types     '(numeric)))))))

(defun logarithm 
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Log")
   (title      nil)
   )
  (send logarithm-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;rounding object proto
;;------------------------------------------------------------------------

(defproto rounding-transf-object-proto '(choice) () transf-object-proto)

(defmeth rounding-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth rounding-transf-object-proto :choice (&optional (value nil set))
    (if set (setf (slot-value  'choice) value))
    (slot-value 'choice))

(defmeth rounding-transf-object-proto :options ()
  (send self :choice
        (choose-item-dialog "Choose Rounding Function" 
                            '("Round (to nearest integer)" "Ceiling (round up)" "Floor (round down)" "Truncate (round toward zero)"))))

(defmeth rounding-transf-object-proto :analysis ()
  (let* ((data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (choice (send self :choice))
         (string nil)
         (rounding-data nil)
         (rounding-data-matrix nil))
    (when choice 
          (cond
            ((= choice 0) 
             (setf rounding-data (mapcar #'round (column-list data-matrix)))
             (setf string "Rnd"))
            ((= choice 1) 
             (setf rounding-data (mapcar #'ceiling(column-list data-matrix)))
             (setf string "Ceil"))
            ((= choice 2) 
             (setf rounding-data (mapcar #'floor  (column-list data-matrix)))
             (setf string "Flr"))
            ((= choice 3) 
             (setf rounding-data(mapcar #'truncate(column-list data-matrix)))
             (setf string "Trun"))
            )
          (setf rounding-data-matrix 
                (transpose (matrix size (combine rounding-data))))
          (data (strcat string "-" (send current-data :name))
                :created   (- (send *desktop* :num-icons) 1)
                :title     (strcat string "-" (send self :title))
                :labels    (send current-data :active-labels) 
                :data      (combine rounding-data-matrix)
                :variables (send self :active-variables '(numeric))
                :types     (send self :active-types     '(numeric))))))

(defun rounding
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Round")
   (title      nil)
   )
  (send rounding-transf-object-proto :new 9 data title name dialog))


;;------------------------------------------------------------------------
;;trigonometric object proto
;;------------------------------------------------------------------------

(defproto trig-transf-object-proto '(choice) () transf-object-proto)

(defmeth trig-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth trig-transf-object-proto :choice (&optional (value nil set))
    (if set (setf (slot-value  'choice) value))
    (slot-value 'choice))

(defmeth trig-transf-object-proto :options ()
  (send self :choice
        (choose-item-dialog "Choose Trigonometric Function" 
                            '("Sine" "Cosine" "Tangent" 
                              "Arc Sine" "Arc Cosine" "Arc Tangent"))))

(defmeth trig-transf-object-proto :analysis ()
  (let* ((data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (choice (send self :choice))
         (string nil)
         (trig-data nil)
         (trig-data-matrix nil))
    (when choice
          (cond
            ((= choice 0) 
             (setf trig-data (mapcar #'sin (column-list data-matrix)))
             (setf string "Sine"))
            ((= choice 1) 
             (setf trig-data (mapcar #'cos (column-list data-matrix)))
             (setf string "Cos"))
            ((= choice 2) 
             (setf trig-data (mapcar #'tan (column-list data-matrix)))
             (setf string "Tan"))
            ((= choice 3) 
             (setf trig-data (mapcar #'asin (column-list data-matrix)))
             (setf string "ASine"))
            ((= choice 4) 
             (setf trig-data (mapcar #'acos (column-list data-matrix)))
             (setf string "ACos"))
            ((= choice 5) 
             (setf trig-data (mapcar #'atan (column-list data-matrix)))
             (setf string "ATan")))
          (setf trig-data-matrix 
                (transpose (matrix size (combine trig-data))))
          (data (strcat string "-" (send current-data :name))
                :created   (- (send *desktop* :num-icons) 1)
                :title     (strcat string "-" (send self :title))
                :labels    (send current-data :active-labels) 
                :data      (combine trig-data-matrix)
                :variables (send self :active-variables '(numeric))
                :types     (send self :active-types     '(numeric))))))

(defun trigonometric
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Trig")
   (title      nil)
   )
  (send trig-transf-object-proto :new 9 data title name dialog))

;;------------------------------------------------------------------------
;;reciprocol object proto
;;------------------------------------------------------------------------

(defproto reciprocal-transf-object-proto '() () transf-object-proto)

(defmeth reciprocal-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth reciprocal-transf-object-proto :options () t)

(defmeth reciprocal-transf-object-proto :analysis ()
  (if (member 0 (combine (send self :active-data-matrix '(numeric))))
      (error-message "Cannot take reciprocal of data containing zeros.")
      (let* ((data-matrix (send self :active-data-matrix '(numeric)))
             (size (reverse (array-dimensions data-matrix)))
             (reciprocal-data (mapcar #'/ (column-list data-matrix)))
             (reciprocal-data-matrix 
              (transpose (matrix size (combine reciprocal-data))))
             )
        (data (send self :name)
              :created   (- (send *desktop* :num-icons) 1)
              :title     (concatenate 'string "Recip-" (send self :title))
              :labels    (send current-data :active-labels) 
              :data      (combine reciprocal-data-matrix)
              :variables (send self :active-variables '(numeric))
              :types     (send self :active-types     '(numeric))))))

(defun reciprocal
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "Recip")
   (title      nil)
   )
  (send reciprocal-transf-object-proto :new 9 data title name dialog))


;;------------------------------------------------------------------------
;;user-defined object proto
;;------------------------------------------------------------------------

(defproto userdef-transf-object-proto '(fname) () transf-object-proto)

(defmeth userdef-transf-object-proto :isnew (&rest args)
  (apply #'call-next-method args))

(defmeth userdef-transf-object-proto :fname (&optional (string nil set))
    (if set (setf (slot-value  'fname) string))
    (slot-value 'fname))

(defmeth userdef-transf-object-proto :options ()
  (send self :fname (get-string-dialog (format nil "Enter the Name of a Bound (Defined) ~%Function that takes One Vector-Valued~%Argument (Name Only, No Parentheses):"))))

(defmeth userdef-transf-object-proto :analysis ()
  (let* ((data-matrix (send self :active-data-matrix '(numeric)))
         (size (reverse (array-dimensions data-matrix)))
         (string (send self :fname))
         (userdef-data nil)
         (userdef-data-matrix nil))
    (when (equal string "") 
          (error-message "You must enter a function name."))
    (when (and string (not (equal string "")))
     (setf userdef-data 
           (mapcar (eval (read-from-string (strcat "#'" string)))
                   (column-list data-matrix)))
     (setf userdef-data-matrix 
           (transpose (matrix size (combine userdef-data))))
     (data (strcat string "-" (send current-data :name))
           :created   (- (send *desktop* :num-icons) 1)
           :title     (strcat string "-" (send self :title))
           :labels    (send current-data :active-labels) 
           :data      (combine userdef-data-matrix)
           :variables (send self :active-variables '(numeric))
           :types     (send self :active-types     '(numeric))))))

(defun user-defined
  (&key 
   (data       current-data)
   (dialog     nil)
   (name       "UserDef")
   (title      nil)
   )
  (send userdef-transf-object-proto :new 9 data title name dialog))

(provide "tranobj2")