;;########################################################################
;; sprdplot.lsp
;; Copyright (c) 1996-7 by Forrest W. Young, Richard A. Faldowski, & 
;; Carla Bann
;; Spread-Plot Object
;;########################################################################

;
;default nil update-plotcell method for all plots
;

(defmeth graph-proto :update-plotcell (i j args)
"Method Args: I J ARGS
Method to update a plotcell. This is the default nil method. Replacement methods must be written for individual plotcells which are to be updated. The replacement methods will be sent information by the update-spreadplot method. This information includes I and J to indicate the row and column of the  sending plotcell. The information also includes ARGS, a list of the union of the arguments that are needed by each receiving plotcell. Each plotcell must decide which arguments it needs."
    nil)

;
;Spread-Plot constructor function
;

(defun spread-plot 
  (plot-matrix &key statistical-object menu-title show)
"Method Args: PLOTMATRIX &KEY STATISTICAL-OBJECT MENU-TITLE SHOW
PLOTMATRIX is a matrix of plot-objects. The spreadplot cells consist of plots identified in this matrix. The plots are constructed and laid out in plot-matrix format, and are shown if SHOW is T. STATISTICAL-OBJECT is an optional object identification of the model or data object that is creating the spreadplot (nil, the default, when there is none). MENU-TITLE specifies the title of the spreadplots menu (nil, the default, means there is no menu). Creates and returns a spreadplot supervisor object."
  (let ((splot (send spreadplot-proto :new 
                     :statistical-object statistical-object 
                     :menu-title menu-title
                     :show show))) 
    (send splot :plot-matrix plot-matrix)
    (send splot :create-spreadplot)
    (when menu-title (send splot :create-menu menu-title)
          (send splot :show-menu))
    splot))


;Define prototype spreadplot and its isnew method
;

(defproto spreadplot-proto 
  '(plot-matrix statistical-object menu menu-title menu-template 
                locations size show))

(defmeth spreadplot-proto :isnew 
  (&key statistical-object 
        menu-title 
        show
        (menu-template '(dash dash dash)) )
  (send self :statistical-object statistical-object) 
  (send self :show show)
  (if menu-title (send self :menu t) (send self :menu nil))
  (send self :menu-title menu-title)
  (send self :menu-template menu-template)
  (setf *current-spreadplot* self)
  (send command-menu-refresh-spreadplot-item :enabled t)
  )

;
;Slot Accessor Methods
;

(defmeth spreadplot-proto :statistical-object (&optional (values nil set))
"Method Args:  &optional (VALUES nil) 
Sets or retrieves the object id# of the statistical object for which the spreadplot is installed."
  (if set (setf (slot-value 'statistical-object) values))
  (slot-value 'statistical-object) )

(defmeth spreadplot-proto :plot-matrix (&optional (matrix nil set))
"Method Args:  &optional PLOT-MATRIX
Sets or retrieves the spreadplot's matrix of plot object ids."
  (if set (setf (slot-value 'plot-matrix) matrix))
  (slot-value 'plot-matrix) )

(defmeth spreadplot-proto :locations (&optional (matrix nil set))
"Method Args:  &optional LOCATIONS-MATRIX
Sets or retrieves the spreadplot's matrix of plot locations."
  (if set (setf (slot-value 'locations) matrix))
  (slot-value 'locations) )

(defmeth spreadplot-proto :size (&optional (list nil set))
"Method Args:  &optional SIZE
Sets or retrieves the spreadplot's plot size list. All plots same size."
  (if set (setf (slot-value 'size) list))
  (slot-value 'size) )

(defmeth spreadplot-proto :menu (&optional (values nil set))
"Method Args:  &optional MENU
Sets or retrieves the spreadplot's menu in its MENU slot."
  (if set (setf (slot-value 'menu) values))
  (slot-value 'menu) )

(defmeth spreadplot-proto :show (&optional (logical nil set))
"Method Args:  &optional show
Sets or retrieves whether to show the spreadplot."
  (if set (setf (slot-value 'show) logical))
  (slot-value 'show) )

(defmeth spreadplot-proto :menu-title (&optional (values nil set))
"Method Args:  &optional ''TITLE''
Sets or retrieves the title of the spreadplot's menu."
  (if set (setf (slot-value 'menu-title) values))
  (slot-value 'menu-title) )

(defmeth spreadplot-proto :menu-template (&optional (values nil set))
"Method Args:  &optional TEMPLATE
Sets or retrieves the template of items for the spreadplot's menu."
  (if set (setf (slot-value 'menu-template) values))
  (slot-value 'menu-template) )

;
; Additional Methods
;

(defun spreadplot-help ()
  (if *current-spreadplot*
      (send *current-spreadplot* :spreadplot-help )
      (send graph-window-proto :spreadplot-help)))

(defmeth spreadplot-proto :spreadplot-help 
  (&key (flush t) skip nothing points bars labels )
  (plot-help-window (strcat "SpreadPlot Help") :flush flush)
  (when (and (not skip) *current-spreadplot*)
        (when (or points bars labels)
              (paste-plot-help (format nil
"The ~a~a~a~a~a in the windows of this spreadplot are linked together. When you brush or click on them in one window, the corresponding objects in other windows are also highlighted. These objects are linked together because they represent the same observations in your data. By looking for the structure revealed in each window you can get a better understanding of your data.~2%" (if points "points" "") 
(if (and points bars labels) "," " ")
(if bars " bars " "") 
(if (or (and points bars) (and points labels) (and bars labels)) "and " "")
(if labels "labels" "")))) 
    
        (when nothing
              (paste-plot-help (format nil "There are no other linkage features of this SpreadPlot. The general help about SpreadPlots is given in the remainer of this window.~2%")))
        )
  (paste-plot-help (format nil "GENERAL SPREADPLOT HELP~%"))
  (show-plot-help)
  (send graph-window-proto :spreadplot-help :flush flush));in displayw

(defmeth spreadplot-proto :create-spreadplot ()
"Method Args: ()
Creates the layout for the spreadplot cells and displays the plots."
  (let* ((plot-matrix (send self :plot-matrix))
         (dash (send dash-item-proto :new))
         (nrows (first (size plot-matrix)))
         (ncols (second (size plot-matrix)))
         (show (send self :show))
         (plot-size nil)
         (locations (make-array (list nrows ncols)))
         (plot nil)
         (plot00 (aref plot-matrix 0 0))
         )
    (when (< nrows 1) (error "SpreadPlots must have at least one row"))
    (when (< ncols 1) (error "SpreadPlots must have at least one column"))
    (setf plot-size  
          (min (- (floor (/ (first screen-size) ncols))
                  window-decoration-width 2)
               (- (floor (/ (- (second screen-size) menu-bar-height) nrows))
                  msdos-fiddle
#+X11                  60
                  (/ window-decoration-height 2))))
    (send self :size (list (+ plot-size window-decoration-width) plot-size))
    (dotimes (i nrows)
         (dotimes (j ncols)
              (setf (aref locations i j)
                    (list (+ border-thickness
                             (* j (+ plot-size window-decoration-width
                                     border-thickness 2)))
                          (+ (* i (+ (* 2 border-thickness)
#+X11 55 
plot-size 2)) 
#+X11                           35
                             border-thickness
                             (* (+ msdos-fiddle) (+ 1 i))
                             (* (1+ i) window-decoration-height))))
                  (send self :locations (add-element-to-list
                                         (send self :locations)
                                         (aref locations i j)))
                  ))
    (if (equal (send (select (send *help-menu* :items) 
                      (- (length (send *help-menu* :items)) 1)) :title) "-")
        (setf dash nil)
        (send *help-menu* :append-items dash))
    (dotimes (i nrows)
       (dotimes (j ncols)
          (setf plot (aref plot-matrix i j))
          (defmeth plot :close () (send plot00 :close))
          (send plot :size (+ plot-size window-decoration-width) plot-size)
          (apply #'send plot :location (aref locations i j))
          (send plot :add-slot 'spreadplot-object self)
          (send plot :add-plot-help-item)
          (defmeth plot :spreadplot-object (&optional (objid nil set))
            (if set (setf (slot-value 'spreadplot-object) objid))
            (slot-value 'spreadplot-object))))
    (when show
          (dotimes (i nrows)
                   (dotimes (j ncols)
                            (setf plot (aref plot-matrix i j))
                            (send plot :show-window))))
    (defmeth plot00 :close ()
      (when dash (send *help-menu* :delete-items dash))
      (setf *current-spreadplot* nil)
      (send command-menu-refresh-spreadplot-item :enabled nil)
      (dotimes (i nrows)
               (dotimes (j ncols)
                        (send (aref plot-matrix i j) :remove))))
    ))

(defmeth spreadplot-proto :update-statistical-object ()
"Method Args: none
Instructs the statistical object to update itself."
    (send (send self :statistical-object) :update))

(defmeth spreadplot-proto :update-spreadplot (i j &rest args)
"Method Args: I J &rest ARGS
This message is sent by an individual plotcell to the spreadplot which broadcasts it to all plotcells so that they can update themselves. The arguments I and J indicate the row and column of the sending plotcell. All remaining arguments must be a union of arguments needed by all receiving plotcells, each of which must decide which arguments it needs."
  (let* ((plot-matrix (send self :plot-matrix))
         (nrows (first (size plot-matrix)))
         (ncols (second (size plot-matrix))))
    (dotimes (k nrows)
             (dotimes (L ncols)
                      (send (select plot-matrix k L) 
                            :update-plotcell i j args)))))
  
(defmeth spreadplot-proto :remove ()
"Method Args: () 
Removes the spreadplot.  First, remove its menu if it has one.  Then, remove the individual plots installed in it. "
  (when (send self :menu)
        (send (send self :menu) :remove)
        (send (send self :menu) :dispose) )
  (mapcar #'(lambda (plot) 
              (when plot
                    (if (send plot :allocated-p)
                        (send plot :remove))))
              (send self :all-plots)) )

(defmeth spreadplot-proto :show-spreadplot ()
"Method Args: () 
Shows the spreadplot.  First, shows the individual plots installed in the spreadplot, then installs the menu (if it has one) into the menu bar."
  (send self :show-visible-plots)
  (when (send self :menu) (send (send self :menu) :install) ))

(defmeth spreadplot-proto :hide-spreadplot ()
"Method Args: () 
Hides the spreadplot.  First, removes its menu (if it has one) from the menu bar.  Then, hides the individual plots installed in it. "
  (when (send self :menu) (send (send self :menu) :remove) )
  (send self :hide-all-plots) )

(defmeth spreadplot-proto :create-menu 
  (&optional title  &key (items (send self :menu-template)) ) 
  (unless title 
          (setq title (slot-value 'menu-title)) )
  (if (slot-value 'menu) 
      (send (slot-value 'menu) :dispose) )
  (flet (
         (make-item (item) (send self :make-menu-item item) )
         )
    (let (
          (menu (send menu-proto :new title) )
          )
      (send self :menu menu) 
      (apply #'send menu :append-items  
             (remove nil (mapcar #'make-item items)))
      menu) ) )

(defmeth spreadplot-proto :make-menu-item (item-template)
  (if (kind-of-p item-template menu-item-proto)
      item-template
      (case item-template
        ( dash (send dash-item-proto :new) ) 
        ( show-plots 
          (send graph-item-proto :new "Show Plots" self 
                :show-visible-plots) )
        ( hide-plots 
          (send graph-item-proto :new "Hide Plots" self 
                :hide-all-plots) )
        ( show-spreadplot 
          (send graph-item-proto :new "Show SpreadPlot" self 
                :show-ssp) )
        ( hide-spreadplot 
          (send graph-item-proto :new "Hide SpreadPlot" self 
                :hide) )
        ( kill-spreadplot
          (send graph-item-proto :new "Kill SpreadPlot" (slot-value 'model)
                :delete-spreadplot-supervisor) ))))

(defmeth spreadplot-proto :all-plots ()
"Method Args: () 
Returns a list of all plots." 
  (combine (send self :plot-matrix)))

(defmeth spreadplot-proto :hide-all-plots ()
  (mapcar #'(lambda (x) (send x :hide-window)) (send self :all-plots)))

(defmeth spreadplot-proto :show-visible-plots ()
  (mapcar #'(lambda (x) (send x :show-window)) (send self :all-plots))
  (send (aref (send self :plot-matrix) 0 0) :show-window))

(defun refresh-spreadplot ()
  (send *current-spreadplot* :refresh-spreadplot))

(defmeth spreadplot-proto :refresh-spreadplot ()
  (let* ((plot-list (combine (send self :plot-matrix)))
         (location-list (send self :locations))
         (nplots (length plot-list))
         ) 
    ;(send self :hide-all-plots)
    (mapcar #'(lambda (i)
                (apply #'send (select plot-list i) :location 
                       (select location-list i))
                (apply #'send (select plot-list i) :size
                       (send self :size)))
            (iseq nplots))
    (send self :show-visible-plots)))


;;########################################################################
;; Old spreadplot5 function and methods
;;########################################################################

(defun spreadplot5 (&rest args)
"Alias for spread-plot function"
  (let ((sp (apply #'spread-plot args)))
    (send sp :show-spreadplot)
    sp))

(defmeth spreadplot-proto :update-plot-objects (args)
"Method Args: none
Instructs the plot objects to update themselves. Replaced by the update-spreadplot method. Uses update-contents rather than update plotcell."
  (let* ((plot-matrix (send self :plot-matrix))
         (nrows (first (size plot-matrix)))
         (ncols (second (size plot-matrix))))
    (dotimes (i nrows)
             (dotimes (j ncols)
                      (send (select plot-matrix i j) 
                            :update-contents args)))))

;
;default nil update-contents method for all plots
;

(defmeth graph-proto :update-contents (args)
"Method Args: Args
Method to update plot contents. This is the default nil method. Replaced by update-plotcell method."
    )

