;;########################################################################
;; displayw.lsp
;; Copyright (c) 1997 by Forrest W. Young
;; Creates a window for displaying text. Extends the Mac-only
;; display window (Tierney, p. 360) to work on all platforms.
;;########################################################################

(defun display-window (text &key (title "Text Window") (size '(475 280)) 
                            (location '(10 20)) (show t) fit)
"Args: TEXT &KEY TITLE SIZE LOCATION SHOW FIT
Displays a window containing TEXT, a required argument. Window is fit to text when FIT is T." 
  (let* ((w (send display-window-proto2 :new :title title :size size
                 :location location :show nil)) ;:show show
         (window-height (second (send w :size)))
         (line-height (send w :line-height))
         (page-increment (* line-height 
                            (floor (/ (- window-height line-height) 
                                      line-height)))))
    (send w :paste-string text)
    (cond 
      (fit (when show (send w :show-window))
           (send w :fit-window-to-text))
      (t (when show (send w :show-window))))
    
    w))

(defproto display-window-proto2 
  '(x y nstrings strings nlines lines line-height line-width write-now
      default-window-height fit-window-to-text-height nowrap v-scroll-value
      x-list y-list reformatting)
  () graph-window-proto)

(defmeth display-window-proto2 :isnew 
  (&key (title "Text Window") (size '(475 280)) (location '(10 20)) (show t))
  (call-next-method :title title :size size :show nil)
                    ;:location (+ (list 0 20) location)
  (apply #'send self :location location);(- (send self :location) (list 4 24))
  (send self :default-window-height (second size))
  (send self :flush-window)
  (send self :line-height (+ (send self :text-ascent)
                             (send self :text-descent)))
  (let* ((window-height (second (send self :size)))
         (line-height (send self :line-height))
         (page-increment (* line-height 
                            (floor (/ (- window-height line-height) 
                                      line-height)))))
    (send self :has-v-scroll t) 
    (send self :v-scroll-incs line-height page-increment))
  (when show (send self :show-window))
  t)

(defmeth display-window-proto2 :x (&optional (number nil set))
  (if set (setf (slot-value 'x) number))
  (slot-value 'x))

(defmeth display-window-proto2 :y (&optional (number nil set))
  (if set (setf (slot-value 'y) number))
  (slot-value 'y))

(defmeth display-window-proto2 :x-list (&optional (list-of-nums nil set))
  (if set (setf (slot-value 'x-list) list-of-nums))
  (slot-value 'x-list))

(defmeth display-window-proto2 :y-list (&optional (list-of-nums nil set))
  (if set (setf (slot-value 'y-list) list-of-nums))
  (slot-value 'y-list))

(defmeth display-window-proto2 :nstrings (&optional (number nil set))
  (if set (setf (slot-value 'nstrings) number))
  (slot-value 'nstrings))

(defmeth display-window-proto2 :strings (&optional (list-of-strings nil set))
  (if set (setf (slot-value 'strings) list-of-strings))
  (slot-value 'strings))

(defmeth display-window-proto2 :nlines (&optional (number nil set))
  (if set (setf (slot-value 'nlines) number))
  (slot-value 'nlines))

(defmeth display-window-proto2 :lines (&optional (list-of-strings nil set))
  (if set (setf (slot-value 'lines) list-of-strings))
  (slot-value 'lines))

(defmeth display-window-proto2 :line-height (&optional (number nil set))
  (if set (setf (slot-value 'line-height) number))
  (slot-value 'line-height))

(defmeth display-window-proto2 :line-width (&optional (number nil set))
  (if set (setf (slot-value 'line-width) number))
  (slot-value 'line-width))

(defmeth display-window-proto2 :write-now  (&optional (logical nil set))
  (if set (setf (slot-value 'write-now) logical))
  (slot-value 'write-now))

(defmeth display-window-proto2 :fit-window-to-text-height 
  (&optional (logical nil set))
  (if set (setf (slot-value 'fit-window-to-text-height) logical))
  (slot-value 'fit-window-to-text-height))

(defmeth display-window-proto2 :reformatting (&optional (logical nil set))
  (if set (setf (slot-value 'reformatting) logical))
  (slot-value 'reformatting))

(defmeth display-window-proto2 :default-window-height 
  (&optional (number nil set))
  (if set (setf (slot-value 'default-window-height) number))
  (slot-value 'default-window-height))

(defmeth display-window-proto2 :nowrap  (&optional (logical nil set))
  (if set (setf (slot-value 'nowrap) logical))
  (slot-value 'nowrap))

(defmeth display-window-proto2 :v-scroll-value (&optional (number nil set))
  (if set (setf (slot-value 'v-scroll-value) number))
  (slot-value 'v-scroll-value))

(defmeth display-window-proto2 :paste-stream (stream)
  (let ((string nil))
    (loop
     (setf string (read-line stream nil))
     (when (not string) (return))
     (setf string (strcat string (string #\newline)))
     (when mytrace (format t "LENGTH: ~3d STREAM: ~s~%" 
                           (length string) string))
     (send self :paste-string string))
    ))

(defmeth display-window-proto2 :paste-string (string)
  (let* ((last-char nil)
         (string-piece string)
         (string-max string-piece)
         (string-length nil)
         (more nil)
         (split-loc 0)
         (max-char 100)
         )
    (when (> (length string) 0)
          (setf last-char (select2 (reverse string) 0))
          (when (and (not (equal last-char #\ ))
                     (not (equal last-char #\newline)))
                (setf string (strcat string " "))
                (setf string-piece string)
                (setf string-max string-piece))
          (loop (setf more nil)
                (setf string-length (length string-piece))
                (when (> string-length max-char)
                      (setf string-max 
                            (reverse (select2 string-piece (iseq max-char))))
                      (setf split-loc (- max-char (position #\  string-max)))
                      (setf string-max 
                            (select2 (reverse string-max) (iseq split-loc)))
                      (setf more t)
                      )
                (send self :strings 
                      (add-element-to-list (send self :strings) string-max))
                (send self :nstrings (1+ (send self :nstrings)))
                (when (> (length string-max) 0) 
                      (send self :write-string-to-window string-max))
                (when more
                      (setf max-char 
                            (min max-char (- string-length split-loc)))
                      (setf string-piece 
                            (select2 string-piece 
                                    (iseq split-loc (- string-length 1))))
                      (setf string-max string-piece)
                      )
                (when (not more) (return))))
    t))

(setf mytrace nil)
(setf mybreak nil)

(defmeth display-window-proto2 :write-string-to-window (string)
  (let ((x (send self :x))
        (y (send self :y))
        (string-width (send self :text-width string))
        (line-width (send self :line-width))
        (line-width-remaining nil)
        (old-line-remaining string)
        (pieces nil)
        (piece-width nil)
        (splitable string)
        (vr (send self :view-rect))
        )
    (when (not line-width)
          (if (= 0 (send self :canvas-width))
              (send self :line-width 400)
              (send self :line-width (- (send self :canvas-width) 20)))
          (setf line-width (send self :line-width)))
    (setf line-width-remaining (- (send self :line-width) x))
    (loop 
     (setf pieces (send self :split-line splitable x line-width-remaining))
     (when (and (equal (third pieces) old-line-remaining)
                (not (position #\  (third pieces)))
                (<= (send self :line-width)
                    (send self :text-width (third pieces))))
           (when mytrace
                 (format t "IN WIERD PLACE - Line Width ~d <= Text Width ~d~%~a~%"
                         (send self :line-width)
                         (send self :text-width (third pieces)) (third pieces)))
           (setf line-width-remaining (send self :line-width))
           (setf piece-width (send self :text-width (third pieces)))
           (send self :lines 
                 (add-element-to-list (send self :lines) (third pieces)))
           (send self :write-line-to-window (third pieces) x y piece-width)
           (setf splitable nil)
           (return)
           )
     (setf old-line-remaining (third pieces))
     (when (first pieces)
           (setf piece-width (send self :text-width (first pieces)))
           (send self :lines 
                 (add-element-to-list (send self :lines) (first pieces)))
           (send self :write-line-to-window (first pieces) x y piece-width))
     (when (<= (send self :line-width) (send self :x)))
     (when (equal (second pieces) "NL")(send self :new-line y))
     (cond 
       ((third pieces) 
        (when (not (equal (second pieces) "NL"))(send self :new-line y))
        (setf x (send self :x))
        (setf y (send self :y))
        (setf line-width-remaining (send self :line-width))
        (setf splitable (third pieces)))
       (t (return))))))

(defmeth display-window-proto2 :split-line (string x line-width)
  (let* ((string-length (length string))
         (nowrap (send self :nowrap))
         (margins 0)
         (real-line-width (- line-width margins))
         (first-newline-loc nil)
         (last-space-loc nil)
         (split-loc nil)
         (print-string string)
         (previous-print-string string)
         (print-string-width (send self :text-width print-string))
         (print-string-length (length print-string))
         (remaining-string nil)
         (print-string-backwards nil)
         (last-space-loc nil)
         (last-nl-loc nil)
         )

    (when (<= real-line-width 0) (setf real-line-width 20))

    (if mytrace (format t "~%SPLITABLE LINE:  ~s~%"string)) 
    (if mybreak (break))

 ;loop finds longest acceptable string - skip if nowrap

    (when (not nowrap)
    (loop
     (if mytrace 
        (format t "OUTER LOOP: psw ~d rlw ~d psl ~d ~s~%" print-string-width
                real-line-width print-string-length print-string)
          )
     (when (< print-string-width real-line-width) 
           (setf print-string previous-print-string)
           (setf print-string-length (length print-string))
           (setf print-string-width (send self :text-width print-string))
           (loop
            (if mytrace 
                (format t "INNER LOOP: psw ~d rlw ~d psl ~d ~s~%" print-string-width
                        real-line-width print-string-length print-string)
                 )
            (when (< print-string-width real-line-width) (return))
            (setf print-string-length (- print-string-length 1))
            (setf print-string (select2 string (iseq print-string-length))) 
            (setf print-string-width (send self :text-width print-string)))
           (return))
     (setf previous-print-string print-string)
     (setf print-string-backwards (reverse print-string))
     (setf last-space-loc (position #\  print-string-backwards))
     (setf last-nl-loc (position #\newline  print-string-backwards))
     (when (and (not last-space-loc) (not last-nl-loc)) (return))
     (when (not last-space-loc) (setf last-space-loc print-string-length))
     (when (not last-nl-loc) (setf last-nl-loc print-string-length))
     (setf print-string-length 
           (- print-string-length (min last-space-loc last-nl-loc) 1))
     (setf print-string (select2 string (iseq print-string-length))) 
     (if print-string
         (setf print-string-width (send self :text-width print-string))
         (setf print-string-width 0))
     ))

    (if mytrace 
        (format t "AFTER LOOP: psw ~d rlw ~d psl ~d ~s~%" print-string-width
                real-line-width print-string-length print-string))
    (if mybreak (break))

    ;now look for first newline in that string

    (setf first-newline-loc (position #\newline print-string))
    (if mytrace (format t "FNL ~d~%" first-newline-loc))

    ;if no newline, look for last space in that string

    (if first-newline-loc
        (setf split-loc (+ 1 first-newline-loc))
        (if (position #\  (reverse print-string))
            (setf split-loc 
                  (- print-string-length 
                     (position #\  (reverse print-string))))
            (setf split-loc 0))) 
    (if mytrace (format t "SPLIT-LOC ~d~%" split-loc))
    ;sometimes strips off spaces that are not to be stripped!
    ;remaining string from after (not including) split-loc to end of string
    (if (= split-loc string-length);sometimes print-string-length
        (setf remaining-string nil)
        (setf remaining-string 
              (select2 string ;sometimes print-string 
                      (iseq split-loc (- string-length 1)))))
    ;above sometimes print-string-length
    ;make print-string the string up to (not incuding) split-loc
    (if mytrace (format t "REMAIN ~s~%" remaining-string))
    (if (= split-loc 0)
        (setf print-string nil)
        (setf print-string 
              (strcat " " (select2 print-string (iseq (- split-loc 1))))))
    (setf break-char (if first-newline-loc "NL" "SP"))
    (if mytrace (format t "AFTER SPLITTING: ~s ~s ~s~%" 
                        print-string break-char remaining-string))
    (if mybreak (break))
    (list print-string break-char remaining-string)))

(defmeth display-window-proto2 :write-line-to-window (string x y st-width)
  (when (send self :write-now) 
        (let ((vr (send self :view-rect)))
          (when (<= (second vr) y (+ (second vr) (fourth vr)))
                 (send self :draw-text string x y 0 1)
                 ))
        )
  (send self :x-list (add-element-to-list (send self :x-list) x))
  (send self :y-list (add-element-to-list (send self :y-list) y))
  (send self :x (+ (send self :x) st-width)))

(defmeth display-window-proto2 :new-line (y)
  (send self :nlines (1+ (send self :nlines)))
  (send self :y (+ (send self :y) (send self :line-height)))
  (send self :x 10)
  (send self :y))

(defmeth display-window-proto2 :redraw ()
  (when (not (send self :reformatting))
  (when (send *vista* :ready-to-redraw self)
        (send self :erase-window)
        (let* ((y-top (second (send self :view-rect)))
               (y-now nil)
               (y-bot (+ y-top (fourth (send self :view-rect)))))
          (dotimes (i (length (send self :y-list)))
                   (setf y-now (select (send self :y-list) i))
                   (cond
                     ((<= y-top y-now y-bot)
                      (send self :draw-text 
                            (select (send self :lines) i)
                            (select (send self :x-list) i) y-now 0 1))
                     ((> y-now y-bot)
                      (return)))))
        (send *vista* :finished-redraw self))
        ))
        

(defmeth display-window-proto2 :reformat ()
  (when (send *vista* :ready-to-redraw self)
        (let ((write-before (send self :write-now)))
          (send self :write-now t)
          (send self :reformatting t)
          (when (not write-before) (send self :start-buffering))
          (cond 
            ((> (first (send self :size)) 200)
             (let ((nstrings (send self :nstrings))
                   (strings (send self :strings))
                   )
               (send self :erase-window)
               (send self :x 10)
               (send self :y 0)
               (send self :nlines 0)
               (send self :lines nil)
               (send self :x-list nil)
               (send self :y-list nil)
               (when nstrings
                     (dotimes (i nstrings)
                              (send self :write-string-to-window 
                                    (select2 strings i))))))
            (t (send self :size 201 (second (send self :size)))))
          (send self :write-now write-before)
          
          (let* ((window-height (second (send self :size)))
                 (line-height (send self :line-height))
                 (content-height 
                  (* line-height (1+ (send self :nlines))))
                 (page-increment (* line-height 
                                    (floor (/ (- window-height line-height) 
                                              line-height))))) 
            (cond
              ((> content-height (second (send self :size)))
               (send self :v-scroll-incs line-height page-increment)
               (send self :has-v-scroll content-height))
              (t
               (send self :has-v-scroll nil)))
            )
          (send self :reformatting nil)
          (when (not write-before) (send self :buffer-to-screen)))
        (send *vista* :finished-redraw self)))

(defmeth display-window-proto2 :resize ()
  ;(call-next-method)
  (send self :scroll 0 0)
  (send self :line-width (- (send self :canvas-width) 20))
  (when (not (send self :reformatting))
        (send self :reformat)))
    

(defmeth display-window-proto2 :fit-window-to-text ()
  (let* ((nlines (send self :nlines))
         (line-height (send self :line-height))
         (window-height (+ 5 (* (1+ nlines) line-height)))
         (dwh (send self :default-window-height))
         )
    (when (= 0 window-height) (setf window-height dwh))
    (send self :size (first (send self :size)) window-height)
    window-height
    ))

(defmeth display-window-proto2 :flush-window ()
  (send self :erase-window)
  (send self :x 10)
  (send self :y 0)
  (send self :lines nil)
  (send self :x-list nil)
  (send self :y-list nil)
  (send self :strings nil)
  (send self :nstrings 0)
  (send self :nlines 0))

(defmeth display-window-proto2 :show-window ()
  (call-next-method)
  (send *vista* :help-showing t))

(defmeth display-window-proto2 :close ()
  (call-next-method)
  (send *vista* :help-showing nil))

(defmeth display-window-proto2 :plot-help 
     (&optional (text "This window displays help and information messages.")
                (title "Help: Text Window"))
  (let* (
         (w (plot-help-window title)))
    (paste-plot-help  (format nil text))
    (show-plot-help)))






(defun paste-text (w text)
  (send w :paste-string text))

(defun show-display-window (w)
  (if (> (* (+ 2 (send w :nlines)) (send w :line-height))
           (second (send w :size)))
       (send w :has-v-scroll (* (+ 2 (send w :nlines)) 
                                (send w :line-height)))
       (send w :has-v-scroll nil))
  #+macintosh (when (not (equal (front-window) w)) (send w :show-window))
  #-macintosh (send w :show-window)
  w)

(defun add-text (w text &key (show t) fit)
  (send w :paste-string text)
  (when fit (send w :fit-window-to-text))
  (when show (send w :show-window)))
  
;The following code is modified to work with the above code.
;The original code is found in function.lsp
;the next functions and methods do help windows (other than plot help)

(defun file-to-window (filename title w &optional (flush t) (add-help t))
  (send *vista* :file-to-help-window filename title w flush add-help)
  w)

(defun file-to-stream (filename title &optional (out-stream *standard-output*))
  (cond
    ((equal out-stream *standard-output*)
     (send *vista* :file-to-help-window 
           filename title (send *vista* :help-window-object)))
    (t
     (gc)
     (if (equal title "Bug List")
         (format out-stream 
                 "~3%**************** ~a  ****************~2%" 
                 title)
         (format out-stream 
                 "~3%****************  Help for ~a  ****************~2%" 
                 title))
     (with-open-file (in-stream filename :direction :input)
                     (let ((char nil))
                       (loop ;loop until eof
                             (if (setq char (read-char in-stream nil nil))
                                 (write-char char out-stream)
                                 (return nil)))))
     (terpri))))

(defmeth VISTA-SYSTEM-OBJECT-PROTO :file-to-help-window 
                      (filename title w &optional (flush t) (add-help t))
  (with-open-file 
   (g filename)
   (when (not w) 
         (let ((size '(500 150)) ;320 264
               (location '(4 332))) ;375 170
           (setf w (send self :create-help-window :title "Help Window"
                         :location location :size size :show nil))
           #+msdos(send w :size 500 
                        (second (- screen-size 12 (send w :location))))
           #+macintosh(send w :size 475 112)
           #+macintosh(send w :location 10 342)
           (send w :show-window)))
   (send self :update-help-window w g title flush add-help)
   t))


(defmeth VISTA-SYSTEM-OBJECT-PROTO :create-help-window 
  (&key (title "Text Window") (size '(475 280)) (location '(100 100)) 
        (show t))
  (let ((w (send display-window-proto2 :new 
                 :show show :size size :location location :title title)))
    (send *vista* :help-window-object w)
    (send *vista* :help-window-status nil)
    (setf *help-window* w)
    (defmeth w :remove ()
      (send self :hide-window)
      (send *vista* :help-window-status nil)
      (when (send help-menu-show-help-item :mark)
            (send *vista* :show-help nil)
            (send help-menu-show-help-item :mark nil)))
    w))

(defmeth VISTA-SYSTEM-OBJECT-PROTO :update-help-window 
                          (w g title &optional (flush t) (add-help t))
  (when flush (send w :flush-window))
  (send w :scroll 0 0)
  (if (or (not add-help) 
          (equal title "Bug List") 
          (equal title "About These Data"))
      (send w :title title)
      (send w :title (strcat "Help: " title)))
  (if g
      (send w :paste-stream g)
      (send w :paste-string (send *current-data* :about)))
  (if (> (send w :y) (second (send w :size))) 
       (send w :has-v-scroll (send w :y))
       (send w :has-v-scroll nil))
  #+macintosh (when (not (equal (front-window) self)) (send w :show-window))
  #-macintosh (send w :show-window)
  (send *vista* :help-window-status t)
  t)

(defun report-header (title)
"Function args :title
Macintosh: Opens a display window TITLE and returns display window object-id.
Dos/Unix:  Writes TITLE to listener and returns nil"
  (let ((w nil))
#+macintosh(setf w (send *vista* :report-window title))
#-macintosh(report-header-help title)
#-macintosh(display-string 
            (format nil "~2% ~a ****************~2%" title))
    w))

#|
;next function is modified from generic.lsp version

(defun about-these-data ()
  (let* ((w (send *vista* :help-window-object))
         (about (send *current-data* :about))
         (title "About These Data"))
    (when (not w) (setf w (send *vista* :create-help-window :title title
                                :location '(375 170) :size '(320 264))))
    (send *vista* :update-help-window w nil title)
    t))
|#

;next function do plot help buttons and menu items
;the original code is in graphics.lsp

(defun plot-help-window (title &key (flush t))
  (let ((size '(500 150))
        (location '(4 332))
        (w (send *vista* :help-window-object)))
    (when (not w) 
          (setf w (send *vista* :create-help-window :title title
                        :location location :size size :show nil))
          #+msdos(send w :size 500 
                       (second (- screen-size 12 (send w :location))))
          #+macintosh(send w :size 475 112)
          #+macintosh(send w :location 10 342))
    (when flush 
          (send w :flush-window)
          (send w :scroll 0 0)
          (send w :title title))
    w))

(defun paste-plot-help (plot-help &optional ignored)
  (let ((w (send *vista* :help-window-object)))
    (send w :paste-string plot-help)
    w))

(defun show-plot-help ()
  (let ((w (send *vista* :help-window-object)))
    #+macintosh (when (not (equal (front-window) w)) (send w :show-window))
    #-macintosh (send w :show-window)
    (if (> (* (+ 2 (send w :nlines)) (send w :line-height))
           (second (send w :size)))
       (send w :has-v-scroll (* (+ 2 (send w :nlines)) 
                                (send w :line-height)))
       (send w :has-v-scroll nil))
    (send *vista* :help-window-status t)

    w))

(defmeth display-window-proto2 :show-message 
     (&optional (text "This window displays information messages.")
                (title "Message Window"))
  (send self :flush-window)
  (send self :scroll 0 0)
  (send self :title title)
  (send self :paste-string text)
  #+macintosh (when (not (equal (front-window) self))
                    (send self :show-window))
  #-macintosh (send self :show-window)
    (if (> (* (+ 2 (send self :nlines)) 
              (send self :line-height))
           (second (send self :size)))
       (send self :has-v-scroll 
             (* (+ 2 (send self :nlines)) 
                (send self :line-height)))
       (send self :has-v-scroll nil))
  )