;;########################################################################
;; dashobj3.lsp
;; Contains methods to draw and manipulate a datasheet
;; Copyright (c) 1994-8 by Forrest W. Young
;;########################################################################

(defmeth datasheet-proto :redraw ()
  (when (send self :redraw-now)
  (when (send *vista* :ready-to-redraw self);fwy 4.31 10/30/97
  (send self :start-buffering)
  (send self :erase-window)
  (let ((dob (send self :data-object))
        (numvar (send self :nvar))
        (numobs (send self :nobs))
        (fw (send self :field-width))  ;datum field width
        (fh (send self :field-height)) ;datum field height
        (lw (send self :label-width))  ;label field width
        (hh nil)                       ;header field height
#+macintosh (cw 6)                     ;character width
#-macintosh (cw 8)                     ;character width
        (nc (send self :number-of-columns))
        (yloc nil)
        (xloc nil)
        (text nil)
        (types (send self :type-strings))
        (matst (send self :data-matrix-strings))
        (editable (send self :editable))
        (hot-cell (send self :hot-cell))
        ) 
    (setf hh (* 2 fh))
    (send self :line-type 'solid)
    (send self :draw-line  0  0 (+ lw (* numvar fw))  0)
    (send self :draw-line  0  0 0 (* (+ 2 numobs) fh))
    (send self :draw-line lw fh (+ lw (* numvar fw)) fh)
    (dotimes (i numvar)
             (send self :draw-line (+ lw (* (+ i 1) fw)) 0
                   (+ lw (* (+ i 1) fw)) hh))
    (send self :draw-line 0 0 lw hh)
    (send self :line-width 2)
    (send self :draw-line  0 hh (+ lw (* numvar fw))  hh)
    (send self :draw-line lw  0    lw (* (+ 2 numobs) fh))
    (send self :draw-line 0 (* (+ 2 numobs) fh)
          (+ lw (* numvar fw)) (* (+ 2 numobs) fh) )
    (send self :draw-line (+ lw (* numvar fw)) 0
          (+ lw (* numvar fw)) (* (+ 2 numobs) fh))
    (send self :line-width 1)
    (send self :line-type 'dashed)
    (send self :draw-text (format nil "~d Vars" numvar)
          (- lw 4) (- fh 3) 2 0)
    (send self :draw-text (format nil "~d Obs" numobs) 3 (- (* 2 fh) 3) 0 0)
    (dotimes (i (- numobs 1))
             (send self :draw-line 0 (* (+ i 3) fh) 
                   (+ lw (* numvar fw)) (* (+ i 3) fh)))
    (dotimes (i (- numvar 1))
             (send self :draw-line (+ lw (* (+ i 1) fw)) hh 
                   (+ lw (* (+ i 1) fw)) (* (+ 2 numobs) fh)))
;draw variable names and types
    (dotimes (i numvar) 
             (send self :draw-text 
                   (subseq 
                    (select (send self :variable-strings) i) 0 
                    (min (floor (/ fw cw));6
                         (length (select (send self :variable-strings) i))))
                   (+ lw (* fw i) (floor (/ fw 2))) (- fh 3) 1 0)
             (send self :draw-text 
                   (subseq 
                    (select (send self :type-strings) i) 0
                    (min (floor (/ fw cw));6
                         (length (select (send self :type-strings) i))))
                   (+ lw (* fw i) (floor (/ fw 2))) (- (* 2 fh) 3) 1 0))
;draw labels and data
    (dotimes (i numobs)
             (setf yloc (+ (* 3 fh) (- (* i fh) 3))) 
             (send self :draw-text (select (send self :label-strings) i)
                   3 yloc 0 0)
             (dotimes (j numvar)  
              (setf xloc (- (+ lw (* (+ 1 j) fw)) 3))
              (if (> (- fw cw) (send self :text-width (aref matst i j)));6
                  (send self :draw-text (aref matst i j) xloc yloc 2 0)
                  (if (equal "Category" (select types j))
                      (send self :draw-text 
                            (subseq (aref matst i j) 0 nc) xloc yloc 2 0)
                      (send self :draw-text 
                            (apply #'strcat (repeat "*" nc)) 
                            xloc yloc 2 0)))))
    (when editable
          (send self :line-type 'solid)
          (when (not (send dob :ways))
                (send self :draw-line (+ lw (* fw (+ numvar 1)))  0 
                      (+ lw (* fw (+ numvar 1))) fh)
                (send self :draw-line (+ lw (* fw    numvar   )) fh 
                      (+ lw (* fw (+ numvar 1))) fh)
                (setf text "New Var")
                (send self :draw-text text 
                      (floor (+ lw (* fw (+ numvar 0.5)))) (- fh 3) 1 0))
          (send self :draw-line 0  (* fh (+ numobs 3))
                                lw (* fh (+ numobs 3)) )
          (send self :line-width 2)
          (send self :draw-line lw (* fh (+ numobs 2)) 
                                lw (- (* fh (+ numobs 3)) 1))
          (send self :line-width 1)
          (if (send dob :matrices) 
              (setf text "New Matrix") (setf text "New Obs"))
          (send self :draw-text text 3 (- (* fh (+ numobs 3)) 3) 0 0)
          (when hot-cell (send self :reverse-cell-color 
                      (first hot-cell) (second hot-cell) lw fw fh 
                               (send self :hot-cell-ready)))
          )
        (send self :buffer-to-screen)
        (send *vista* :finished-redraw self);fwy 4.31 10/30/97
        ))))

(defmeth datasheet-proto :do-click (x y m1 m2) 
 #+msdos (when (not (equal *current-data* (send self :data-object)))
               (setcd (send self :data-object)))
  (let ((dob (send self :data-object))
        (fw  (send self :field-width))
        (fh  (send self :field-height))
        (lw  (send self :label-width))
        (old-hot-cell (send self :hot-cell))
        (ready (send self :hot-cell-ready))
        (nobs (send self :nobs))
        (nvar (send self :nvar))
        (new-var nil)
        (new-obs nil)
        (body nil)
        (row nil)
        (col nil)) 
    (send self :edited t)
    (when (send self :editable)
          (setf row (- (ceiling y fh) 2))
          (setf col (ceiling (- x lw) fw))
          (when (and (not (and (< row 1) (< col 1)))
                     (and (<= row nobs) (<= col nvar)))
                (setf body t)
                (send self :hot-cell (list row col))
                (send self :reverse-cell-color row col lw fw fh))
          (when (not body)
                (if (and (= row -1) (= col (+ 1 nvar)) 
                         (not (send dob :ways)))
                    (setf new-var t) (setf new-var nil))
                (if (and (= row (+ 1 nobs)) (< col 1))
                    (setf new-obs t) (setf new-obs nil)))
          (when old-hot-cell
                (setf row (first  old-hot-cell))
                (setf col (second old-hot-cell))
                (send self :reverse-cell-color row col lw fw fh ready))
          (when (not body)
                (send self :hot-cell nil)
                (when (or new-var new-obs)
                      (if (send dob :matrices)
                          (send self :expand-mat-datasheet new-var new-obs 1)
                          (send self :expand-mv-datasheet new-var new-obs 1))
                      ))
          (when ready (send self :hot-cell-ready nil)))))

(defmeth datasheet-proto :set-window-scroll-size (new-obs new-var)
  (let ((fw  (send self :field-width))
        (fh  (send self :field-height))
        (lw  (send self :label-width))
        (nobs (send self :nobs))
        (nvar (send self :nvar))
        )
    (send self :set-window-size fw fh lw nvar nobs)))

(defmeth datasheet-proto :set-window-scroll 
                         (new-obs new-var fh fw lw nvar nobs)
  (let ((table-size  (list (+ lw (* nvar fw)) (* (+ 2 nobs) fh)))
        (window-size (send self :size))
        (scroll (send self :scroll)))
    (when (and new-obs (> (+ (second table-size) fh) 
                          (+ (second window-size) (second scroll))))
          (setf (second scroll) (+ (second scroll) fh))
          (apply #'send self :scroll scroll))
    (when (and new-var (> (+ (first table-size) fw) 
                          (+ (first window-size) (first scroll))))
          (setf (first  scroll) (+ (first  scroll) fw))
          (apply #'send self :scroll scroll))))

(defmeth datasheet-proto :expand-labels (n)
  (let ((nobs (length (send self :label-strings))))
    (send self :label-strings 
          (append (send self :label-strings)
                  (mapcar #'(lambda (i) (format nil "~a~d" "Obs" (+ nobs i)))
                          (iseq n))))))

(defmeth datasheet-proto :expand-names (n name)
  (let ((nvar (length (send self :variable-strings))))
    (send self :variable-strings 
          (append (send self :variable-strings)
                  (mapcar #'(lambda (i) (format nil "~a~d" name (+ nvar i)))
                          (iseq n))))))     

(defmeth datasheet-proto :expand-mv-datasheet (new-var new-obs n)
"Args: new-var new-obs n
Method to add N new variables (when NEW-VAR is t) and/or observations (when NEW-OBS is t) to a multivariate datasheet."
  (let ((varname "Var")
        (vartype "Numeric")
        (vars (send self :variable-strings))
        (newstrs nil)
        (types (send self :type-strings))
        (labels (send self :label-strings))
        (nvar (send self :nvar))
        (nobs (send self :nobs)))
    (when new-var   
          (when (equal (subseq (select vars (1- nvar)) 0 3) "Way")
                (setf varname "Way")
                (setf vartype "Category"))
          (send self :nvar (+ nvar n))
          (send self :newvar (+ (send self :newvar) n))
          (send self :expand-names n varname)
          (send self :type-strings (append types (repeat vartype n)))
          (send self :data-matrix-strings
                (bind-columns (send self :data-matrix-strings)
                              (matrix (list nobs n)
                                      (repeat "NIL" (* nobs n))))))
    (when new-obs
          (send self :nobs (+ nobs n))
          (send self :newobs (+ (send self :newobs) n))
          (send self :expand-labels n)
          (send self :data-matrix-strings 
                (bind-rows (send self :data-matrix-strings) 
                           (matrix (list n nvar)
                                   (repeat "NIL" (* n nvar)))))))
  (send self :edited t)
  (send self :set-window-scroll-size new-obs new-var)
  (send self :redraw-now t)
  (send self :redraw))

(defmeth datasheet-proto :make-editable ()
  ;(break)
  (when (not (equal *current-data* (send self :data-object)))
        (setcd (send self :data-object)))
  (edit-data)
  (send (first (send (send self :menu) :items)) :enabled nil)
  )

(defmeth datasheet-proto :expand-mat-datasheet (new-row&col new-mat n)
"Args: new-row&col new-mat
Method to add a new row and column (when new-row&col is t) or new matrix (when new-mat is t) to a matrix datasheet."
  (let ((nvar (send self :nvar))
        (nobs (send self :nobs))
        (nmat (send self :nmat))
        (types (send self :type-strings))
        )
    (when new-row&col
          (dotimes (i n) 
                   (send self :data-matrix-strings 
                         (send self :add-rows&col 
                               (send self :data-matrix-strings) 
                               (send self :nvar)
                               (send self :nmat)))
                   (send self :nvar (+ (send self :nvar) 1)))
          (send self :nobs (+ nobs (* n nmat)))
          (send self :newvar (+ n (send self :newvar)))
          (send self :newobs (+ (send self :newobs) (* n nmat)))
          (send self :expand-names n "Var")
          (send self :type-strings (append types (repeat "Numeric" n)))
          (send self :label-strings nil)
          (send self :create-matrix-label-strings 0 (send self :nmat))
          )
    (when new-mat 
          (send self :nmat (+ nmat n))
          (send self :newmat (+ n (send self :newmat)))
          (send self :nobs (+ nobs (* n nvar)))
          (send self :newobs (+ (send self :newobs) (* n nvar)))
          (send self :matrix-strings 
                (append (send self :matrix-strings) 
                        (mapcar #'(lambda (i) 
                                    (format nil "~a~d" "Mat" (+ nmat i)))
                                (iseq n))))
          (dotimes (i n)
                   (send self :data-matrix-strings
                         (bind-rows (send self :data-matrix-strings) 
                                    (make-array (list nvar nvar)
                                                :initial-element "NIL"))))
          (send self :create-matrix-label-strings nmat n)
          )    
    )
  (send self :edited t)
  (send self :set-window-scroll-size new-mat new-row&col)
  (send *datasheet* :redraw-now t)
  (send self :redraw))

(defmeth datasheet-proto :add-rows&col (themat nvar nmat)
  (let ((nobs (* nvar nvar))
        )
    (setf themat (bind-rows themat (repeat "NIL" nvar)))
    (when (> nmat 1) 
          (dotimes (i (- nmat 1))
                   (setf j (- nmat 1 i)) 
                   (setf rowin (* nvar j))
                   (setf themat 
                         (bind-rows 
                          (select themat (iseq rowin) (iseq nvar))
                          (repeat "NIL" nvar)
                          (select themat 
                                  (iseq rowin (- (first (size themat)) 1)) 
                                  (iseq nvar))))))
    (bind-columns themat (repeat "NIL" (first (size themat))))))

(defmeth datasheet-proto :reverse-cell-color (row col lw fw fh &optional ready)
"Args: row col lw fw fh & optional ready
Reverses color of cell at intersection of col and row, where (in pixels) lw is label width, fw is field (cell) width, fh is field height. Ready indicates if cell ready for typing."
  (let ((x nil) (y nil) (w nil) (h nil) (xf 0) (yf 0)
        (nrow (send self :nobs)) (ncol (send self :nvar)))
#+macintosh(when (= 1 row) (setf yf 1))
#+macintosh(when (= 1 col) (setf xf 1))
    (when (< col 1) (setf xf (- fw lw)))
    (setf x (+ lw 1 xf (* (- col 1) fw)))
    (when (< x 1) (setf x 1))
    (setf y (+ 1 yf (* (+ row 1) fh)))
    (setf w (- fw 1 xf))
    (setf h (- fh 1 yf))
#-macintosh(when (or (= col 0) (= col ncol)) (setf w (1- w)))
#-macintosh(when (or (= row 0) (= row nrow)) (setf h (1- h)))
    (send self :draw-mode 'xor) 
#+macintosh(if ready
           (send self :frame-rect x y w h)
           (send self :paint-rect x y w h))
#-macintosh(send self :frame-rect x y w h)
    (send self :draw-mode 'normal)))

(defmeth datasheet-proto :cell-size-location (row col lw fw fh)
  (let ((x nil) (y nil) (w nil) (h nil) (xf nil) (yf nil)
        )
    (if (= 1 row) (setf yf 1) (setf yf 0))
    (if (= 1 col) (setf xf 1) (setf xf 0))
    (when (< col 1) (setf xf (- fw lw)))
    (setf x (+ lw 1 xf (* (- col 1) fw)))
    (when (< x 1) (setf x 1))
    (setf y (+ 1 yf (* (+ row 1) fh)))
    (setf w (- fw 1 xf))
    (setf h (- fh 1 yf))
    (list x y w h)))

(defmeth datasheet-proto :do-key (c m1 m2)
"Method Args: c m1 m2
Senses character c and shift (m1=t) or option (m2=t)"
  (let ((editable (send self :editable))
        (hot (send self :hot-cell))
        (fw  (send self :field-width))
        (fh  (send self :field-height))
        (lw  (send self :label-width))
        (nobs (send self :nobs))
        (nvar (send self :nvar))
        (x   nil)
        (y   nil)
        ) 
;DISPLAYS CHARACTER TYPED
;(format t "~%~s" c)
    (when 
     editable 
     (when 
      hot
      (setf row (first hot))
      (setf col (second hot))
      (setf x (+ lw 1 (* fw (1- col))))
      (setf y (+ 1 (* fh (+ 1 row))))
#+macintosh      (case c
        ( (#\C-\ #\C-] #\C-M #\Newline #\Tab #\C-C #\C-^ #\C-_ #\C-A #\C-D)
;            left right return? return   tab  enter  up   down  home  end    
         (send self :move-cell c row col nvar nobs x y lw fw fh m1))
        (t
         (send self :store-and-show-char c row col lw fw fh)))
#+msdos    (case c
        ( (  #\;   #\C-M #\[ #\'  #\Tab)
;            left  right up  down   tab
         (send self :move-cell c row col nvar nobs x y lw fw fh m1))
        (t
         (send self :store-and-show-char c row col lw fw fh)))
#+X11   (case c
        ( (  #\;   #\Newline #\[ #\'  #\Tab) ;temp same as msdos
;            left  right up  down   tab
         (send self :move-cell c row col nvar nobs x y lw fw fh m1))
        (t
         (send self :store-and-show-char c row col lw fw fh)))
      ))))

(defmeth datasheet-proto :store-and-show-char (c row col lw fw fh)
"Method Args: c row col lw fw fh
Method to store and show a non cursor moving character c when in row and col of datasheet with lw label width, fw and fh field width and height."
  (let* ((hot-cell-ready (send self :hot-cell-ready))
         (xywh (send self :cell-size-location row col lw fw fh))
         (data-strings (send self :data-matrix-strings))
         (x (first  xywh))
         (y (second xywh))
         (w (third  xywh))
         (h (fourth xywh))
         ) 
    (cond 
      ((not hot-cell-ready) 
       (send self :hot-cell-ready t)
       (send self :erase-rect (1+ x) (1+ y) (- w 2) (- h 2))
       (if (eq c #\C-H) ;if delete
           (send self :hot-cell-string "")
           (send self :hot-cell-string (coerce (list c) 'string))))
      (t
       (if (eq c #\C-H)
           (when (> (length (send self :hot-cell-string)) 0)
                 (send self :hot-cell-string
                       (subseq (send self :hot-cell-string) 0 
                               (1- (length (send self :hot-cell-string))))))
           (send self :hot-cell-string (strcat (send self :hot-cell-string)
                                               (coerce (list c) 'string))))))
    (cond 
      ((and (> row 0) (> col 0)) ;when in main body of table
       (send self :draw-cell-text self ':data-matrix-strings
             row col x y w h fw fh lw 2 (1- row) (1- col))      
;The following statement updates data-object's data after every key-stroke. 
;This is fast enough on my machine, but isnt necessary unless code is 
;changed to update objects dependent on the current-data on the fly. 
;Currently, the close method updates the data.
;      (send (send self :data-object) :data (mapcar #'number-from-string 
;              (combine (send self :data-matrix-strings)))))
       )
      ((< col 1) ;when in labels
       (send self :draw-cell-text self ':label-strings 
             row col x y w h fw fh lw 0 (1- row)))
      ((= row -1) ;when in variable names
       (send self :draw-cell-text self ':variable-strings 
             row col x y w h fw fh lw 1 (1- col)))
      ((= row 0) ;when in variable types
       (send self :draw-cell-text self ':type-strings
             row col x y w h fw fh lw 1 (1- col)))
      )))
    

(defmeth datasheet-proto :draw-cell-text 
  (object message row col x y w h fw fh lw justify element1 &optional element2)
"Args: Object - dataobj; message - message sent to data-object;
row col - of data sheet; x y w h - position and size of cell; fw fh lw field sizes of sheet; justify - 0 1 2 left center right; element1 element2 row and col of dataobj"
  (let ((string nil)
        (maxstring 0))
    (if element2 
        (setf (select (send object message) element1 element2)
              (send self :hot-cell-string))
        (setf (select (send object message) element1)
              (send self :hot-cell-string)))
    (send self :erase-rect (+ x 2) (+ y 2) (- w 3) (- h 3))
    (if element2
        (setf string (select (send object message) element1 element2))
        (setf string (select (send object message) element1))) 
    (if (< col 1) (setf maxstring lw) (setf maxstring fw))
    (when (> (send self :text-width string) (- maxstring 6)) 
          (setf string "*****"))
    (case justify
      (2 (send self :draw-text string (- (+ lw (* col fw)) 3) 
               (- (* fh (+ 2 row)) 3) 2 0))
      (1 (send self :draw-text string (- (+ lw (* col fw)) (floor (/ fw 2))) 
               (- (* fh (+ 2 row)) 3) 1 0))
      (0 (send self :draw-text string 3 
               (- (* fh (+ 2 row)) 3) 0 0)))
    ))

(defmeth datasheet-proto :move-cell (c row col nvar nobs x y lw fw fh m1)
"Method Args: c row col nvar nobs x y lw fw fh m1
Method to move to another cell by simulating a do-click.  Movement character is c.  In row and col. There are nvar and nobs cols and rows.  Simulated click will be at x and y. Fields are fw and fh wide and high. Lables are lw wide. M1 t for shift."
  (when (< col 1) ;when in labels column
     (case c 
       (
#+macintosh(#\C-\ #\C-^) ;left,up
#+msdos    (#\;   #\[)
#+X11    (#\;   #\[)
        (when (> row 1) (send self :do-click x (- y fh) nil nil))) ;up
       (#\Tab
        (if m1 
            (when (> row 1)
                  (send self :do-click 3 (- y fh) nil nil))
            (when (< row nobs)
                  (send self :do-click 3 (+ y fh) nil nil))))
       (#\C-A (send self :do-click (+ lw 3) (+ (* 2 fh) 3) nil nil)) ;home
       (#\C-D (send self :do-click (+ lw -3 (* nvar fw)) 
                    (+ 3 fh (* nobs fh)) nil nil)) ;end
       (t ;down
        (when (< row nobs)
              (send self :do-click x (+ y fh) nil nil)))
       ))
  (when (> col 0) ;when in datasheet columns
     (case c
       (
#+macintosh #\C-\ ;left
#+msdos     #\;
#+X11     #\;
         (when (and (not (and (= col 1) (= row 1)))
                    (not (and (= col 1) (= row -1))))
         (if (> col 1) 
             (send self :do-click (- x fw) y nil nil)
             (send self :do-click (+ lw -3 (* nvar fw)) (- y fh) nil nil))))

       ((#\C-] #\C-M #\C-C #\Newline)  ;right
         (when (not (and (= col nvar) (= row nobs)))
         (if (< col nvar)
             (send self :do-click (+ x fw) y nil nil)
             (send self :do-click (+ lw 3) (+ y fh) nil nil))))
       (#\Tab ; first in next (previous if shift-tab) row
         (cond 
           ((and (> row 1) (< row nobs))
            (if m1
                (send self :do-click (+ lw 3) (- y fh) nil nil)
                (send self :do-click (+ lw 3) (+ y fh) nil nil)))
           ((and (= row 1) (not m1))
            (send self :do-click (+ lw 3) (+ y fh) nil nil))
           ((and (= row nobs) m1)
            (send self :do-click (+ lw 3) (- y fh) nil nil))))
       (
#+macintosh #\C-^ ;up
#+msdos     #\[
#+X11     #\[
         (when (or (> row 1) (= row 0))
               (send self :do-click x (- y fh) nil nil)))
       (
#+macintosh #\C-_ ;down
#+msdos     #\'
#+X11     #\'
         (when (< row nobs)
         (send self :do-click x (+ y fh) nil nil)))
        (#\C-A (send self :do-click (+ lw 3) (+ (* 2 fh) 3) nil nil)) ;home
        (#\C-D (send self :do-click (+ lw -3 (* nvar fw)) 
                    (+ 3 fh (* nobs fh)) nil nil)) ;end
        )))