/*
 * nasd_shutdown.h
 *
 * Structures and definitions for automated cleanup lists
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#ifndef _NASD_SHUTDOWN_H_
#define _NASD_SHUTDOWN_H_

#include <nasd/nasd_options.h>
#include <nasd/nasd_threadstuff.h>

#if defined (DEC_OSF) && defined(KERNEL)
#include <vm/vm_page.h>
#endif /* DEC_OSF && KERNEL */

typedef struct nasd_shutdown_s         nasd_shutdown_t;
typedef struct nasd_shutdown_list_s    nasd_shutdown_list_t;

/*
 * nasd_shutdown
 *
 * Entry in a shutdown list.
 * If proc is NULL, it's a memory entry.
 */
struct nasd_shutdown_s {
  void             (*proc)(void *arg);
  void              *arg;
  int                size;
  int                type;
  char              *name;
  nasd_shutdown_t   *next;
  char              *file;
  int                line;
};

/*
 * nasd_shutdown_list
 *
 * List of shutdowns.
 */
struct nasd_shutdown_list_s {
  NASD_DECLARE_MUTEX(lock)
  nasd_shutdown_t       *list;
  nasd_shutdown_list_t  *next;
};

#define nasd_shutdown_proc(_sl_,_proc_,_arg_) \
  _nasd_shutdown_proc(_sl_, _proc_, _arg_, NASD_STRING(_proc_), \
    __FILE__, __LINE__)

#define nasd_shutdown_mem(_sl_,_addr_,_size_) \
  _nasd_shutdown_mem(_sl_, _addr_, _size_, __FILE__, __LINE__)

#define nasd_shutdown_mutex(_sl_,_addr_) \
  _nasd_shutdown_mutex(_sl_, _addr_, __FILE__, __LINE__)

#define nasd_shutdown_cond(_sl_,_addr_) \
  _nasd_shutdown_cond(_sl_, _addr_, __FILE__, __LINE__)

#define nasd_shutdown_rwlock(_sl_,_addr_) \
  _nasd_shutdown_rwlock(_sl_, _addr_, __FILE__, __LINE__)

#define NASD_SHUTDOWN_ANNOUNCE_NONE        0
#define NASD_SHUTDOWN_ANNOUNCE_PROC   0x0001
#define NASD_SHUTDOWN_ANNOUNCE_MEM    0x0002
#define NASD_SHUTDOWN_ANNOUNCE_MUTEX  0x0004
#define NASD_SHUTDOWN_ANNOUNCE_COND   0x0008
#define NASD_SHUTDOWN_ANNOUNCE_RWLOCK 0x0010
/*      NASD_SHUTDOWN_ANNOUNCE_VMPAGE 0x0020 defined iff (DEC_OSF && KERNEL) */
#define NASD_SHUTDOWN_ANNOUNCE_ALL    0x003f
/*      NASD_SHUTDOWN_ANNOUNCE_SCHED  0x1000 defined iff (LINUX && KERNEL) */

#define NASD_SHUTDOWN_TYPE_BOGUS   0
#define NASD_SHUTDOWN_TYPE_PROC    1
#define NASD_SHUTDOWN_TYPE_MEM     2
#define NASD_SHUTDOWN_TYPE_MUTEX   3
#define NASD_SHUTDOWN_TYPE_COND    4
#define NASD_SHUTDOWN_TYPE_RWLOCK  5
/*      NASD_SHUTDOWN_TYPE_VMPAGE  6    defined iff (DEC_OSF && KERNEL) */

/*
 * from nasd_shutdown.c
 */
nasd_status_t nasd_shutdown_sys_init(void);
nasd_status_t nasd_shutdown_list_init(nasd_shutdown_list_t **slp);
nasd_status_t nasd_shutdown_list_shutdown(nasd_shutdown_list_t *sl,
  int verbose);
nasd_status_t _nasd_shutdown_proc(nasd_shutdown_list_t *sl,
  void (*proc)(void *), void *arg, char *name, char *file, int line);
nasd_status_t _nasd_shutdown_mem(nasd_shutdown_list_t *sl,
  void *addr, int size, char *file, int line);
nasd_status_t _nasd_shutdown_mutex(nasd_shutdown_list_t *sl,
  void *addr, char *file, int line);
nasd_status_t _nasd_shutdown_cond(nasd_shutdown_list_t *sl,
  void *addr, char *file, int line);
nasd_status_t _nasd_shutdown_rwlock(nasd_shutdown_list_t *sl,
  void *addr, char *file, int line);
void nasd_shutdown_cleanup(void);

#if defined(DEC_OSF) && defined(KERNEL)

nasd_status_t _nasd_shutdown_vmpage(nasd_shutdown_list_t *sl,
  void *addr, char *file, int line);
#define nasd_shutdown_vmpage(_sl_,_page_) \
  _nasd_shutdown_vmpage(_sl_,_page_,__FILE__,__LINE__)

#define NASD_SHUTDOWN_TYPE_VMPAGE          6
#define NASD_SHUTDOWN_ANNOUNCE_VMPAGE 0x0020

#endif /* DEC_OSF && KERNEL */

#if defined(LINUX) && defined(KERNEL)

/*
 * "ANNOUNCE_SCHED" isn't an announcement- it checks
 * to see if we've smashed the stack before and after
 * each shutdown op
 *
 * Note that we define this in such a way that it isn't
 * part of "ANNOUNCE_ALL"
 */

#define NASD_SHUTDOWN_ANNOUNCE_SCHED  0x1000

#endif /* LINUX && KERNEL */

#endif /* !_NASD_SHUTDOWN_H_ */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
