/* sread.c - SIO read primitive test program
 */

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sio_fs.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <nasd/nasd_timer.h>

void usage()
{
  printf("sread [opts] source target\n");
  printf("  -t show times  -v be verbose  -b <x> set block size to <x>\n");
  exit(1);
}

int
main(int argc, char **argv)
{
  sio_file_io_list_t fl[10];
  sio_mem_io_list_t ml[10];
  sio_transfer_len_t xfer;
  sio_async_handle_t handles[2];
  sio_return_t rv;
  sio_count_t ind;
  sio_async_status_t status;
  sio_transfer_len_t bytes;
  void *buf;
  char *p;
  char *src_fname;
  char *dst_fname;
  int bsize;
  int done;
  int fd;
  int fd1;
  int nb;
  int i;
  int c;
  int show_times = 0;
  int verbose = 0;
  int chunk_size;
  int chk_cnt = 0;
  nasd_timer_t timer;
  extern char *optarg;
  extern int   optind;

  chunk_size = 256 * 1024;
  while ((c = getopt(argc, argv, "?tvb:")) != EOF) {
    switch (c) {
    case 'b':				/* set block (chunk) size */
      if (sscanf(optarg, "%d", &chunk_size) != 1) {
	usage();
      }
      break;
    case 't':				/* print timings */
      show_times = 1;
      break;
    case 'v':
      verbose = 1;
      break;
    default:
      usage();
      break;
    }
  }
	  
  if (argc - optind != 2) {
    usage();
  }
  src_fname = argv[optind++];
  dst_fname = argv[optind++];

  if (verbose) {
    printf("Chunk size %d\n", chunk_size);
  }
  rv = sio_open(&fd, src_fname, SIO_MODE_READ, NULL , 0);
  if (rv != SIO_SUCCESS) {
    fprintf(stderr, "sread:(can't create %s): %s\n", 
	    dst_fname, sio_error_string(rv));
    exit(1);
  }

  fl[0].offset = 0;
  fl[0].size = chunk_size;
  fl[0].stride = 0;
  fl[0].element_cnt = 1;

  buf = (void *)malloc(chunk_size);
  if (buf == NULL) {
    fprintf(stderr, "malloc failed\n");
    exit(1);
  }
	
  ml[0].addr = buf;
  ml[0].size = chunk_size;
  ml[0].stride = 0;
  ml[0].element_cnt = 1;

  fd1 = open(dst_fname, O_CREAT | O_RDWR, 0644);
  if (fd1 <=0) {
    fprintf(stderr,"sread: failed to open %s\n", dst_fname);
    exit(1);
  }

  done = FALSE;
  bytes = 0;
  if (show_times)
    NASD_TM_START(&timer);
  while (!done) {	
    fl[0].size = chunk_size;
    ml[0].size = chunk_size;
    rv = sio_sg_read(fd, fl, 1, ml, 1, &xfer);
    if (xfer < chunk_size)
      done= TRUE;
    fl[0].offset += xfer;	
    if (rv != SIO_SUCCESS) {
      fprintf(stderr, "sread: sio_sg_read failed: %s\n", 
	      sio_error_string(rv));
    }
    nb = write(fd1, buf, xfer);
    if (nb<xfer)
      fprintf(stderr, "sread: write failed size=%d/%d\n",
	      xfer, nb);
    bytes += xfer;
    if (verbose) {
      /* print a . each MB */
      if(!(bytes % (1024 * 1024))) {
	write(1, ".", 1);
      }
    }
  }
  if (show_times) {
    NASD_TM_STOP(&timer);
  }
  if (verbose)
    printf("\n");
  if (show_times) {
    if (bytes == 0)
      printf("No data transfered.\n");
    else {
      double secs = (double)0;
      double mbytes = (double)0;
      double mbps = (double)0;
      nasd_timespec_t diff;

      NASD_TM_ELAPSED_TS(&timer, &diff);
      secs = ((double)diff.ts_nsec/(double)1000000000.0) + (double)diff.ts_sec;
      mbytes = (double)bytes / (double)(1024 * 1024);
      mbps = mbytes / secs;
      printf("Transfered %gMB/s (%" NASD_64u_FMT
	     " total bytes in %g seconds)\n", mbps,
	     bytes, secs);
    }
  }

  rv = sio_close(fd);
  if (rv != SIO_SUCCESS) {
    fprintf(stderr, "sio_close: %s\n", sio_error_string(rv));
  }

  close(fd1);	
  free(buf);

  exit(0);
}

/*
 * Local variables:
 * comment-column: 40
 * c-basic-offset: 2
 * c-indentation-style: gnu
 * End:
 */
