/*
 * nasd_cheops_aio.c
 *
 * io queue which manages multiple parallel ios for striped objects
 *
 * Authors: Khalil Amiri, CMU SCS/ECE, July 18 1997
 *          Sean Levy, CMU SCS, July 1999
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1996,1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

#include <nasd/nasd_options.h>
#include <stdio.h>
#include <malloc.h>
#include <errno.h>
#include <stdlib.h>

#include <nasd/nasd_types.h>
#include <nasd/nasd_mem.h>
#include <nasd/nasd_threadstuff.h>
#include <nasd/nasd_cheops_common.h>
#include <nasd/nasd_cheops_types.h>
#include <nasd/nasd_cheops_raidmap.h> 
#include <nasd/nasd_cheops_ios.h>
#include <nasd/nasd_cheops_client_internal.h>

#define _NASD_CHEOPS_AIO_DEBUG 0

static _nasd_cheops_ioq_t *gl_io_queue = NULL;

/* local functions */
int _nasd_cheops_ioq_enq(_nasd_cheops_ioq_t *, _nasd_cheops_io_t *);
void _nasd_cheops_ioq_deq(_nasd_cheops_ioq_t *, _nasd_cheops_io_t **);

/* Queue operations */
int
_nasd_cheops_ioq_init(
  void)
{
	int rc;
	_nasd_cheops_ioq_t *ioq;

	NASD_Malloc(ioq, sizeof(_nasd_cheops_ioq_t), (_nasd_cheops_ioq_t *));
	if (ioq==NULL) {
	  fprintf(stderr, "CHEOPS: malloc failed..\n");
	  return ENOMEM;
	} else
    gl_io_queue = ioq;

#if _NASD_CHEOPS_AIO_DEBUG
	fprintf(stderr, "CHEOPS: _nasd_cheops_ioq_init(): gl_io_queue=0x%x\n",
          (unsigned int) gl_io_queue);
#endif /* _NASD_HAPPY_CHEOPS_DEBUG */

	ioq->q_head = NULL;
	ioq->q_tail = NULL;
  ioq->q_n_sig = 0;
  ioq->q_n = 0;
  ioq->q_n_deq = 0;

	rc = nasd_mutex_init(&(ioq->q_mutex));
	if (rc) {
	  fprintf(stderr, "CHEOPS: Error: can not initialize mutex\n");
	  return rc;
	}

	rc = nasd_cond_init(&(ioq->q_not_empty));
	if (rc) {
	  fprintf(stderr, "CHEOPS: Error: can not initialize condition variable\n");
	  return rc;
	}

  return 0;
}

int
_nasd_cheops_ioq_enq(
  _nasd_cheops_ioq_t    *ioq,
  _nasd_cheops_io_t     *new_item)
{
  NASD_ASSERT(ioq != NULL);
  NASD_ASSERT(new_item != NULL);
	NASD_LOCK_MUTEX(ioq->q_mutex);
	if (ioq->q_head == NULL) {
	  new_item->qnext = NULL;
	  new_item->qprev= NULL;
	  ioq->q_head = new_item;
	  ioq->q_tail = new_item;
	} else {
	  new_item->qnext = ioq->q_head;
	  new_item->qprev = NULL;
 	  ioq->q_head->qprev = new_item;
	  ioq->q_head = new_item;
	}
  ioq->q_n_sig++;
  ioq->q_n++;
  NASD_SIGNAL_COND(ioq->q_not_empty);
	NASD_UNLOCK_MUTEX(ioq->q_mutex);
  return 0;
}

void
_nasd_cheops_ioq_deq(
  _nasd_cheops_ioq_t     *ioq,
  _nasd_cheops_io_t     **ret_item)
{
  int trips;

  trips = 0;
  NASD_ASSERT(ioq != NULL);
  NASD_ASSERT(ret_item != NULL);
	NASD_LOCK_MUTEX(ioq->q_mutex);
  while (ioq->q_head == NULL) {
    trips++;
    NASD_WAIT_COND(ioq->q_not_empty, ioq->q_mutex);
#if _NASD_CHEOPS_AIO_DEBUG > 1
    fprintf(stderr, "trip#%d through the loop waiting on 0x%08lx\n", trips);
#endif /* _NASD_CHEOPS_AIO_DEBUG > 1 */
  }
	if (ioq->q_head == ioq->q_tail) {
#if _NASD_CHEOPS_AIO_DEBUG > 2
    fprintf(stderr, "last queue element taken\n");
#endif /* _NASD_CHEOPS_AIO_DEBUG > 2 */
	  *ret_item = ioq->q_head;
	  ioq->q_head = ioq->q_tail = NULL;
	} else {
    _nasd_cheops_io_t *item_bef_last;

	  *ret_item = ioq->q_tail;
	  item_bef_last = ioq->q_tail->qprev;
 	  item_bef_last->qnext = NULL;
	  ioq->q_tail = item_bef_last;
	}
  ioq->q_n_deq++;
  ioq->q_n--;
#if _NASD_CHEOPS_AIO_DEBUG > 2
  fprintf(stderr, "%d i/o queue elements left\n", ioq->q_n);
#endif /* _NASD_CHEOPS_AIO_DEBUG > 2 */
	NASD_UNLOCK_MUTEX(ioq->q_mutex);
}
  
nasd_boolean_t 
_nasd_cheops_ioq_empty(
  _nasd_cheops_ioq_t            *ioq)
{
	if (ioq->q_head == NULL) 
    return NASD_TRUE;
	else
    return NASD_FALSE;
}

void 
_nasd_cheops_asm_freeall(
  nasd_asm_t    *a)
{
  NASD_stripe_access_t *sac, *psac;
  int i;

  if (a == NULL)
    return;
  sac = a->first_stripe_access;	 
  while (sac != NULL) {
    psac = sac;
    sac = sac->next;
    NASD_Free(psac, sizeof(NASD_stripe_access_t));
  }

  NASD_Free(a, sizeof(nasd_asm_t));
}

void 
_nasd_cheops_io_freeall(
  _nasd_cheops_aggr_io_t        *aggr_io)
{
  int i=0;
	int in_size = aggr_io->in_datalen;
	int out_size = aggr_io->in_datalen;
  _nasd_cheops_io_t *cio, *pcio;

  cio = aggr_io->first_io;
  while (cio) {
	  pcio = cio;
	  cio= cio->cnext;
	  NASD_Free(pcio, sizeof(_nasd_cheops_io_t));
	  i++;
	}

	nasd_mutex_destroy(&(aggr_io->flag_mutex));	
	nasd_cond_destroy(&(aggr_io->ios_all_done));

  NASD_Free(aggr_io, sizeof(_nasd_cheops_aggr_io_t));

#if _NASD_CHEOPS_AIO_DEBUG > 0
  fprintf(stderr, "CHEOPS: freed io (size=%d/%d), num_ios=%d\n",
          in_size, out_size, i);
#endif

}

void 
_nasd_cheops_init_io(
  NASD_pda_t                    *pda,
  _nasd_cheops_aggr_io_t        *pio, 
  _nasd_cheops_io_t             *io)
{
  io->ni = pda->data_obj_id.nasd_identifier;
  io->di = pda->data_obj_id.disk_identifier;
	io->pni = pda->parity_obj_id.nasd_identifier;
	io->pdi = pda->parity_obj_id.disk_identifier;
	io->in_offset = pda->start_address;
	io->in_datalen = pda->len;
	io->out_datalen = 0;
	io->bms_targ = 0;
	io->databuf = pda->buf+(pda->databuf_offset);
	io->parent_io = pio;	
	io->lblock = pda->lblock;
	io->qnext = io->qprev = NULL;
}

int 
_nasd_cheops_enq_io(
  _nasd_cheops_io_t     *io)
{
  return _nasd_cheops_ioq_enq(gl_io_queue, io);
}

void
_nasd_cheops_deq_io(
  _nasd_cheops_io_t    **ret_io)
{
  _nasd_cheops_ioq_deq(gl_io_queue, ret_io);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
