/*
 * Copyright (c) 2009 Nhat Minh Lê <nhat.minh.le@huoc.org>
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * Simple fixed-size bit vector implementation. Please note that the
 * size of these vectors is not stored anywhere, so in order to use
 * these, you'll need to know the size.
 */

#ifndef REGXML_BITSTRING_H
#define REGXML_BITSTRING_H

#include <sys/cdefs.h>

#include <assert.h>
#include <limits.h>
#include <stdint.h>
#include <stdlib.h>

#ifndef __NetBSD__
#include <regxml/compat.h>
#endif

/*
 * C99 stdint fixed-width types are guaranteed not to include any
 * padding, so we can assume the size of the type is actually the
 * number of usable bytes.
 *
 * Use of larger stdint types may improve performance with complex
 * patterns, and may be slower with small patterns.
 */
#if UINT64_MAX > UCHAR_MAX && UINT64_MAX <= UINT_MAX
#define REGXML_BIT_WIDTH 64
#define REGXML_BIT_1U ((uint64_t)1)
typedef uint64_t regxml_bitstr_t;
#elif UINT32_MAX > UCHAR_MAX && UINT32_MAX <= UINT_MAX
#define REGXML_BIT_WIDTH 32
#define REGXML_BIT_1U ((uint32_t)1)
typedef uint32_t regxml_bitstr_t;
#elif UINT16_MAX > UCHAR_MAX && UINT16_MAX <= UINT_MAX
#define REGXML_BIT_WIDTH 16
#define REGXML_BIT_1U ((uint16_t)1)
typedef uint16_t regxml_bitstr_t;
#else
#define REGXML_BIT_WIDTH CHAR_BIT
#define REGXML_BIT_1U 1U
typedef unsigned char regxml_bitstr_t;
#endif

#define regxml_bitstr_size(n) ((size_t)(((n)-1) / REGXML_BIT_WIDTH + 1))
#define regxml_bit_decl(name, n) name[regxml_bitstr_size(n)]
#define REGXML_BIT_DECL regxml_bit_decl

#define REGXML_BIT_FOREACH(i, s, n)					    \
do {									    \
	int _regxml_bit_k, _regxml_bit_ni;				    \
	regxml_bitstr_t _regxml_bit_x;					    \
									    \
	(i) = 0;							    \
	for (_regxml_bit_k = 0; (i) < (n); ++_regxml_bit_k) {		    \
		_regxml_bit_x = (s)[_regxml_bit_k];			    \
		_regxml_bit_ni = (i) + REGXML_BIT_WIDTH;		    \
		while (_regxml_bit_x != 0 && (i) < (n)) {		    \
			if (_regxml_bit_x & REGXML_BIT_1U)

#define REGXML_BIT_ENDFOREACH						    \
			_regxml_bit_x >>= 1;				    \
			++(i);						    \
		}							    \
		(i) = _regxml_bit_ni;					    \
	}								    \
} while (/* CONSTCOND */ 0)

static __inline regxml_bitstr_t *regxml_bit_alloc(int);
static __inline void regxml_bit_free(regxml_bitstr_t *);

static __inline void regxml_bit_set(regxml_bitstr_t *, int);
static __inline void regxml_bit_clear(regxml_bitstr_t *, int);
static __inline int regxml_bit_test(regxml_bitstr_t *, int);

static __inline int regxml_bit_cmp(const regxml_bitstr_t *,
    const regxml_bitstr_t *, int);
static __inline int regxml_bit_iszero(regxml_bitstr_t *, int);
static __inline void regxml_bit_zero(regxml_bitstr_t *, int);
static __inline void regxml_bit_copy(regxml_bitstr_t * __restrict,
    const regxml_bitstr_t *, int);
static __inline void regxml_bit_and(regxml_bitstr_t * __restrict,
    const regxml_bitstr_t *, int);
static __inline void regxml_bit_or(regxml_bitstr_t * __restrict,
    const regxml_bitstr_t *, int);
static __inline void regxml_bit_xor(regxml_bitstr_t * __restrict,
    const regxml_bitstr_t *, int);
static __inline int regxml_bit_andand(const regxml_bitstr_t *,
    const regxml_bitstr_t *, int);
static __inline int regxml_bit_oror(const regxml_bitstr_t *,
    const regxml_bitstr_t *, int);
static __inline int regxml_bit_xorxor(const regxml_bitstr_t *,
    const regxml_bitstr_t *, int);

static __inline regxml_bitstr_t *
regxml_bit_alloc(int _n)
{
	return calloc(regxml_bitstr_size(_n), sizeof(regxml_bitstr_t));
}

static __inline void
regxml_bit_free(regxml_bitstr_t *_v)
{
	free(_v);
}

static __inline void
regxml_bit_set(regxml_bitstr_t *_v, int _i)
{
	_DIAGASSERT(_v != NULL);
	_v[_i/REGXML_BIT_WIDTH] |= REGXML_BIT_1U << _i%REGXML_BIT_WIDTH;
}

static __inline void
regxml_bit_clear(regxml_bitstr_t *_v, int _i)
{
	_DIAGASSERT(_v != NULL);
	_v[_i/REGXML_BIT_WIDTH] &= ~(REGXML_BIT_1U << _i%REGXML_BIT_WIDTH);
}

static __inline int
regxml_bit_test(regxml_bitstr_t *_v, int _i)
{
	_DIAGASSERT(_v != NULL);
	return _v[_i/REGXML_BIT_WIDTH] &
	    REGXML_BIT_1U << _i%REGXML_BIT_WIDTH;
}

static __inline void
regxml_bit_copy(regxml_bitstr_t * __restrict _u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		_u[_i] = _v[_i];
}

static __inline int
regxml_bit_cmp(const regxml_bitstr_t *_u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i) {
		if (_u[_i] != _v[_i])
			return _u[_i] - _v[_i];
	}
	return 0;
}

static __inline void
regxml_bit_zero(regxml_bitstr_t *_v, int _n)
{
	int _i;

	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		_v[_i] = 0;
}

static __inline int
regxml_bit_iszero(regxml_bitstr_t *_v, int _n)
{
	int _i;

	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i) {
		if (_v[_i] != 0)
			return 0;
	}
	return 1;
}

static __inline void
regxml_bit_and(regxml_bitstr_t * __restrict _u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		_u[_i] &= _v[_i];
}

static __inline void
regxml_bit_or(regxml_bitstr_t * __restrict _u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		_u[_i] |= _v[_i];
}

static __inline void
regxml_bit_xor(regxml_bitstr_t * __restrict _u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		_u[_i] ^= _v[_i];
}

static __inline int
regxml_bit_andand(const regxml_bitstr_t *_u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		if (_u[_i] & _v[_i])
			return 1;
	return 0;
}

static __inline int
regxml_bit_oror(const regxml_bitstr_t *_u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		if (_u[_i] | _v[_i])
			return 1;
	return 0;
}

static __inline int
regxml_bit_xorxor(const regxml_bitstr_t *_u, const regxml_bitstr_t *_v,
    int _n)
{
	int _i;

	_DIAGASSERT(_u != NULL);
	_DIAGASSERT(_v != NULL);

	_n = regxml_bitstr_size(_n);
	for (_i = 0; _i < _n; ++_i)
		if (_u[_i] ^ _v[_i])
			return 1;
	return 0;
}

#endif	/* !REGXML_BITSTRING_H */
