/*-
 * Copyright (c) 2015 Taylor R. Campbell
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <assert.h>
#include <errno.h>
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "err.h"
#include "emalloc.h"

#include "strung.h"

struct string
string_emalloc(size_t len)
{
	char *p;

	if (len == SIZE_MAX)
		errx(1, "Cannot allocate %zu + 1 bytes", len);
	p = emalloc(len + 1);
	p[len] = '\0';

	return string(p, len);
}

struct string
string_erealloc(struct string s, size_t len)
{
	char *p;

	if (len == SIZE_MAX)
		errx(1, "Cannot allocate %zu + 1 bytes", len);
	p = erealloc(string_ptr(s), (len + 1));
	p[len] = '\0';

	return string(p, len);
}

struct string
string_ecopy(struct string s)
{
	const size_t len = string_len(s);
	char *p = emalloc(len + 1);

	(void)memcpy(p, string_ptr(s), len);
	p[len] = '\0';

	return string(p, len);
}

void
string_free(struct string s)
{

	free(s.s_ptr);
	s.s_ptr = NULL;		/* paranoia */
	s.s_len = 0;
}

struct string
string_adoptz(char *p, size_t limit)
{
	size_t i;

	assert(0 < limit);

	for (i = 0; i < limit; i++)
		if (p[i] == '\0')
			break;
	/*
	 * Caller provides up to limit bytes, and must guarantee
	 * NUL-termination within those bytes.
	 */
	assert(i < limit);
	assert(p[i] == '\0');
	p[limit - 1] = '\0';

	return string(p, i);
}

struct string
string_edupz(const char *p, size_t maxlen)
{
	char *q;
	size_t i;

	for (i = 0; i < maxlen; i++)
		if (p[i] == '\0')
			break;
	assert(i <= maxlen);
	assert(p[i] == '\0');

	q = emalloc(i + 1);
	(void)memcpy(q, p, i);
	q[i] = '\0';

	return string(q, i);
}

static int
string_valen(va_list va, size_t *len)
{
	const struct string *t;

	while ((t = va_arg(va, const struct string *)) != NULL) {
		if (string_len(*t) > (SIZE_MAX - *len - 1)) {
			errno = ENOMEM;
			return -1;
		}
		*len += string_len(*t);
	}

	return 0;
}

static void
string_vaconcat(va_list va, char **pp)
{
	const struct string *t;

	while ((t = va_arg(va, const struct string *)) != NULL) {
		(void)memcpy(*pp, string_ptr(*t), string_len(*t));
		*pp += string_len(*t);
	}
}

struct string
string_econcat(struct string s0, struct string s1)
{
	return string_econcatn(&s0, &s1, NULL);
}

struct string
string_econcatn(const struct string *s, ...)
{
	struct string result;
	char *p;
	size_t len = 0;
	va_list va;

	len += string_len(*s);
	va_start(va, s);
	if (string_valen(va, &len) == -1)
		errx(1, "string_concat");
	va_end(va);

	result = string_emalloc(len);
	p = string_ptr(result);
	(void)memcpy(p, string_ptr(*s), string_len(*s));
	p += string_len(*s);
	va_start(va, s);
	string_vaconcat(va, &p);
	va_end(va);
	assert(string_ptr(result) <= p);
	assert(string_len(result) == (size_t)(p - string_ptr(result)));
	assert(string_ptr(result)[string_len(result)] == '\0');

	return result;
}

struct string
string_econcat_into(struct string s0, struct string s1)
{
	return string_econcatn_into(&s0, &s1, NULL);
}

struct string
string_econcatn_into(const struct string *s, ...)
{
	struct string result;
	char *p;
	size_t len = 0;
	va_list va;

	len += string_len(*s);
	va_start(va, s);
	if (string_valen(va, &len) == -1)
		errx(1, "string_concat_into");
	va_end(va);

	result = string_erealloc(*s, len);
	p = string_ptr(result) + string_len(*s);
	va_start(va, s);
	string_vaconcat(va, &p);
	va_end(va);
	assert(string_ptr(result) <= p);
	assert(string_len(result) == (size_t)(p - string_ptr(result)));
	assert(string_ptr(result)[string_len(result)] == '\0');

	return result;
}

#define	MIN(A, B)	((A) < (B)? (A) : (B))

int
string_order(struct string a, struct string b)
{
	size_t na = string_len(a), nb = string_len(b);
	int order;

	order = memcmp(string_ptr(a), string_ptr(b), MIN(na, nb));
	if (order != 0)
		return order;

	if (na < nb)
		return -1;
	else if (na > nb)
		return +1;
	else
		return 0;
}

/* vis to ASCIZ to make strings safe to print to the terminal.  */
char *
string_evisciiz(struct string s)
{
	size_t i, n = string_len(s);
	char *v;
	size_t vi, vn;

	if (n > ((SIZE_MAX / 4) - 1)) {
		errno = ENOMEM;
		err(1, "string_visciiz");
	}

	vn = n*4;
	v = emalloc(vn + 1);
	vi = 0;
	for (i = 0; i < n; i++) {
		unsigned char ch = string_ptr(s)[i];
		const char *esc = "\\^";
		const char *noesc = "";

		assert(vi <= vn);
		if (ch == 0 || ch == 9 || ch == 10 || ch == 32 || ch == '\\') {
			vi += snprintf(v + vi, vn + 1 - vi, "\\%03o", ch);
			continue;
		}

		if (ch & 0200) {
			esc = "\\M^";
			noesc = "\\M-";
			ch &= 0177;
		}
		assert(strlen(esc) <= 3);
		assert(strlen(noesc) <= 3);

		if (ch == 127)
			vi += snprintf(v + vi, vn + 1 - vi, "%s%c", esc, '?');
		else if (ch < 32)
			vi += snprintf(v + vi, vn + 1 - vi, "%s%c", esc,
			    ch + '@');
		else
			vi += snprintf(v + vi, vn + 1 - vi, "%s%c", noesc, ch);
	}
	assert(vi <= vn);
	v[vi] = '\0';

	return v;
}
