/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: audiorec.c,v 1.28 2006/03/30 09:21:26 jakemsr Exp $
 */

#include "includes.h"

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/mman.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bsdav.h"

volatile sig_atomic_t audiorec_quit;


char	*audio_dev;
char	*outpath;
uint8_t	*buffer;
FILE	*outfile;
size_t	 buffer_size;
int	 audio_fd;
int	 verbose;
int	 format;
int	 channels;
int	 sample_rate;
int	 samples_per_buffer;

extern char *__progname;

void	 usage(void);
void	 catchsignal(int);
int	 audio_grab_samples(void);
int	 stream(void);
void	 cleanup(int);

/* getopt externs */
extern char *optarg;
extern int opterr;
extern int optind;
extern int optopt;
extern int optreset;


void
usage(void)
{
	printf("usage: %s [-lv] [-c channels] [-e encoding] [-f file]\n"
	       "          [-o output] [-p pidfile] [-r rate]\n",
	    __progname);

	exit(1);
}

void
catchsignal(int signal)
{
	switch(signal) {
	case SIGALRM:
		/* FALLTHROUGH */
	case SIGINT:
		audiorec_quit = 1;
		break;
	default:
		break;
	}
}


int
audio_grab_samples(void)
{
size_t	left;
off_t	offset;
ssize_t	received;

	for (left = buffer_size, offset = 0; left > 0;) {
		received = read(audio_fd, buffer + offset, left);
		if (received == 0) {
			/* EOF from /dev/audio ? */
			warnx("audio grab: received == 0");
		}
		if (received < 0) {
			if (errno == EINTR) {
				received = 0;
				/* warnx("audio grab: interrupted"); */
			} else {
				warn("audio grab");
				return (1);
			}
		}
		if (received > left) {
			warnx("read returns more bytes than requested");
			warnx("requested: %llu, returned: %lld",
			    (unsigned long long)left, (long long)received);
			return (1);
		}
		offset += received;
		left -= received;
	}
	return (left);
}


int
stream(void)
{
#if defined(HAVE_SUN_AUDIO)
audio_info_t audio_if;
#endif
struct sigaction act;
struct itimerval tim;
struct timeval tp_start;
struct timeval tp_end;
struct timeval tp_run;
double	 run_time;
long	 frames;
double	 usecs_per_frame;
int	 sequence;
int	 frames_ready;
int	 got_frame;

	audiorec_quit = 0;
	frames = 0;
	frames_ready = 0;
	sequence = 64;

	memset(&act, 0, sizeof(act));
	sigemptyset(&act.sa_mask);
	act.sa_handler = catchsignal;
	sigaction(SIGALRM, &act, NULL);
	sigaction(SIGINT, &act, NULL);

	usecs_per_frame = (double)buffer_size * 1000000 * 8 /
	    (sample_rate * bsdav_aud_fmts[format].bps * channels);

	if (verbose > 1)
		warnx("usecs_per_frame = %f", usecs_per_frame);

	/* timeout if there is no audio data within 1 second */
	tim.it_value.tv_sec = 1;
	setitimer(ITIMER_REAL, &tim, NULL);

#if defined(HAVE_SUN_AUDIO)
	AUDIO_INITINFO(&audio_if);
	/* unpause recording.  needed for ASYNC */
	audio_if.record.pause = 0;
	if (ioctl(audio_fd, AUDIO_SETINFO, &audio_if) < 0) {
		warn("AUDIO_SETINFO");
		return (1);
	}
#endif

	/* drain out the buffer before really starting */
	audio_grab_samples();

	gettimeofday(&tp_start, NULL);

	while (audiorec_quit == 0) {

		got_frame = 0;

		frames_ready = bsdav_get_audhw_buf_pos(audio_fd,
		    BSDAV_AUDMODE_REC) / buffer_size;
		while ((frames_ready < 1) && (audiorec_quit == 0)) {
			usleep((useconds_t)usecs_per_frame / 8);
			frames_ready =
			    bsdav_get_audhw_buf_pos(audio_fd,
			    BSDAV_AUDMODE_REC) / buffer_size;
		}

		while ((frames_ready > 0) && (audiorec_quit == 0)) {
			if (audio_grab_samples() > 0)
				warnx("incomplete read   ");
			frames_ready--;

			/* reset the timeout timer */
			setitimer(ITIMER_REAL, &tim, NULL);

			if (bsdav_write_frame_data(outfile,
			    buffer, buffer_size, 0) != 0)
				warn("write data, size = %llu   ",
				    (unsigned long long)buffer_size);
			else
				frames++;
			got_frame = 1;
		}

		if (audiorec_quit == 1)
			break;

		if ((got_frame == 1) && (frames % sequence == 0)) {
			gettimeofday(&tp_end, NULL);
			timersub(&tp_end, &tp_start, &tp_run);
			run_time = tp_run.tv_sec +
			    (double)tp_run.tv_usec / 1000000;
			if (verbose > 1) {
				fprintf(stderr,
				    "%s: samples: %09ld, seconds: %09.3f, "
				    "sample rate: %08.3f\r", __progname,
			    	    samples_per_buffer * frames,
				    run_time, (double)samples_per_buffer *
				    frames / run_time);
				fflush(stderr);
			}
		}
	}
	gettimeofday(&tp_end, NULL);

	fprintf(stderr, "\n");

	/* shutdown signals */

	tim.it_value.tv_sec = 0;
	setitimer(ITIMER_REAL, &tim, NULL);

	act.sa_handler = SIG_DFL;
	sigaction(SIGALRM, &act, NULL);
	sigaction(SIGINT, &act, NULL);

	if (verbose > 0) {
		timersub(&tp_end, &tp_start, &tp_run);
		run_time = tp_run.tv_sec + (double)tp_run.tv_usec / 1000000;
		warnx("%ld samples: %f sec: %f Hz",
		    frames * samples_per_buffer, run_time,
		    (frames * samples_per_buffer) / run_time);
	}

	return (0);
}


void
cleanup(int excode)
{
	if (audio_fd >= 0) {
		close(audio_fd);
		audio_fd = -1;
	}

	if (outfile != NULL) {
		fclose(outfile);
		outfile = NULL;
	}

	if (excode > 0)
		exit(excode);
}


int
main(int argc, char *argv[])
{
const char *errstr;
char	*pidfile;
int	 ch;
int	 lflag;

	/* defaults */
	outpath = NULL;
	audio_dev = DEFAULT_AUDIO_DEVICE;
	samples_per_buffer = 2048;
	sample_rate = 48000;
	channels = 2;
	format = BSDAV_AUDFMT_SLLE;
	pidfile = NULL;
	lflag = 0;

	while ((ch = getopt(argc, argv, "lvc:e:f:r:o:p:")) != -1) {
		switch (ch) {
		case 'c':
			channels = (int)strtonum(optarg, 0, 2, &errstr);
			if (errstr != NULL)
				errx(1, "channels is %s: %s", errstr, optarg);
			break;
		case 'e':
			format = bsdav_find_aud_fmt(optarg);
			if (format < 0)
				errx(1, "invalid encoding %s", optarg);
			break;
		case 'f':
			audio_dev = optarg;
			break;
		case 'l':
			lflag++;
			break;
		case 'o':
			outpath = optarg;
			break;
		case 'p':
			pidfile = optarg;
			break;
		case 'r':
			sample_rate = (int)strtonum(optarg, 0, 96000, &errstr);
			if (errstr != NULL)
				errx(1, "rate is %s: %s", errstr, optarg);
			break;
		case 'v':
			verbose++;
			break;
		default:
			usage();
		}
	}

	argc -= optind;
	argv += optind;

	audio_fd = open(audio_dev, O_RDONLY);
	if (audio_fd < 0) {
		warn("%s", audio_dev);
		cleanup(1);
	}

	if (lflag > 0) {
		bsdav_list_audio_formats(audio_dev, audio_fd);
		cleanup(0);
		exit(0);
	}

	if (outpath == NULL) {
		warnx("output file not specified, exiting");
		cleanup(1);
	} else {
		if (pidfile != NULL) {
			if (bsdav_write_pid(pidfile) > 0) {
				warnx("couldn't write pid file");
				cleanup(1);
			}
		}

		if (bsdav_audio_init(audio_fd, BSDAV_AUDMODE_REC, format,
		    channels, sample_rate) != 0) {
			warnx("could not initialize audio device");
			cleanup(1);
		}

		if (verbose > 0) {
			warnx("format: %s", bsdav_aud_fmts[format].name);
			warnx("channel: %d", channels);
			warnx("sample rate: %d", sample_rate);
		}

		buffer_size = samples_per_buffer *
		    ((bsdav_aud_fmts[format].bps / 8) * channels);

		if (verbose > 2) {
			warnx("buffer_size: %llu", (unsigned long long)buffer_size);
			warnx("samples_per_buffer: %d", samples_per_buffer);
		}

		if ((buffer = (uint8_t *)malloc(buffer_size)) == NULL) {
			warn("buffer");
			cleanup(1);
		}

		if ((outfile = fopen(outpath, "w")) == NULL) {
			warn("%s", outpath);
			cleanup(1);
		}

		if (stream() > 0)
			cleanup(1);

		fclose(outfile);
		outfile = NULL;
	}

	cleanup(0);
	return (0);
}

