package imap

import (
	"context"
	"fmt"
	"log/slog"

	"github.com/emersion/go-imap/v2"
	"github.com/emersion/go-imap/v2/imapclient"
)

// Client wraps the IMAP client with required capabilities.
type Client struct {
	client *imapclient.Client
	logger *slog.Logger
}

// Message represents an IMAP message metadata without body.
type Message struct {
	UID   imap.UID
	Flags []imap.Flag
}

// MessageWithBody represents an IMAP message with body.
type MessageWithBody struct {
	Message
	Body []byte
}

// Connect connects to an IMAP server and verifies required capabilities.
func Connect(ctx context.Context, server, username, password string, plaintext bool, handler *imapclient.UnilateralDataHandler, logger *slog.Logger) (*Client, error) {
	var client *imapclient.Client
	var err error

	if plaintext {
		logger.Warn("Connecting to IMAP server (plaintext)", "server", server)
		client, err = imapclient.DialInsecure(server, &imapclient.Options{
			UnilateralDataHandler: handler,
		})
	} else {
		logger.Info("Connecting to IMAP server", "server", server)
		client, err = imapclient.DialTLS(server, &imapclient.Options{
			UnilateralDataHandler: handler,
		})
	}
	if err != nil {
		return nil, fmt.Errorf("failed to connect: %w", err)
	}

	if username != "" {
		logger.Info("Logging in.", "username", username)
		if err := client.Login(username, password).Wait(); err != nil {
			_ = client.Close() // Best effort close
			return nil, fmt.Errorf("login failed: %w", err)
		}
		logger.Info("Logged in successfully.")
	} else {
		logger.Info("Skipping logging in.")
	}

	// Check for required capabilities (capabilities may only be advertised post-auth).
	caps := client.Caps()
	requiredCaps := []imap.Cap{"UIDPLUS", "NOTIFY", "CONDSTORE", "QRESYNC"}

	logger.Debug("Server capabilities", "capabilities", caps)
	for _, reqCap := range requiredCaps {
		if !caps.Has(reqCap) {
			_ = client.Close() // Best effort close
			return nil, fmt.Errorf("server does not support required capability: %s", reqCap)
		}
	}

	logger.Info("Server capabilities verified", "capabilities", requiredCaps)

	logger.Debug("Enabling QRESYNC")
	enableCmd := client.Enable(imap.CapQResync)
	if _, err := enableCmd.Wait(); err != nil {
		_ = client.Close() // Best effort close
		return nil, fmt.Errorf("failed to enable QRESYNC: %w", err)
	}
	logger.Info("QRESYNC enabled")

	return &Client{
		client: client,
		logger: logger,
	}, nil
}

// Close closes the IMAP connection.
func (c *Client) Close() error {
	return c.client.Close()
}

// SelectResult contains the result of a SELECT operation.
type SelectResult struct {
	UIDValidity   uint32
	HighestModSeq uint64
	VanishedUIDs  []imap.UID
}

// SelectMailbox selects a mailbox for subsequent operations.
func (c *Client) SelectMailbox(ctx context.Context, mailbox string, qresync *imap.SelectQResyncOptions) (*SelectResult, error) {
	c.logger.Debug("selecting mailbox", "mailbox", mailbox)

	selectOpts := &imap.SelectOptions{
		CondStore: true,
		QResync:   qresync,
	}

	if qresync != nil {
		c.logger.Debug("using QRESYNC", "uidvalidity", qresync.UIDValidity, "modseq", qresync.ModSeq)
	}

	selectCmd := c.client.Select(mailbox, selectOpts)
	data, err := selectCmd.Wait()
	if err != nil {
		return nil, fmt.Errorf("failed to select mailbox %s: %w", mailbox, err)
	}

	result := &SelectResult{
		UIDValidity:   data.UIDValidity,
		HighestModSeq: data.HighestModSeq,
	}

	if len(data.VanishedUIDs) > 0 {
		uids, ok := data.VanishedUIDs.Nums()
		if ok {
			result.VanishedUIDs = uids
			c.logger.Debug("received VANISHED UIDs", "count", len(result.VanishedUIDs))
		}
	}

	c.logger.Debug("mailbox selected", "mailbox", mailbox, "uidvalidity", result.UIDValidity, "highestmodseq", result.HighestModSeq)

	return result, nil
}

// ListMessages lists all messages in the currently selected mailbox.
func (c *Client) ListMessages(ctx context.Context) (map[imap.UID]Message, error) {
	messages := make(map[imap.UID]Message)

	// Fetch all UIDs, flags.
	uidSet := imap.UIDSet{}
	uidSet.AddRange(1, 0) // 1:* (all UIDs)

	fetchCmd := c.client.Fetch(uidSet, &imap.FetchOptions{
		UID:   true,
		Flags: true,
	})

	for {
		msg := fetchCmd.Next()
		if msg == nil {
			break
		}

		// Collect buffered data.
		buf, err := msg.Collect()
		if err != nil {
			_ = fetchCmd.Close() // Best effort close
			return nil, fmt.Errorf("failed to collect message data: %w", err)
		}

		messages[buf.UID] = Message{
			UID:   buf.UID,
			Flags: filterFlags(buf.Flags),
		}
	}

	if err := fetchCmd.Close(); err != nil {
		return nil, fmt.Errorf("fetch failed: %w", err)
	}

	c.logger.Debug("listed messages", "count", len(messages))

	return messages, nil
}

// ListUIDs lists all message UIDs in the currently selected mailbox.
func (c *Client) ListUIDs(ctx context.Context) (map[imap.UID]bool, error) {
	uids := make(map[imap.UID]bool)

	uidSet := imap.UIDSet{}
	uidSet.AddRange(1, 0) // 1:* (all UIDs)

	fetchCmd := c.client.Fetch(uidSet, &imap.FetchOptions{
		UID: true,
	})

	for {
		msg := fetchCmd.Next()
		if msg == nil {
			break
		}

		buf, err := msg.Collect()
		if err != nil {
			_ = fetchCmd.Close() // Best effort close
			return nil, fmt.Errorf("failed to collect UID: %w", err)
		}

		uids[buf.UID] = true
	}

	if err := fetchCmd.Close(); err != nil {
		return nil, fmt.Errorf("fetch UIDs failed: %w", err)
	}

	c.logger.Debug("listed UIDs", "count", len(uids))

	return uids, nil
}

// FetchChangesSince fetches messages that have changed since the given MODSEQ.
// Returns a map of UIDs to Messages without body.
func (c *Client) FetchChangesSince(ctx context.Context, sinceModSeq uint64) (map[imap.UID]Message, error) {
	c.logger.Debug("fetching changes since MODSEQ", "modseq", sinceModSeq)

	messages := make(map[imap.UID]Message)

	uidSet := imap.UIDSet{}
	uidSet.AddRange(1, 0) // 1:* (all UIDs)
	fetchCmd := c.client.Fetch(uidSet, &imap.FetchOptions{
		UID:          true,
		Flags:        true,
		ChangedSince: sinceModSeq,
	})

	for {
		msg := fetchCmd.Next()
		if msg == nil {
			break
		}

		// Collect buffered data.
		buf, err := msg.Collect()
		if err != nil {
			_ = fetchCmd.Close() // Best effort close
			return nil, fmt.Errorf("failed to collect message data: %w", err)
		}

		messages[buf.UID] = Message{
			UID:   buf.UID,
			Flags: filterFlags(buf.Flags),
		}
	}

	if err := fetchCmd.Close(); err != nil {
		return nil, fmt.Errorf("fetch failed: %w", err)
	}

	c.logger.Debug("fetched changed messages", "count", len(messages), "since_modseq", sinceModSeq)

	return messages, nil
}

// FetchMessage fetches a complete message including body.
func (c *Client) FetchMessage(ctx context.Context, uid imap.UID) (*MessageWithBody, error) {
	c.logger.Debug("fetching message", "uid", uid)

	seqSet := imap.UIDSet{}
	seqSet.AddNum(uid)

	fetchCmd := c.client.Fetch(seqSet, &imap.FetchOptions{
		UID:         true,
		Flags:       true,
		BodySection: []*imap.FetchItemBodySection{{Peek: true}},
	})

	msg := fetchCmd.Next()
	if msg == nil {
		if err := fetchCmd.Close(); err != nil {
			return nil, fmt.Errorf("fetch failed: %w", err)
		}
		return nil, fmt.Errorf("message not found: %d", uid)
	}

	// Collect buffered data.
	buf, err := msg.Collect()
	if err != nil {
		_ = fetchCmd.Close() // Best effort close
		return nil, fmt.Errorf("failed to collect message data: %w", err)
	}

	// Get body from buffer.
	var body []byte
	for _, section := range buf.BodySection {
		body = section.Bytes
		break
	}

	if err := fetchCmd.Close(); err != nil {
		return nil, fmt.Errorf("fetch failed: %w", err)
	}

	return &MessageWithBody{
		Message: Message{
			UID:   buf.UID,
			Flags: filterFlags(buf.Flags),
		},
		Body: body,
	}, nil
}

// AppendMessage appends a message to a mailbox.
func (c *Client) AppendMessage(ctx context.Context, mailbox string, flags []imap.Flag, body []byte) (imap.UID, error) {
	c.logger.Debug("appending message", "mailbox", mailbox, "size", len(body))

	appendCmd := c.client.Append(mailbox, int64(len(body)), &imap.AppendOptions{
		Flags: flags,
	})

	// Write message body.
	if _, err := appendCmd.Write(body); err != nil {
		return 0, fmt.Errorf("failed to write message: %w", err)
	}

	if err := appendCmd.Close(); err != nil {
		return 0, fmt.Errorf("failed to finalise message append: %w", err)
	}

	// Wait for APPENDUID response.
	data, err := appendCmd.Wait()
	if err != nil {
		return 0, fmt.Errorf("append failed: %w", err)
	}

	c.logger.Debug("message appended", "uid", data.UID, "uidvalidity", data.UIDValidity)

	return data.UID, nil
}

// DeleteMessage marks a message as deleted and expunges it.
func (c *Client) DeleteMessage(ctx context.Context, uid imap.UID) error {
	c.logger.Debug("deleting message", "uid", uid)

	uidSet := imap.UIDSet{}
	uidSet.AddNum(uid)

	// Set \Deleted flag.
	storeCmd := c.client.Store(uidSet, &imap.StoreFlags{
		Op:    imap.StoreFlagsAdd,
		Flags: []imap.Flag{imap.FlagDeleted},
	}, nil)

	if err := storeCmd.Close(); err != nil {
		return fmt.Errorf("failed to mark message as deleted: %w", err)
	}

	expungeCmd := c.client.UIDExpunge(uidSet)
	if err := expungeCmd.Close(); err != nil {
		return fmt.Errorf("expunge failed: %w", err)
	}

	c.logger.Debug("message deleted", "uid", uid)

	return nil
}

// UpdateFlags updates the flags for a message by adding and removing specific flags.
// Returns the resulting flags from the server after the update.
func (c *Client) UpdateFlags(ctx context.Context, uid imap.UID, toAdd, toRemove []imap.Flag) ([]imap.Flag, error) {
	c.logger.Debug("updating flags", "uid", uid, "add", toAdd, "remove", toRemove)

	uidSet := imap.UIDSet{}
	uidSet.AddNum(uid)

	var resultFlags []imap.Flag
	var err error

	if len(toAdd) > 0 {
		c.logger.Debug("adding flags", "uid", uid, "flags", toAdd)
		resultFlags, err = c.storeFlags(uidSet, imap.StoreFlagsAdd, toAdd)
		if err != nil {
			return nil, err
		}
	}

	if len(toRemove) > 0 {
		c.logger.Debug("removing flags", "uid", uid, "flags", toRemove)
		resultFlags, err = c.storeFlags(uidSet, imap.StoreFlagsDel, toRemove)
		if err != nil {
			return nil, err
		}
	}

	c.logger.Debug("flags updated", "uid", uid, "result_flags", resultFlags)
	return resultFlags, nil
}

// storeFlags executes a STORE command and returns the resulting flags.
func (c *Client) storeFlags(uidSet imap.UIDSet, op imap.StoreFlagsOp, flags []imap.Flag) ([]imap.Flag, error) {
	storeCmd := c.client.Store(uidSet, &imap.StoreFlags{
		Op:    op,
		Flags: flags,
	}, nil)

	var resultFlags []imap.Flag
	for {
		msg := storeCmd.Next()
		if msg == nil {
			break
		}

		buf, err := msg.Collect()
		if err != nil {
			_ = storeCmd.Close() // Best effort close
			return nil, fmt.Errorf("failed to collect STORE response: %w", err)
		}

		resultFlags = filterFlags(buf.Flags)
	}

	if err := storeCmd.Close(); err != nil {
		return nil, fmt.Errorf("store command failed: %w", err)
	}
	return resultFlags, nil
}

// ListMailboxes lists all available mailboxes.
func (c *Client) ListMailboxes(ctx context.Context) ([]string, error) {
	c.logger.Debug("listing mailboxes")

	var mailboxes []string

	listCmd := c.client.List("", "*", nil)
	for {
		mbox := listCmd.Next()
		if mbox == nil {
			break
		}
		mailboxes = append(mailboxes, mbox.Mailbox)
	}

	if err := listCmd.Close(); err != nil {
		return nil, fmt.Errorf("list failed: %w", err)
	}

	c.logger.Debug("Discovered mailboxes", "count", len(mailboxes))

	return mailboxes, nil
}

// InitNotify starts NOTIFY mode to receive push notifications (RFC 5465).
// Returns immediately after the NOTIFY command completes successfully.
func (c *Client) InitNotify(options *imap.NotifyOptions) (*imapclient.NotifyCommand, error) {
	c.logger.Debug("initializing NOTIFY", "items", len(options.Items))

	notifyCmd, err := c.client.Notify(options)
	if err != nil {
		return nil, fmt.Errorf("failed to start NOTIFY: %w", err)
	}

	// Wait for the NOTIFY command to complete (successfully or with error).
	if err := notifyCmd.Wait(); err != nil {
		return nil, fmt.Errorf("NOTIFY command failed: %w", err)
	}

	c.logger.Info("NOTIFY enabled")

	return notifyCmd, nil
}

// WaitForNotifyEnd blocks until NOTIFY ends due to overflow.
func (c *Client) WaitForNotifyEnd(ctx context.Context, notifyCmd *imapclient.NotifyCommand) error {
	// Monitor for overflow and context cancellation.
	select {
	case <-notifyCmd.Overflow():
		c.logger.Warn("NOTIFICATIONOVERFLOW received, notifications disabled by server")
		return fmt.Errorf("notification overflow")
	case <-ctx.Done():
		c.logger.Debug("disabling NOTIFY")
	}

	return ctx.Err()
}

// Noop sends a NOOP command to the server (keep-alive).
// Returns nil if the connection is still alive.
func (c *Client) Noop(ctx context.Context) error {
	noop := c.client.Noop()
	// FIXME: should take context.
	return noop.Wait()
}

// filterFlags removes flags that cannot be stored in Maildir.
// Only the 5 standard IMAP flags can be represented in Maildir filenames.
func filterFlags(flags []imap.Flag) []imap.Flag {
	var filtered []imap.Flag
	for _, flag := range flags {
		switch flag {
		case imap.FlagSeen, imap.FlagAnswered, imap.FlagFlagged, imap.FlagDeleted, imap.FlagDraft:
			filtered = append(filtered, flag)
		}
	}
	return filtered
}
