/******************************************************************************
 * Top contributors (to current version):
 *   Andrew Reynolds, Morgan Deters, Aina Niemetz
 *
 * This file is part of the cvc5 project.
 *
 * Copyright (c) 2009-2022 by the authors listed in the file AUTHORS
 * in the top-level source directory and their institutional affiliations.
 * All rights reserved.  See the file COPYING in the top-level source
 * directory for licensing information.
 * ****************************************************************************
 *
 * The preprocessor of the SolverEngine.
 */

#include "cvc5_private.h"

#ifndef CVC5__SMT__PREPROCESSOR_H
#define CVC5__SMT__PREPROCESSOR_H

#include <memory>

#include "smt/env_obj.h"
#include "smt/expand_definitions.h"
#include "smt/process_assertions.h"
#include "theory/booleans/circuit_propagator.h"

namespace cvc5::internal {

class TheoryEngine;

namespace preprocessing {
class PreprocessingPassContext;
}
namespace prop {
class PropEngine;
}

namespace smt {

class AbstractValues;
class PreprocessProofGenerator;

/**
 * The preprocessor module of an SMT engine.
 *
 * This class is responsible for:
 * (1) preprocessing the set of assertions from input before they are sent to
 * the SMT solver,
 * (2) implementing methods for expanding and simplifying formulas. The latter
 * takes into account the substitutions inferred by this class.
 */
class Preprocessor : protected EnvObj
{
 public:
  Preprocessor(Env& env, AbstractValues& abs, SolverEngineStatistics& stats);
  ~Preprocessor();
  /**
   * Finish initialization
   */
  void finishInit(TheoryEngine* te, prop::PropEngine* pe);
  /**
   * Process the assertions that have been asserted in argument as. Returns
   * true if no conflict was discovered while preprocessing them.
   *
   * @param as The assertions.
   */
  bool process(Assertions& as);
  /**
   * Clear learned literals from the Boolean propagator.
   */
  void clearLearnedLiterals();
  /** Get learned literals */
  std::vector<Node> getLearnedLiterals() const;
  /**
   * Cleanup, which deletes the processing passes owned by this module. This
   * is required to be done explicitly so that passes are deleted before the
   * objects they refer to in the SolverEngine destructor.
   */
  void cleanup();
  /**
   * Simplify a formula without doing "much" work.  Does not involve
   * the SAT Engine in the simplification, but uses the current
   * definitions, assertions, and the current partial model, if one
   * has been constructed.  It also involves theory normalization.
   *
   * @param n The node to simplify
   * @return The simplified term.
   */
  Node simplify(const Node& n);
  /**
   * Expand the definitions in a term or formula n.  No other
   * simplification or normalization is done.
   *
   * @param n The node to expand
   * @return The expanded term.
   */
  Node expandDefinitions(const Node& n);
  /** Same as above, with a cache of previous results. */
  Node expandDefinitions(const Node& n, std::unordered_map<Node, Node>& cache);
  /** Same as above, for a list of assertions, updating in place */
  void expandDefinitions(std::vector<Node>& ns);
  /**
   * Enable proofs for this preprocessor. This must be called
   * explicitly since we construct the preprocessor before we know
   * whether proofs are enabled.
   *
   * @param pppg The preprocess proof generator of the proof manager.
   */
  void enableProofs(PreprocessProofGenerator* pppg);

 private:
  /** Reference to the abstract values utility */
  AbstractValues& d_absValues;
  /**
   * A circuit propagator for non-clausal propositional deduction.
   */
  theory::booleans::CircuitPropagator d_propagator;
  /**
   * User-context-dependent flag of whether any assertions have been processed.
   */
  context::CDO<bool> d_assertionsProcessed;
  /** The preprocessing pass context */
  std::unique_ptr<preprocessing::PreprocessingPassContext> d_ppContext;
  /** Expand definitions module, responsible for expanding definitions */
  ExpandDefs d_exDefs;
  /**
   * Process assertions module, responsible for implementing the preprocessing
   * passes.
   */
  ProcessAssertions d_processor;
};

}  // namespace smt
}  // namespace cvc5::internal

#endif
