/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file implements classes SKGPayeeObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgpayeeobject.h"

#include <klocalizedstring.h>

#include "skgcategoryobject.h"
#include "skgdocumentbank.h"
#include "skgoperationobject.h"
#include "skgtraces.h"

SKGPayeeObject::SKGPayeeObject(): SKGPayeeObject(nullptr)
{}

SKGPayeeObject::SKGPayeeObject(SKGDocument* iDocument, int iID): SKGNamedObject(iDocument, QLatin1String("v_payee"), iID)
{}

SKGPayeeObject::~SKGPayeeObject()
    = default;

SKGPayeeObject::SKGPayeeObject(const SKGPayeeObject& iObject) = default;

SKGPayeeObject::SKGPayeeObject(const SKGObjectBase& iObject)

{
    if (iObject.getRealTable() == QLatin1String("payee")) {
        copyFrom(iObject);
    } else {
        *this = SKGNamedObject(iObject.getDocument(), QLatin1String("v_payee"), iObject.getID());
    }
}

SKGPayeeObject& SKGPayeeObject::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGPayeeObject& SKGPayeeObject::operator= (const SKGPayeeObject& iObject)
{
    copyFrom(iObject);
    return *this;
}

SKGError SKGPayeeObject::createPayee(SKGDocumentBank* iDocument,
                                     const QString& iName,
                                     SKGPayeeObject& oPayee,
                                     bool iSendPopupMessageOnCreation)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err)

    // Check if refund is already existing
    if (iName.isEmpty()) {
        oPayee = SKGPayeeObject(nullptr, 0);
    } else if (iDocument != nullptr) {
        iDocument->getObject(QLatin1String("v_payee"), "t_name='" % SKGServices::stringToSqlString(iName) % QLatin1Char('\''), oPayee);
        if (oPayee.getID() == 0) {
            // No, we have to create it
            oPayee = SKGPayeeObject(iDocument);
            err = oPayee.setName(iName);
            IFOKDO(err, oPayee.save())

            if (!err && iSendPopupMessageOnCreation) {
                err = iDocument->sendMessage(i18nc("Information message", "Payee '%1' has been created", iName), SKGDocument::Positive);
            }
        }
    }

    return err;
}

SKGError SKGPayeeObject::getOperations(SKGListSKGObjectBase& oOperations) const
{
    SKGError err = getDocument()->getObjects(QLatin1String("v_operation"),
                   "r_payee_id=" % SKGServices::intToString(getID()),
                   oOperations);
    return err;
}

SKGError SKGPayeeObject::setAddress(const QString& iAddress)
{
    return setAttribute(QLatin1String("t_address"), iAddress);
}

QString SKGPayeeObject::getAddress() const
{
    return getAttribute(QLatin1String("t_address"));
}

SKGError SKGPayeeObject::setClosed(bool iClosed)
{
    return setAttribute(QLatin1String("t_close"), iClosed ? QLatin1String("Y") : QLatin1String("N"));
}

bool SKGPayeeObject::isClosed() const
{
    return (getAttribute(QLatin1String("t_close")) == QLatin1String("Y"));
}

SKGError SKGPayeeObject::bookmark(bool iBookmark)
{
    return setAttribute(QLatin1String("t_bookmarked"), iBookmark ? QLatin1String("Y") : QLatin1String("N"));
}

bool SKGPayeeObject::isBookmarked() const
{
    return (getAttribute(QLatin1String("t_bookmarked")) == QLatin1String("Y"));
}

SKGError SKGPayeeObject::getCategory(SKGCategoryObject& oCategory) const
{
    SKGError err = getDocument()->getObject(QLatin1String("v_category"), "id=" % getAttribute(QLatin1String("r_category_id")), oCategory);
    return err;
}

SKGError SKGPayeeObject::setCategory(const SKGCategoryObject& iCategory)
{
    return setAttribute(QLatin1String("r_category_id"), SKGServices::intToString(iCategory.getID()));
}

SKGError SKGPayeeObject::merge(const SKGPayeeObject& iPayee)
{
    SKGError err;

    SKGObjectBase::SKGListSKGObjectBase ops;
    IFOKDO(err, iPayee.getOperations(ops))
    int nb = ops.count();
    for (int i = 0; !err && i < nb; ++i) {
        SKGOperationObject op(ops.at(i));
        err = op.setPayee(*this);
        IFOKDO(err, op.save(true, false))
    }

    IFOKDO(err, iPayee.remove(false))
    return err;
}


