/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <stdlib.h>
#include <time.h>
#include <string.h>

#include <qprogressdialog.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qsize.h>
#include <qfont.h>
#include <qpushbutton.h>
#include <qtooltip.h>
#include <qglobal.h>

#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kseparator.h>

#include "maildlg.h"
#include "mlofile.h"
#include "senderaliases.h"
#include "global.h"




MailClient::MailClient(): DwSmtpClient()
{
	connected = false;
	srand(time(0));
}


MailClient::~MailClient()
{
	if (connected)
		disconnect();
}


bool MailClient::SendDataMult(const DwString &aStr)
{
	return SendDataMult(aStr.data(), aStr.length());
}


bool MailClient::SendDataMult(const char *aBuf, int aBufLen)
{
	return PSend(aBuf, aBufLen) == aBufLen;
}


void MailClient::createBoundary()
{
	int i;

	strcpy(boundary, "0_=_vwxy");
	for (i=8; i < 39; i++)
		boundary[i] = 65 + (int)(26.0 * rand() / (RAND_MAX + 1.0));
	boundary[39] = 0;
}


bool MailClient::connect(const char *server, int port)
{
	if (Open(server, port))
		if (Helo() == 250)
			connected = true;
		else
			Close();

	return connected;
}


bool MailClient::sendHeader(const char *from,  const char *to, const char *subj)
{
	DwString msg;

	createBoundary();

	if (Mail(from) != 250)
		return false;

	if (Rcpt(to) != 250)
		return false;

	if (Data() != 354)
		return false;

	msg.clear();

	if (*subj) {
		msg.append("Subject: ");
		msg.append(subj);
		msg.append("\r\012");
	}

	msg.append("X-Mailer: KMLOFax ");
	msg.append(VERSION);
	msg.append("\r\012MIME-Version: 1.0\r\012");
	msg.append("Content-Type: multipart/mixed;\r\012 boundary=\"");
	msg.append(boundary);
	msg.append("\"\r\012\r\012");

	return SendDataMult(msg);
}


bool MailClient::sendBody(const char *body)
{
	DwString msg;
	char buff[1000];
	int i = 0, j = 0;

	msg.assign("--");
	msg.append(boundary);
	msg.append("\r\012Content-Type: text/plain; charset=iso-8859-1\r\012");
	msg.append("Content-Transfer-Encoding: 8bit\r\012\r\012");

	if (!SendDataMult(msg))
		return false;

	while (body[i]) {
		buff[j++] = body[i++];
		if ((!body[i]) || (buff[j-1] == '\n') || (j > 996)) {
			if (buff[j-1] == '\n')
				j--;
			if ((j == 1) && (buff[0] == '.'))
				buff[j++] = '.';
			buff[j++] = '\r';
			buff[j++] = '\012';
			buff[j] = 0;
			if (!SendDataMult(buff, j))
				return false;
			j = 0;
		}
	}

	return true;
}


bool MailClient::sendFile(FILE *stream)
{
	char alphabet[64];
	DwString msg;
	unsigned char ibuff[3000], c1, c2, c3;
	char obuff[80];
	size_t i, j, pad, result;
	QProgressDialog *prog;

	for (i=0; i < 26; i++) {
		alphabet[i] = 'A' + i;
		alphabet[26 + i] = 'a' + i;
	}
	for (i=0; i < 10; i++)
		alphabet[52 + i] = '0' + i;
	alphabet[62] = '+';
	alphabet[63] = '/';
		
	fseek(stream, 0, SEEK_END);
	
	msg.assign("--");
	msg.append(boundary);
	msg.append("\r\012Content-Type: image/tiff; name=\"facsimile.tif\"\r\012");
	msg.append("Content-Transfer-Encoding: base64\r\012");
	msg.append("Content-Disposition: attachment; filename=\"facsimile.tif\"\r\012\r\012");

	if (!SendDataMult(msg)) {
		return false;
	}

	prog = new QProgressDialog(i18n("Sending mail..."), i18n("Cancel"), (int)ftell(stream), 0, "mailprogress", true);
	Q_CHECK_PTR(prog);
	prog->setCaption(i18n("Mail Sending"));

	rewind(stream);
	prog->setProgress(0);
	
	j = 0;
	while ((result = fread((void *)ibuff, 1, 3000, stream)) > 0) {
		if (prog->wasCancelled()) {
			delete prog;
			return false;
		}

		i = 0;
		while (i < result) {
			c1 = ibuff[i++];
			if (i >= result) {
				pad = 2;
				c2 = 0;
				c3 = 0;
			} else {
				c2 = ibuff[i++];
				if (i >= result) {
					pad = 1;
					c3 = 0;
				} else {
					c3 = ibuff[i++];
					pad = 0;
				}
			}
			obuff[j++] = alphabet[c1 >> 2];
			obuff[j++] = alphabet[((c1 << 4) | (c2 >> 4)) & 0x3f];
			if (pad < 2)
				obuff[j++] = alphabet[((c2 << 2) | (c3 >> 6)) & 0x3f];
			else
				obuff[j++] = '=';
			if (pad < 1)
				obuff[j++] = alphabet[c3 & 0x3f];
			else
				obuff[j++] = '=';
			if (j >= 76) {
				obuff[j++] = '\r';
				obuff[j++] = '\012';
				obuff[j] = 0;
				if (!SendDataMult(obuff, j)) {
					delete prog;
					return false;
				}
				j = 0;
			}
		}
		prog->setProgress((int)ftell(stream));
	}
	if (j) {
		obuff[j++] = '\r';
		obuff[j++] = '\012';
		obuff[j] = 0;
		if (!SendDataMult(obuff, j)) {
			delete prog;
			return false;
		}
	}

	delete prog;

	msg.assign("\r\012--");
	msg.append(boundary);
	msg.append("--\r\012");

	return SendData(msg) == 250;
}


void MailClient::disconnect()
{
	Quit();
	Close();
}



MailDlg::MailDlg(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QGroupBox *gb;
	QGridLayout *grid;
	QLabel *label;
	QPushButton *help, *send, *cancel;
	QSize size;
	QFont font;
	KSeparator *separator;

	config = kapp->config();
	
	vbox = new QVBoxLayout(this, marginHint());

	gb = new QGroupBox(this);

	grid = new QGridLayout(gb, 2, 2, marginHint());

	label = new QLabel(i18n("&To:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	header_to = new QLineEdit(gb);
	label->setBuddy(header_to);
	font = header_to->font();
	font.setFamily("courier");
	header_to->setFont(font);
	header_to->setMinimumSize(header_to->sizeHint());
	grid->addWidget(header_to, 0, 1);

	label = new QLabel(i18n("&Subject:"), gb);
	label->setMinimumSize(label->sizeHint());

	grid->addWidget(label, 1, 0);

	header_subject = new QLineEdit(gb);
	label->setBuddy(header_subject);
	header_subject->setFont(font);
	header_subject->setMinimumSize(header_subject->sizeHint());
	grid->addWidget(header_subject, 1, 1);

	grid->setColStretch(1, 1);

	grid->activate();

	vbox->addWidget(gb);

	body = new QMultiLineEdit(this);
	body->setFont(font);
	
	vbox->addWidget(body, 1);

	label = new QLabel(QString("%1: facsimile.tif (image/tiff)").arg(i18n("Attachment")), this);
	label->setMinimumSize(label->sizeHint());
	label->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	
	vbox->addWidget(label);

	separator = new KSeparator(KSeparator::HLine, this);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	help = new QPushButton(i18n("&Help"), this);
	send = new QPushButton(i18n("&Send"), this);
	send->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);

	size = help->sizeHint().expandedTo(send->sizeHint().expandedTo(cancel->sizeHint()));

	help->setFixedSize(size);
	send->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addWidget(help);
	hbox->addStretch();
	hbox->addWidget(send);
	hbox->addWidget(cancel);

	vbox->activate();

	config->setGroup(ID_PREF_GROUP_GENERAL);
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(header_to, i18n("Email address of the receiver"));
		QToolTip::add(header_subject, i18n("Subject of the email"));
		QToolTip::add(body, i18n("Body of the email"));
	}

	connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(send, SIGNAL(clicked()), SLOT(slotSend()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	resize(400, 300);

	setCaption(i18n("Mail"));
}


MailDlg::~MailDlg()
{
}


void MailDlg::mailFax(const QString &name)
{
	QString sender;
	TiffFile tifffile;
	
	tifffile.setName(expandPath(name));
	if (tifffile.open(IO_ReadOnly)) {
		sender = SENDER_ALIAS(tifffile.sender());
		tifffile.close();
	}
	
	m_name = name;

	if (sender.isEmpty())
		header_subject->setText(i18n("Facsimile"));
	else
		header_subject->setText(i18n("Facsimile from %1").arg(sender));
	
	body->setText(i18n("This is a facsimile file forwarded with KMLOFax."));
}


void MailDlg::slotHelp()
{
	kapp->invokeHelp("mail-help");
}


void MailDlg::slotSend()
{
	MailClient mclient;
	QString from, server, name, org, email, reply;
	int port;
	FILE *f;
	
	config->setGroup(ID_PREF_GROUP_MAIL);
	server = config->readEntry(ID_PREF_MAIL_SERVER, PREF_MAIL_SERVER);
	port = config->readNumEntry(ID_PREF_MAIL_PORT, PREF_MAIL_PORT);
	name = config->readEntry(ID_PREF_MAIL_NAME, PREF_MAIL_NAME);
	org = config->readEntry(ID_PREF_MAIL_ORG, PREF_MAIL_ORG);
	email = config->readEntry(ID_PREF_MAIL_EMAIL, PREF_MAIL_EMAIL);
	reply = config->readEntry(ID_PREF_MAIL_REPLY, PREF_MAIL_REPLY);
	
	if (server.isEmpty()) {
		KMessageBox::sorry(this, i18n("No SMTP server specified."), i18n("Mail Error"));
		return;
	}

	if (email.isEmpty()) {
		KMessageBox::sorry(this, i18n("No sender specified."), i18n("Mail Error"));
		return;
	}

	if (header_to->text().isEmpty()) {
		KMessageBox::sorry(this, i18n("No receiver specified."), i18n("Mail Error"));
		return;
	}

	if (!mclient.connect(server.latin1(), port)) {
		KMessageBox::sorry(this, i18n("Cannot connect to server."), i18n("SMTP Error"));
		return;
	}

	from = name;

	if (name.isEmpty())
		from = email;
	else {
		from += " <";
		from += email;
		from += ">";
	}

	if (!mclient.sendHeader(from.latin1(), header_to->text().latin1(), header_subject->text().latin1())) {
		KMessageBox::sorry(this, i18n("Cannot send mail header."), i18n("SMTP Error"));
		mclient.disconnect();
		return;
	}

	if (!mclient.sendBody(body->text().latin1())) {
		KMessageBox::sorry(this, i18n("Cannot send mail body."), i18n("SMTP Error"));
		mclient.disconnect();
		return;
	}

	if (!(f = fopen(expandPath(m_name).latin1(), "r"))) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("SMTP Error"));
		mclient.disconnect();
		return;
	}

	if (!mclient.sendFile(f)) {
		KMessageBox::sorry(this, i18n("Cannot send file."), i18n("SMTP Error"));
	}

	mclient.disconnect();

	accept();
}
