/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifndef TAMSETUPDLG_H
#define TAMSETUPDLG_H


#include <qmap.h>
#include <qstring.h>
#include <qwidget.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlistview.h>
#include <qpushbutton.h>

#include <kdialogbase.h>
#include <kdialog.h>




struct TAMEntryKey
{
	TAMEntryKey(const QCString &group = 0, const QCString &key = 0): mGroup(group), mKey(key) {}

	QCString mGroup;
	QCString mKey;
};


inline bool operator < (const TAMEntryKey &k1, const TAMEntryKey &k2)
{
	int result = qstrcmp(k1.mGroup.data(), k2.mGroup.data());
	if (result != 0)
		return (result < 0);

	result = qstrcmp(k1.mKey.data(), k2.mKey.data());

	return (result < 0);
}


typedef QMap<TAMEntryKey,QCString> TAMEntryMap;
typedef QMap<TAMEntryKey,QCString>::Iterator TAMEntryMapIterator;
typedef QMap<TAMEntryKey,QCString>::ConstIterator TAMEntryMapConstIterator;



class TAMConfig
{
public:
	TAMConfig();
	~TAMConfig();
	
	int numberOfTAMs();

	void setGroup(const QCString &group);
	void setTAM(int num);

	void deleteTAM(int num);

	void addTAM(int model, bool fax);
	
	void readFile(const QString &fname);
	void writeFile(const QString &fname);

	void writeEntry(const QCString &key, const QCString &value);
	void writeStrEntry(const QCString &key, const QString &value);
	void writeNumEntry(const QCString &key, int value);
	void writeBoolEntry(const QCString &key, bool value);

	QCString readEntry(const QCString &key);
	QString readStrEntry(const QCString &key);
	int readNumEntry(const QCString &key);
	bool readBoolEntry(const QCString &key);

	QString readTAMName();
	QString readTAMMSN();

	bool isFax();
	bool isTAM();
	
private:
	QCString mGroup;
	int mNumber;
	TAMEntryMap *entry_map;
};


inline int TAMConfig::numberOfTAMs()
{ return mNumber; }


inline void TAMConfig::setGroup(const QCString& group)
{	mGroup = group; }


inline void TAMConfig::setTAM(int num)
{	mGroup.sprintf("TAM_%02d", num); }

		
inline QString TAMConfig::readTAMName()
{ return readStrEntry("Name"); }


inline QString TAMConfig::readTAMMSN()
{ return readStrEntry("MSN"); }


inline bool TAMConfig::isFax()
{ return readBoolEntry("AnswerIfFax"); }


inline bool TAMConfig::isTAM()
{ return readBoolEntry("AnswerIfVoice"); }


	

class PageGeneral: public QWidget
{
	Q_OBJECT
public:
	PageGeneral(QWidget *parent = 0, const char *name = 0);
	~PageGeneral();

	void setConfig(TAMConfig &conf);
	void getConfig(TAMConfig &conf);

private:
	QCheckBox *call_accept;
	QLineEdit *tam_name;
	QLineEdit *msn;
	QSpinBox *rings;
};



class PageVoiceOutput: public QWidget
{
	Q_OBJECT
public:
	PageVoiceOutput(QWidget *parent = 0, const char *name = 0);
	~PageVoiceOutput();

	void setConfig(TAMConfig &conf);
	void getConfig(TAMConfig &conf);

private slots:
	void sysInfoToggled(bool on);

private:
	QComboBox *language;
	QCheckBox *time_date;
	QCheckBox *caller_id;
	QCheckBox *sys_info;
	QRadioButton *comprehensive;
	QRadioButton *reduced;
};



class PageAnsweringMachine: public QWidget
{
	Q_OBJECT
public:
	PageAnsweringMachine(QWidget *parent = 0, const char *name = 0);
	~PageAnsweringMachine();

	void setConfig(TAMConfig &conf);
	void getConfig(TAMConfig &conf);

private slots:
	void recordToggled(bool on);

private:
	QCheckBox *record;
	QSpinBox *rec_max;
	QSpinBox *rec_min;
	QCheckBox *enable_speaker;
	QCheckBox *early_accept;
	QRadioButton *low_quality;
	QRadioButton *high_quality;
};



class PageFax: public QWidget
{
	Q_OBJECT
public:
	PageFax(QWidget *parent = 0, const char *name = 0);
	~PageFax();

	void setConfig(TAMConfig &conf);
	void getConfig(TAMConfig &conf);

private:
	QLineEdit *fax_id;
};



class PageSecurity: public QWidget
{
	Q_OBJECT
public:
	PageSecurity(QWidget *parent = 0, const char *name = 0);
	~PageSecurity();

	void setConfig(TAMConfig &conf);
	void getConfig(TAMConfig &conf);

private slots:
	void remoteConfigToggled(bool on);

private:
	QCheckBox *remote_config;
	QLineEdit *password;
	QLineEdit *vip_number;
	QCheckBox *monitor;
	QCheckBox *keyboard;
};



class TAMEditDlg: public KDialogBase
{
	Q_OBJECT
public:
	TAMEditDlg(int model, bool fax, QWidget * parent = 0, const char * name = 0);
	virtual ~TAMEditDlg();

	void setConfig(TAMConfig &conf);
	void getConfig(TAMConfig &conf);

private:
	int modem_model;
	bool isdn_fax;
	QWidget *page1, *page2, *page3, *page4;
};



class TAMSetupDlg: public KDialog
{
	Q_OBJECT
public:
	TAMSetupDlg(int model, QWidget *parent = 0, const char *name = 0);
	~TAMSetupDlg();

	void loadList();

protected:
	void addItem(bool fax);

protected slots:
	virtual void accept();
	virtual void reject();

	void selectionChangedSlot(QListViewItem *item);
	void executedSlot(QListViewItem *);
	void addAMItem();
	void addFaxItem();
	void editItem();
	void deleteItem();
	void slotHelp();

private:
	bool tamsAvailable();

	bool modified;
	int modem_model;
	TAMConfig tam_config;
	QPushButton *add_am, *add_fax, *edit_item, *del_item, *ok;
	QListView *list;
};




#endif // TAMSETUPDLG_H
