/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <qglobal.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qwhatsthis.h>
#include <qfile.h>

#include <kapplication.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kstdaction.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>
#include <kstatusbar.h>
#include <kpopupmenu.h>

#include "kmlocfg.h"
#include "global.h"
#include "configure.h"
#include "mloloader.h"
#include "setupdlg.h"
#include "tamsetupdlg.h"



StatusProgress::StatusProgress(QWidget *parent = 0, const char *name = 0): QFrame(parent, name)
{
	QBoxLayout *box = new QHBoxLayout(this, 0, 0);

	abort = new QPushButton("X", this);
	Q_CHECK_PTR(abort);
	abort->setMinimumSize(20, fontMetrics().height());
	box->addWidget(abort);

	progress = new KProgress(this);
	Q_CHECK_PTR(progress);
	progress->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
	progress->setLineWidth(0);
	progress->setMargin(0);
	progress->setBackgroundMode(QWidget::PaletteBackground);
	progress->setFixedSize(80, fontMetrics().height());
	box->addWidget(progress);

	box->activate();

	connect(abort, SIGNAL(clicked()), SLOT(abortSlot()));

	setFixedSize(sizeHint());
}


StatusProgress::~StatusProgress()
{
}


void StatusProgress::setEnabled(bool enable)
{
	if (enable) {
		abort->show();
		progress->show();
	} else {
		abort->hide();
		progress->hide();
		setProgress(0);
	}
}


void StatusProgress::abortSlot()
{
	emit aborted();
}



KMLOCfg::KMLOCfg(QWidget *parent, const char *name): KMainWindow(parent, name)
{
	config = kapp->config();

	has_config = false;
	has_dev_config = false;
	
	initActions();
	initMenuBar();
	initToolBar();
	initStatusBar();
	
	loader = new MLOLoader(this, "loader");
	Q_CHECK_PTR(loader);

	view = new SetupWidget(MODEL_UNKNOWN, this, "setupwidget");
	Q_CHECK_PTR(view);
	view->setMinimumSize(view->sizeHint());
	setCentralWidget(view);

	fileWrite->setEnabled(false);
	extrasDevConfig->setEnabled(false);
	
	readOptions();

	connect(loader, SIGNAL(totalSteps(int)), progress, SLOT(setTotalSteps(int)));
	connect(loader, SIGNAL(progress(int)), progress, SLOT(setProgress(int)));
	connect(loader, SIGNAL(message(const QString &)), SLOT(slotStatusMsg(const QString &)));
	connect(loader, SIGNAL(done(int, bool)), SLOT(slotFetchDone(int, bool)));
	
	resize(sizeHint());
}


KMLOCfg::~KMLOCfg()
{
}


void KMLOCfg::readOptions()
{
	config->setGroup("MainWindow Options");

	settingsShowToolbar->setChecked(config->readBoolEntry("Show ToolBar", true));
	slotSettingsShowToolbar();
	settingsShowStatusbar->setChecked(config->readBoolEntry("Show StatusBar", true));
	slotSettingsShowStatusbar();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
}


void KMLOCfg::initActions()
{
	fileRead = new KAction(i18n("&Read Configuration"), "download", KShortcut(CTRL+Key_R), this, SLOT(slotFileRead()), actionCollection(), "file_read");
	Q_CHECK_PTR(fileRead);
	fileRead->setWhatsThis(i18n("Click this button to read the configuration from your modem.<br><br>" \
		"You can also select the command <b>Read Configuration</b> from the \"File\" menu."));
	fileWrite = new KAction(i18n("&Write Configuration"), "upload", KShortcut(CTRL+Key_W), this, SLOT(slotFileWrite()), actionCollection(), "file_write");
	Q_CHECK_PTR(fileWrite);
	fileWrite->setWhatsThis(i18n("Click this button to write the configuration to your modem.<br><br>" \
		"You can also select the command <b>Write Configuration</b> from the \"File\" menu."));
	fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
	Q_CHECK_PTR(fileQuit);
	fileQuit->setWhatsThis(i18n("Click this button to quit KMLOCfg.<br><br>" \
		"You can also select the command <b>Quit</b> from the \"File\" menu."));

	extrasDevConfig = new KAction(i18n("Configure &Devices..."), "devcfg", KShortcut(CTRL+Key_D), this, SLOT(slotExtrasDevConfig()), actionCollection(), "extras_dev_config");
	Q_CHECK_PTR(extrasDevConfig);
	extrasDevConfig->setWhatsThis(i18n("Click this button to configure the answering machines and fax devices of your modem.<br><br>" \
		"You can also select the command <b>Configure Devices</b> from the \"Extras\" menu."));

	settingsShowToolbar = KStdAction::showToolbar(this, SLOT(slotSettingsShowToolbar()), actionCollection());
	settingsShowStatusbar = KStdAction::showStatusbar(this, SLOT(slotSettingsShowStatusbar()), actionCollection());
	settingsSaveOptions = new KAction(i18n("Save &Options"), 0, 0, this, SLOT(slotSettingsSaveOptions()), actionCollection(), "settings_save_options");
	Q_CHECK_PTR(settingsSaveOptions);
	settingsConfigure = new KAction(i18n("&Configure KMLOCfg..."), "configure", 0, this, SLOT(slotSettingsConfigure()), actionCollection(), "settings_configure");
	Q_CHECK_PTR(settingsConfigure);
	settingsConfigure->setWhatsThis(i18n("Opens the Configure Dialog.<br><br>" \
		"You can also select the command <b>Configure KMLOCfg</b> from the \"Settings\" menu."));
}


void KMLOCfg::initMenuBar()
{
	QPopupMenu *fileMenu, *extrasMenu, *settingsMenu;

	fileMenu = new QPopupMenu(0, "filemenu");
	Q_CHECK_PTR(fileMenu);
	fileRead->plug(fileMenu);
	fileWrite->plug(fileMenu);
	fileMenu->insertSeparator();
	fileQuit->plug(fileMenu);

	extrasMenu = new QPopupMenu(0, "extrasmenu");
	Q_CHECK_PTR(extrasMenu);
	extrasDevConfig->plug(extrasMenu);

	settingsMenu = new QPopupMenu(0, "settingsmenu");
	Q_CHECK_PTR(settingsMenu);
	settingsShowToolbar->plug(settingsMenu);
	settingsShowStatusbar->plug(settingsMenu);
	settingsMenu->insertSeparator();
	settingsSaveOptions->plug(settingsMenu);
	settingsMenu->insertSeparator();
	settingsConfigure->plug(settingsMenu);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Extras"), extrasMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu());
}


void KMLOCfg::initToolBar()
{
	fileRead->plug(toolBar());
	fileWrite->plug(toolBar());
	toolBar()->insertSeparator();
	extrasDevConfig->plug(toolBar());
	toolBar()->insertSeparator();
	settingsConfigure->plug(toolBar());
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KMLOCfg help pages displayed.<br><br>" \
		"You can also select the command <b>Contents</b> from the \"Help\" menu."));
	toolBar()->alignItemRight(fileQuit->itemId(fileQuit->plug(toolBar())));
}

	
void KMLOCfg::initStatusBar()
{
	progress = new StatusProgress(statusBar());	
	Q_CHECK_PTR(progress);
	
	statusBar()->addWidget(progress, 0, true);
	
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);

	progress->setEnabled(false);
}


void KMLOCfg::slotFileRead()
{
	if (loader->startRead()) {
		progress->setEnabled(true);
		slotStatusMsg(i18n("Reading modem configuration..."));
		fileRead->setEnabled(false);
		fileWrite->setEnabled(false);
		extrasDevConfig->setEnabled(false);
		view->setEnabled(false);
	}
}


void KMLOCfg::slotFileWrite()
{
	loader->setConfig(view->getConfig());
	if (loader->startWrite()) {
		progress->setEnabled(true);
		slotStatusMsg(i18n("Writing modem configuration..."));
		fileRead->setEnabled(false);
		fileWrite->setEnabled(false);
		extrasDevConfig->setEnabled(false);
		view->setEnabled(false);
	}
}


void KMLOCfg::slotFileQuit()
{
	close();
}


void KMLOCfg::slotExtrasDevConfig()
{
	if (loader->startFileRead()) {
		progress->setEnabled(true);
		slotStatusMsg(i18n("Reading device configuration..."));
		fileRead->setEnabled(false);
		fileWrite->setEnabled(false);
		extrasDevConfig->setEnabled(false);
	}
}


void KMLOCfg::slotSettingsShowToolbar()
{
	if (settingsShowToolbar->isChecked())
		toolBar()->show();
	else
		toolBar()->hide();

	setFixedSize(minimumSizeHint());
}


void KMLOCfg::slotSettingsShowStatusbar()
{
	if (settingsShowStatusbar->isChecked())
		statusBar()->show();
	else
		statusBar()->hide();

	setFixedSize(minimumSizeHint());
}


void KMLOCfg::slotSettingsSaveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());
}


void KMLOCfg::slotSettingsConfigure()
{
	Configure configure(this, "configure");
	if (configure.exec())
		view->setToolTips();
}


void KMLOCfg::slotStatusMsg(const QString &text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOCfg::slotStatusHelpMsg(const QString &text)
{
	statusBar()->message(text, 2000);
}


void KMLOCfg::slotFetchDone(int action, bool success)
{
	bool enable = true;

	if (success)
		slotStatusMsg(i18n("Ready."));

	progress->setEnabled(false);
	view->setEnabled(true);

	if ((action == LOADER_ACTION_CFG_READ) && success) {
		const mlo_config_t *c = loader->getConfig();

		has_config = true;
//		has_dev_config = (c->model == MODEL_OFFICE_II) || (c->model == MODEL_ISDN_OFFICE);
		has_dev_config = (c->model == MODEL_ISDN_OFFICE);
	
		if (view->model() != c->model) {
			view->setToolTips(false);
			SetupWidget *nview = new SetupWidget(c->model, this, "setupwidget");
			Q_CHECK_PTR(nview);
			nview->setMinimumSize(nview->sizeHint());
			setCentralWidget(nview);
			delete view;
			view = nview;
			view->show();
		}
		view->setConfig(c);
	}
	
	if ((action == LOADER_ACTION_CFG_WRITE) && success) {
		view->resetTime();
	}

	if ((action == LOADER_ACTION_FILE_READ) && success) {
		TAMSetupDlg setupdlg(view->model(), this, "tamsetupdlg");
		if (setupdlg.exec()) {
			if (loader->startFileWrite()) {
				progress->setEnabled(true);
				slotStatusMsg(i18n("Writing device configuration..."));
				enable = false;
			}
		} else
			QFile::remove(expandPath("config.cfg"));
	}

	if (action == LOADER_ACTION_FILE_WRITE) {
		QFile::remove(expandPath("config.cfg"));
	}
	
	fileRead->setEnabled(enable);
	fileWrite->setEnabled(enable && has_config);
	extrasDevConfig->setEnabled(enable && has_dev_config);
}
