/*************************************************/
/* methods for class DlgEqu                      */
/*                                               */
/* dialog to modify device or circuit equations  */
/* if editing circuit equations, new equation    */
/* can be created with this dialog               */
/*                                               */
/* Andreas Rostin                                */
/* 15.01.99                                      */
/*                                               */
/* replaced the PropEqu dialog using this one:   */
/* 06.10.00                                      */
/*************************************************/
#include <klineedit.h>

#include <qdialog.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qlistbox.h>
#include <qlabel.h>
#include <qmessagebox.h>

#include <klogic.h>
#include <netw.h>
#include <xnet.h>
#include <dlgEqu.h>
#include <dlgPattern.h>

#include "dlgEqu.moc"

// -----------------------------------------
// circuit equation editing / device creator
// -----------------------------------------
DlgEqu::DlgEqu(QWidget *parent, QString _caption, XDeviceNet *_net)
	: QDialog(parent, _caption, TRUE, WStyle_DialogBorder)
{
    net = _net;
    dev = (XDevice *)NULL;

    equations = (list<OutputInfo> *)NULL;
    substituted = 0;
    edit_slot = 0;
    edit_key = (char *)NULL;

    createDialogContent(_caption);
    parseCircuit();
}

// -----------------------------------------
// device equation editing
// -----------------------------------------
DlgEqu::DlgEqu(QWidget *parent, QString _caption, XDevice *_dev)
	: QDialog(parent, _caption, TRUE, WStyle_DialogBorder)
{
    net = (XDeviceNet *)NULL;
    dev = _dev;

    equations = (list<OutputInfo> *)NULL;
    substituted = 0;
    edit_slot = 0;
    edit_key = (char *)NULL;

    createDialogContent(_caption);
    parseCircuit();
}

void DlgEqu::createDialogContent(QString _caption)
{

//------------------------------------------------------------------
// a list box containing the equation list

    list_equ_label = new QLabel((QWidget *)this, (const char *)NULL);
    list_equ_label->setGeometry(10, 10, 100, 20);
    if (net) list_equ_label->setText(klocale->translate("Circuit Equations"));
    if (dev) list_equ_label->setText(klocale->translate("Device Equations"));

    list_equ = new QListBox((QWidget *)this, (const char *)NULL);
    list_equ->setGeometry(10, 30, 590, 240);

//------------------------------------------------------------------
// elements for manipulating a single equation

    edit_output_name = new KLineEdit((QWidget *)this, (const char *)NULL);
    edit_output_name->setGeometry(10, 285, 120, 20);

    edit_equation_label = new QLabel((QWidget *)this, (const char *)NULL);
    edit_equation_label->setGeometry(136, 282, 10, 20);
    edit_equation_label->setText("=");

    edit_equation = new KLineEdit((QWidget *)this, (const char *)NULL);
    edit_equation->setGeometry(150, 285, 450, 20);

    bgType = new QButtonGroup((QWidget *)this);
    bgType->setGeometry(18, 318, 105, 70);
    rbFinal = new QRadioButton(bgType);
    rbFinal->setGeometry(5, 5, 90, 20);
    rbFinal->setText(klocale->translate("final"));
    rbFinal->setChecked(true);
    rbInternal = new QRadioButton(bgType);
    rbInternal->setGeometry(5, 25, 90, 20);
    rbInternal->setText(klocale->translate("internal"));
    rbTemp = new QRadioButton(bgType);
    rbTemp->setGeometry(5, 45, 90, 20);
    rbTemp->setText(klocale->translate("temporary"));

    bEdit = new QPushButton((QWidget *)this);
    bEdit->setGeometry(140, 335, 55, 25);
    bEdit->setText(klocale->translate("Change"));

    bNew = new QPushButton((QWidget *)this);
    bNew->setGeometry(205, 335, 45, 25);
    bNew->setText(klocale->translate("Add"));
    bNew->setEnabled(TRUE);

    bDelete = new QPushButton((QWidget *)this);
    bDelete->setGeometry(260, 335, 55, 25);
    bDelete->setText(klocale->translate("Delete"));
    bDelete->setEnabled(TRUE);

    bgNormalize_s = new QButtonGroup(this);
    bgNormalize_s->setGeometry(325, 315, 110, 75);
    rbSubjunctive_s = new QRadioButton(bgNormalize_s);
    rbSubjunctive_s->setGeometry(5, 5, 100, 20);
    rbSubjunctive_s->setText(klocale->translate("subjunctive"));
    rbSubjunctive_s->setChecked(TRUE);
    rbDisjunctive_s = new QRadioButton(bgNormalize_s);
    rbDisjunctive_s->setGeometry(5, 25, 100, 20);
    rbDisjunctive_s->setText(klocale->translate("disjunctive"));
    bNormalize_s = new QPushButton(bgNormalize_s);
    bNormalize_s->setGeometry(5, 50, 100, 20);
    bNormalize_s->setText(klocale->translate("Normalize"));

    bShowKarnaugh = new QPushButton(this);
    bShowKarnaugh->setGeometry(445, 335, 95, 25);
    bShowKarnaugh->setText(klocale->translate("show Karnaugh"));

    bExtend = new QPushButton((QWidget *)this);
    bExtend->setGeometry(570, 335, 30, 25);
    if (net) {
    	dlg_extended = 0;	// will be switched!
    } else {
    	dlg_extended = 1;	// will be switched!
    }

//------------------------------------------------------------------
// elements for manipulating all equations

    global_label = new QLabel((QWidget *)this, (const char *)NULL);
    global_label->setGeometry(610, 10, 110, 20);
    global_label->setText(klocale->translate("List Methods"));

    bSubst = new QPushButton((QWidget *)this);
    bSubst->setGeometry(610, 30, 110, 20);
    bSubst->setText(klocale->translate("Substitute"));

    bCollect = new QPushButton((QWidget *)this);
    bCollect->setGeometry(610, 66, 110, 20);
    bCollect->setText(klocale->translate("Tidy Up"));
    bCollect->setEnabled(FALSE);

    bgReplace = new QButtonGroup((QWidget *)this);
    bgReplace->setGeometry(610, 100, 110, 80);
    edit_replace = new KLineEdit((QWidget *)bgReplace, (const char *)NULL);
    edit_replace->setGeometry(5, 5, 100, 20);
    edit_replace_with = new KLineEdit((QWidget *)bgReplace, (const char *)NULL);
    edit_replace_with->setGeometry(5, 30, 100, 20);
    bReplace = new QPushButton((QWidget *)bgReplace, (const char *)NULL);
    bReplace->setGeometry(5, 55, 100, 20);
    bReplace->setText("replace - with");

    bgNormalize = new QButtonGroup(this);
    bgNormalize->setGeometry(610, 195, 110, 75);
    rbSubjunctive = new QRadioButton(bgNormalize);
    rbSubjunctive->setGeometry(5, 5, 100, 20);
    rbSubjunctive->setText(klocale->translate("subjunctive"));
    rbSubjunctive->setChecked(TRUE);
    rbDisjunctive = new QRadioButton(bgNormalize);
    rbDisjunctive->setGeometry(5, 25, 100, 20);
    rbDisjunctive->setText(klocale->translate("disjunctive"));
    bNormalize = new QPushButton(bgNormalize);
    bNormalize->setGeometry(5, 50, 100, 20);
    bNormalize->setText(klocale->translate("Normalize"));
    bNormalize->setEnabled(FALSE);

    bRestore = new QPushButton(this);
    bRestore->setGeometry(610, 285, 110, 20);
    bRestore->setText(klocale->translate("Restore All"));

//------------------------------------------------------------------
// common elements

    sep1 = new QButtonGroup(this);
    sep1->setGeometry(10, 394, 710, 2);

    bOK = new QPushButton(this);
    bOK->setGeometry(260, 402, 120, 25);
    if (net)
    	bOK->setText(klocale->translate("Create Sub Circuit"));
    if (dev)
    	bOK->setText(klocale->translate("OK"));

    bCancel = new QPushButton(this);
    bCancel->setGeometry(400, 402, 60, 25);
    bCancel->setText(klocale->translate("Cancel"));

    sep2 = new QButtonGroup(this);
    sep2->setGeometry(10, 433, 710, 2);

    setCaption(_caption);
    extendDialog();	// depends on wether net or dev is set!

//------------------------------------------------------------------
// signal processing

    connect(list_equ, SIGNAL(highlighted(int)), SLOT(editEquation(int)));
    connect(bEdit, SIGNAL(clicked()), SLOT(editEquationOK()));
    connect(bNew, SIGNAL(clicked()), SLOT(newEquation()));
    connect(bDelete, SIGNAL(clicked()), SLOT(deleteEquation()));
    connect(bNormalize_s, SIGNAL(clicked()), SLOT(normalizeEquation()));
    connect(bRestore, SIGNAL(clicked()), SLOT(parseCircuit()));
    connect(bSubst, SIGNAL(clicked()), SLOT(substituteEquations()));
    connect(bCollect, SIGNAL(clicked()), SLOT(collectEquations()));
    connect(bNormalize, SIGNAL(clicked()), SLOT(normalizeEquations()));
    connect(bReplace, SIGNAL(clicked()), SLOT(replaceString()));
    connect(bExtend, SIGNAL(clicked()), SLOT(extendDialog()));
    connect(bShowKarnaugh, SIGNAL(clicked()), SLOT(showKarnaugh()));

    connect(bOK, SIGNAL(clicked()), SLOT(accept()));
    connect(bCancel, SIGNAL(clicked()), SLOT(reject()));

}

DlgEqu::~DlgEqu()
{
	if (equations) freemem();
}

void DlgEqu::done(int r)
{
	if (r == Accepted) {
		list<OutputInfo> *lout;
		opStack oop;
		int i;
		QString input;
		int io_id;
		int res;

		if (net) {
			// create a new subcircuit
			// substitite if possible
			while(0 < (res = substituteEquations()));
			if (res == -1) {
				res = QMessageBox::warning(this,
					klocale->translate("Create sub circuit"),
					klocale->translate("Creating devices without substitution\nmay result in unuseable devices\n\ncontinue anyway?"),
					QMessageBox::Yes, QMessageBox::No);
					if (res == QMessageBox::No) QDialog::done(r);
			}
			collectEquations();

			// create the equation device
			dev = net->newDevice(Device::fEQU, 20, 20);

			if (NetWidget::getSimMode() == NetWidget::MODE_SIM_MULT)
				NetWidget::simTimer.stop();

			// create all outputs and set the equations
			lout = equations->First();
			while(lout) {
				if (lout->Get()->output_type == Device::INTERNAL_OUTPUT) {
					dev->setSize(dev->neededSize() + 1);
					io_id = dev->addInternalName(lout->Get()->output_name);
					dev->setEquation(lout->Get()->equation, io_id);
					dev->setSize(dev->neededSize());
				}
				if (lout->Get()->output_type == Device::FINAL_OUTPUT) {
					dev->setSize(dev->neededSize() + 1);
					io_id = dev->addOutputName(lout->Get()->output_name);
					dev->setEquation(lout->Get()->equation, io_id);
					dev->setSize(dev->neededSize());
				}
				lout = lout->Next();
			}

			// create all input vars
			lout = equations->First();
			while(lout) {
				// create a list of all available symbols
				oop.setEquation(lout->Get()->equation);
				oop.parse();

				i = 0;
				while(NULL != (input = oop.getSymbol(i++))) {
					// if the current symbol is not an output, create a new input
					if (!equations->With(input) && dev->inputNameIsUnique(input)) {
						// names are currently unique, so look for an existing input with this name
						dev->setSize(dev->neededSize() + 1);
						dev->addInputName(input);
						dev->setSize(dev->neededSize());
					}
				}
				lout = lout->Next();
			}
		} else if (dev) {
			// change output names and equations of the given device
			lout = equations->First();
			int output_id;
			while(lout) {
				output_id = lout->Get()->output_id;
				if (lout->Get()->outputChanged()) {
					// if the output does not exist, it'll be created
					output_id = dev->changeOutputName(output_id, lout->Get()->output_name);
				}
				dev->setEquation(lout->Get()->equation, output_id);
				dev->setOutputNameType(output_id, lout->Get()->output_type);
				lout = lout->Next();
			}

			// create a list with new, and a list with all input vars
			lout = equations->First();
			list<QString> all_inputs;
			list<QString> new_inputs;
			while(lout) {
				// create a list of all available symbols
				oop.setEquation(lout->Get()->equation);
				oop.parse();

				i = 0;
				while(NULL != (input = oop.getSymbol(i++))) {
					// input is used as a dummy pointer within the lists!!
					if (!equations->With(input) && dev->inputNameIsUnique(input))
						new_inputs.Append((QString *)&input)->setText((const char *)input);
					all_inputs.Append((QString *)&input)->setText((const char *)input);
				}
				lout = lout->Next();
			}

			// remove input vars not needed anymore
			// (replace name if possible)
			list<value> *named_input = dev->getNamedIRef();
			list<QString> *ls = new_inputs.First();
			while(named_input) {
				if (NULL == all_inputs.With(named_input->getText())) {
					if (ls) {
						// replace the old with the new input
						dev->changeInputName(named_input->getID1(), ls->getText());
						ls = ls->Next();
					} else {
						// no new input vars left: remove the old input
						dev->removeInputName(named_input->getID1());
					}
				}
				named_input = named_input->Next();
			}

			// add the new input vars left
			while(ls) {
				dev->addInputName(ls->getText());
				ls = ls->Next();
			}
		}

		// prepare device operation
		dev->parseEquation();

		// restart the simulation
		if (NetWidget::getSimMode() == NetWidget::MODE_SIM_MULT)
			NetWidget::setSimMode(NetWidget::MODE_SIM_MULT);
	}
	QDialog::done(r);
}

void DlgEqu::extendDialog()
{
	if (dlg_extended) {
		setFixedSize(610, 440);
		dlg_extended = 0;
		bExtend->setText(">>");
		sep1->setGeometry(10, 394, 590, 2);
		bOK->setGeometry(200, 402, 120, 25);
		bCancel->setGeometry(340, 402, 60, 25);
		sep2->setGeometry(10, 433, 590, 2);
	} else {
		setFixedSize(730, 440);
		dlg_extended = 1;
		bExtend->setText("<<");
		sep1->setGeometry(10, 394, 710, 2);
		bOK->setGeometry(260, 402, 120, 25);
		bCancel->setGeometry(400, 402, 60, 25);
		sep2->setGeometry(10, 433, 710, 2);
	}
}

// free memory allocated by equation parsing (in class Device)
void DlgEqu::freemem()
{	list<OutputInfo> *ls;

	list_equ->clear();
	ls = equations->First();
	while(ls) {
		delete ls->Get();
		ls = ls->Next();
	}
	equations->Destroy();
	delete equations;
	equations = (list<OutputInfo> *)NULL;
}

// list box item highlighted
void DlgEqu::editEquation(int idx)
{	const char *buf;
	char *search;
	char repl;
	static char cbuf[1001];

	// get the output name for which equation should be edited
	edit_slot = idx;
	buf = list_equ->text(idx);
	if (!buf) {
		edit_slot = 0;
		if (edit_key) free(edit_key);
		edit_key = (char *)NULL;
		return;
	}

	// remove pre- and suffixes from the output name
	search = strstr(buf, "(intern)");
	if (!search) {
		search = strstr(buf, "(final)");
		if (!search) {
			search = strstr(buf, " = ");
		}
	}
	if (!search) {
		clearSelection();
		return;
	}
	repl = *search;
	*search = 0;
	edit_key = strdup(buf);
	*search = repl;

	// list box items may be truncated, so use original qstring
	list<OutputInfo> *ls = equations->With(edit_key);
	if (ls && ls->Get()) {
		// display equation in the edit-box
		edit_equation->setText(ls->Get()->equation);

		// display new edit box label
		sprintf(cbuf, "%s", (const char *)ls->Get()->output_name);
		if (ls->Get()->output_type == Device::INTERNAL_OUTPUT) {
			rbInternal->setChecked(TRUE);
		} else if (ls->Get()->output_type == Device::FINAL_OUTPUT) {
			rbFinal->setChecked(TRUE);
		} else {
			rbTemp->setChecked(TRUE);
		}
		edit_output_name->setText(cbuf);
	} else {
		clearSelection();
		return;
	}
}

// equation editing finished (change button pressed)
void DlgEqu::editEquationOK()
{	char *newTypeName;
	int newType;
	char old_edit_slot;

	if (edit_key) {
		old_edit_slot = edit_slot;
		list<OutputInfo> *ls = equations->With(edit_key);
		if (ls) {
			// set new output type
			if (rbInternal->isChecked()) {
				newTypeName = "(intern)";
				newType = Device::INTERNAL_OUTPUT;
			} else if (rbFinal->isChecked()) {
				newTypeName = "(final)";
				newType = Device::FINAL_OUTPUT;
			} else {
				newTypeName = "";
				newType = Device::TMP_OUTPUT;
			}

			// change output name
			QString new_name = edit_output_name->text();
			if (ls->Get()->output_name != new_name) {
				if (changeOutputName(ls->Get()->output_name, new_name)) {
					if (new_name.contains("(final)") || new_name.contains("(intern)")) {
						QMessageBox::warning((QWidget *)this, "Reserved content", "Output names cannot contain\n(final) or (intern)");
						return;
					}

					ls->Get()->output_name = new_name;
					ls->setText(new_name);
					fillList();
					return;
				} else {
					QMessageBox::information(this, klocale->translate("Change output name"),
						klocale->translate("This output name already exists ..."));
						return;
				}
			}

			ls->setID2(newType);
			ls->Get()->output_type = newType;

			// set new equation
			ls->Get()->equation = edit_equation->text();

			QString s = ls->Get()->output_name;
			s += newTypeName;
			s += " = ";
			s += ls->Get()->equation;
			list_equ->changeItem(s, edit_slot);	// this causes a signal which calls editEquation() .. the key will be lost!
			edit_key = strdup((const char *)ls->Get()->output_name);
			edit_slot = old_edit_slot;
		}
	}
}

// normalize the currently selected equations
void DlgEqu::normalizeEquation()
{
	if (!edit_equation->text().length()) return;

	opStack opp;
	// build result table
	opp.setEquation(edit_equation->text());
	opp.parse();
	if (rbSubjunctive->isChecked()) normalize_status = 1;
	else normalize_status = 0;
	edit_equation->setText(opp.calculateSymbolicNormalized(normalize_status, 0));
}

void DlgEqu::showKarnaugh()
{
	if (!edit_equation->text().length()) return;

	DlgPattern *dlg = new DlgPattern(this, "Karnaugh Diagram", edit_equation->text());
	dlg->show();
	delete dlg;
}

// change an existing output name
int DlgEqu::changeOutputName(QString old_name, QString new_name)
{	int start, len;

	list<OutputInfo> *ls = equations->First();
	while(ls) {
		if (ls->Get()->output_name == new_name) return 0;
		ls = ls->Next();
	}

	len = old_name.length();
	ls = equations->First();
	while(ls) {
		while (-1 != (start = ls->Get()->equation.find(old_name)))
			ls->Get()->equation.replace(start, len, new_name);
		ls = ls->Next();
	}
	return 1;
}

// replace button pressed: replace string in all equations
void DlgEqu::replaceString()
{	QString s_old, s_new;
	int start, len, new_len;
	int found = 0;

	s_old = edit_replace->text();
	s_new = edit_replace_with->text();

	len = s_old.length();
	new_len = s_new.length();
	if (!len) return;

	list<OutputInfo> *ls = equations->First();
	while(ls) {
		start = 0;
		while (-1 != (start = ls->Get()->equation.find(s_old, start))) {
			ls->Get()->equation.replace(start, len, s_new);
			start += new_len;
			found = 1;
		}
		start = 0;
		while (-1 != (start = ls->Get()->output_name.find(s_old, start))) {
			ls->Get()->output_name.replace(start, len, s_new);
			start += new_len;
			found = 1;
		}
		start = 0;
		while (-1 != (start = ls->Get()->prefix.find(s_old, start))) {
			ls->Get()->prefix.replace(start, len, s_new);
			start += new_len;
			found = 1;
		}

		int ek_found = 0;
		QString s;
		if (edit_key) {
			s = edit_key;
			start = 0;
			while (-1 != (start = s.find(s_old, start))) {
				s.replace(start, len, s_new);
				start += new_len;
				ek_found = 1;
			}
			if (ek_found) {
				free(edit_key);
				edit_key = strdup((char *)(const char *)s);
				found = 1;
			}
		}
		ek_found = 0;
		s = ls->getText();
		start = 0;
		while (-1 != (start = s.find(s_old, start))) {
			s.replace(start, len, s_new);
			start += new_len;
			ek_found = 1;
		}
		if (ek_found) ls->setText(s);

		ls = ls->Next();
	}
	if (found) fillList();
}

void DlgEqu::clearSelection()
{
	list_equ->clearSelection();
	edit_slot = 0;
	if (edit_key) free(edit_key);
	edit_key = (char *)NULL;
	edit_equation->setText("");
	edit_output_name->setText("");
}

void DlgEqu::editEquationCancel()
{
	if (list_equ->currentItem() == -1) {
		clearSelection();
	} else editEquation(list_equ->currentItem());
}

void DlgEqu::newEquation()
{
	if (strlen(edit_equation->text()) && strlen(edit_output_name->text())) {
		// the new name must be unique!
		if (equations->With(edit_output_name->text())) {
			QMessageBox::warning((QWidget *)this, "Output name already exists", "Output names of equation devices must be unique");
			return;
		}
		QString s(edit_output_name->text());
		if (s.contains("(final)") || s.contains("(intern)")) {
			QMessageBox::warning((QWidget *)this, "Reserved content", "Output names cannot contain\n(final) or (intern)");
			return;
		}

		// retrieve the equation string
		OutputInfo *oi = new OutputInfo();
		oi->equation = edit_equation->text();
		oi->output_name = edit_output_name->text();

		// set output type
		if (rbInternal->isChecked()) {
			oi->output_type = Device::INTERNAL_OUTPUT;
		} else if (rbFinal->isChecked()) {
			oi->output_type = Device::FINAL_OUTPUT;
		} else {
			oi->output_type = Device::TMP_OUTPUT;
		}

		// append the new "output<-equation" to the list
		list<OutputInfo> *ls = equations->Append(oi);
		ls->setText(oi->output_name);
		ls->setID2(oi->output_type);

		// update the displayed list
		fillList();

		// display new item highlighed
		int idx = list_equ->count() - 1;
		list_equ->setSelected(idx, TRUE);
		list_equ->centerCurrentItem();
		editEquation(idx);
	}
}

void DlgEqu::deleteEquation()
{
	if (edit_key) {
		list<OutputInfo> *ls = equations->With(edit_key);
		if (ls) {
			ls->Destroy(ls->Get());
			list_equ->removeItem(edit_slot);
			fillList();
			editEquationCancel();
		}
	}
}

// retrieve the list of equations; either from the XDeviceNet or from the XDevice
void DlgEqu::parseCircuit()
{
	if (equations) freemem();

	if (net) {
		equations = net->parseCircuit();
	}

	if (dev) {
		equations = new list<OutputInfo>;
		dev->getAllEquations(equations, (char *)NULL, Device::NO_QUALS);
	}

	fillList();

	bSubst->setEnabled(TRUE);
	bNormalize->setEnabled(FALSE);
	bCollect->setEnabled(FALSE);
	editEquationCancel();
}

// normalize all equations
void DlgEqu::normalizeEquations()
{	list<OutputInfo> *ls = equations->First();
	opStack opp;

	// build result table
	while(ls) {
		opp.setEquation(ls->Get()->equation);
		opp.parse();
		if (rbSubjunctive->isChecked()) normalize_status = 1;
		else normalize_status = 0;
		ls->Get()->equation = opp.calculateSymbolicNormalized(normalize_status, 0);
		ls = ls->Next();
	}
	fillList();

	editEquationCancel();
}

// leave only internal and final equations left in the list
void DlgEqu::collectEquations()
{	list<OutputInfo> *ls1 = equations->First();

	while(ls1) {
		if (ls1->Get()->output_type != Device::INTERNAL_OUTPUT && ls1->Get()->output_type != Device::FINAL_OUTPUT) {
			OutputInfo *oi = ls1->Get();
			ls1->Destroy(oi);
			delete oi;
			ls1 = equations->First();
		} else ls1 = ls1->Next();
	}

	fillList();

	bSubst->setEnabled(FALSE);
	bNormalize->setEnabled(FALSE);
	bCollect->setEnabled(FALSE);
	editEquationCancel();

}

int DlgEqu::substituteEquations()
{	list<OutputInfo> *ls1;				// all equations
	opStack nt;					// equation-parser
	opStack opt;					// equation-optimizer
	list<OutputInfo> *ls2;				// equation to substitute with
	list<QString> substituted;			// list of substituted non-terminal
	QString dummy;					// dummy entry for substituted - list
	int found;					// device equation changed
	int changed;					// circuit equations changed
	int i, j;
	int iplane;
	int plane = Device::TMP_OUTPUT;

	// security question
	if (equations->counter() > 50) {
		if (QMessageBox::Yes != QMessageBox::warning(this, klocale->translate("Substitute Equations"),
			klocale->translate("This may take a long time ...\nTry a lower circuit level!\ncontinue anyway"),
			QMessageBox::Yes, QMessageBox::No)) return -1;
	}

	changed = 0;
	for (iplane = 0; iplane < 2; iplane++) {
		if (iplane == 0) plane = Device::TMP_OUTPUT;
		if (iplane == 1) plane = Device::FINAL_OUTPUT;
		ls1 = equations->First();
		while(ls1) {
			if (plane == ls1->Get()->output_type) {
				// initialize list of already substituted non-terminal
				substituted.Destroy();

				do {
					found = 0;

					// create a calculation queue with all names of the current equation in it
					nt.setEquation(ls1->Get()->equation);
					nt.parse();

					// nterns contains all names ...
					char **nterms = nt.getQueue();
					int cnt = nt.getCounter();

					// nterm_info contains the position of a name within its equation
					StackInfo **nterm_info = nt.getInfo();

					// search equation for each name, then substitute it
					// (the name could also be an operator: ignored)
					for (i = 0; i < cnt; i++) {
						// substitute non-terminals with their equations
						if (NULL != (ls2 = equations->With(nterms[i]))) {
							// already marked as internal output?
							if (ls2->Get()->output_type == Device::INTERNAL_OUTPUT) {
								nterm_info[i]->type = Device::INTERNAL_OUTPUT;
							}

							// ls1 and ls2 are the same equation?
							// already subsituted? there is a loop, mark as internal output
							if (nterm_info[i]->type != Device::INTERNAL_OUTPUT && (ls1 == ls2 || substituted.With(nterms[i]))) {
								nterm_info[i]->type = Device::INTERNAL_OUTPUT;
								ls2->Get()->output_type = Device::INTERNAL_OUTPUT;
							}

							// substitute..
							if (nterm_info[i]->type != Device::INTERNAL_OUTPUT) {
								QString s = ls1->Get()->equation.left(nterm_info[i]->position);
								s += "(";
								s += ls2->Get()->equation;
								s += ")";
								s += ls1->Get()->equation.right(ls1->Get()->equation.length() - nterm_info[i]->position - nterm_info[i]->length);

								// correct the positions..
								for (j=0; j < cnt; j++)
									if (nterm_info[j]->position > nterm_info[i]->position)
										nterm_info[j]->position += s.length() - ls1->Get()->equation.length();

								// replace the equation with the substituted one
								ls1->Get()->equation = s;

								// substitute again
								changed = 1;
								found = 1;
							}
						}
					}

					// remember all substituted non-terminals
					for(i = 0; i < cnt; i++) {
						if (!substituted.With(nterms[i]))
							substituted.Append(&dummy)->setText(nterms[i]);
					}
				} while(found);

				// ls1 is substituted now. optimize the newly generated equation
				opt.setEquation(ls1->Get()->equation);
				opt.parse();
				QString s = opt.calculateSymbolic();
				ls1->Get()->equation = s;
			} // if (plane)

			// retrieve the next equation
			ls1 = ls1->Next();
		} // while equations left
	} // next plane

	fillList();

	if (!changed) {
		bNormalize->setEnabled(TRUE);
		bCollect->setEnabled(TRUE);
	}
	editEquationCancel();
	return changed;
}

void DlgEqu::fillList()
{	list<OutputInfo> *ls = equations->First();

	list_equ->clear();	// the list box
	QString s;
	while(ls) {
		s = ls->Get()->output_name;
		if (ls->Get()->output_type == Device::INTERNAL_OUTPUT) s += "(intern)";
		else if (ls->Get()->output_type == Device::FINAL_OUTPUT) s += "(final)";
		s += " = ";
		s += ls->Get()->equation;
		if (s.length() > 4096) {
			s.truncate(4096);
			s += " (...)";
		}
		list_equ->insertItem(s);
		ls = ls->Next();
	}
}

