/*
 * KMLOFax
 *
 * A utility to process facsimiles received with the ELSA
 * MicroLink(tm) Office or MicroLink(tm) ISDN Office modem.
 *
 * Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen.
 */

#include <termios.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qstring.h>

#include <kapp.h>
#include <kiconloader.h>
#include <klocale.h>

#include "configure.h"
#include "global.h"



Configure *configure = 0;


Configure::Configure(QWidget *parent, const char *name) : KDialogBase (IconList, i18n("Configure"), Help|Default|Ok|Apply|Cancel, Ok, parent, name, false, true)
{
	QGridLayout *grid;
	QBoxLayout *vbox, *svbox;
	QLabel *label;
	QGroupBox *gb;
	QFrame *page;

	config = kapp->config();

	gotToolTips = false;

	setHelp("CONFIGURE-HELP");
	setIconListAllVisible(true);
	
	/* Page 1: General */
	page = addPage(i18n("General"), i18n("General Settings"), UserIcon("general"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("User Interface"), page);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	gen_tooltips = new QCheckBox(i18n("Show tooltips"), gb);
	gen_tooltips->setMinimumSize(gen_tooltips->sizeHint());
	svbox->addWidget(gen_tooltips);
	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Download"), page);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	gen_loadnew = new QCheckBox(i18n("Download new facsimiles only"), gb);
	gen_loadnew->setMinimumSize(gen_loadnew->sizeHint());
	svbox->addWidget(gen_loadnew);
	gen_deletefax = new QCheckBox(i18n("Delete facsimiles from modem"), gb);
	gen_deletefax->setMinimumSize(gen_deletefax->sizeHint());
	svbox->addWidget(gen_deletefax);
	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();

	/* Page 2: Modem */
	page = addPage(i18n("Modem"), i18n("Settings for the Modem"), UserIcon("modem"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("Modem Device"), page);

	grid = new QGridLayout(gb, 3, 3, 12);

	grid->addRowSpacing(0, fontMetrics().lineSpacing());

	label = new QLabel(i18n("Device:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mod_device = new QLineEdit(gb);
	mod_device->setMinimumSize(mod_device->sizeHint());
	grid->addMultiCellWidget(mod_device, 1, 1, 1, 2);

	label = new QLabel(i18n("Speed:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mod_speed = new QComboBox(false, gb);
	mod_speed->insertItem(" 19200 Bps ", 0);
	mod_speed->insertItem(" 38400 Bps ", 1);
#ifdef B57600
	mod_speed->insertItem(" 57600 Bps ", 2);
#endif
#ifdef B115200
	mod_speed->insertItem(" 115200 Bps ", 3);
#endif
#ifdef B230400
	mod_speed->insertItem(" 230400 Bps ", 4);
#endif
	mod_speed->setMinimumSize(mod_speed->sizeHint());
	grid->addWidget(mod_speed, 2, 1);

	grid->setColStretch(2, 1);

	grid->activate();

	vbox->addWidget(gb);
	vbox->addStretch(1);
	vbox->activate();


	/* Page 3: Print */
	page = addPage(i18n("Print"), i18n("Settings for Printing"), UserIcon("print"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("Paper Margins"), page);
	grid = new QGridLayout(gb, 9, 2, 10);

	grid->addRowSpacing(0, 8);
	label = new QLabel(i18n("Left (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);
	prt_lmargin = new QSpinBox(gb);
	prt_lmargin->setMinimumSize(prt_lmargin->sizeHint());
	grid->addWidget(prt_lmargin, 1, 1);

	label = new QLabel(i18n("Right (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 3, 0);
	prt_rmargin = new QSpinBox(gb);
	prt_rmargin->setMinimumSize(prt_rmargin->sizeHint());
	grid->addWidget(prt_rmargin, 3, 1);

	label = new QLabel(i18n("Top (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 5, 0);
	prt_tmargin = new QSpinBox(gb);
	prt_tmargin->setMinimumSize(prt_tmargin->sizeHint());
	grid->addWidget(prt_tmargin, 5, 1);

	label = new QLabel(i18n("Bottom (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 7, 0);
	prt_bmargin = new QSpinBox(gb);
	prt_bmargin->setMinimumSize(prt_bmargin->sizeHint());
	grid->addWidget(prt_bmargin, 7, 1);

	grid->setColStretch(1, 1);
	grid->setRowStretch(0, 1);
	grid->setRowStretch(2, 1);
	grid->setRowStretch(4, 1);
	grid->setRowStretch(6, 1);
	grid->setRowStretch(8, 1);

	grid->activate();

	vbox->addWidget(gb);

	vbox->addStretch();

	vbox->activate();

	/* Page 4: Mail */
	page = addPage(i18n("Mail"), i18n("Settings for Sending Mails"), UserIcon("mail"));
	
	vbox = new QVBoxLayout(page, 12);

	gb = new QGroupBox(i18n("SMTP Server"), page);

	grid = new QGridLayout(gb, 3, 3, 12);

	grid->addRowSpacing(0, fontMetrics().lineSpacing());

	label = new QLabel(i18n("Name:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mail_server = new QLineEdit(gb);
	mail_server->setMinimumSize(mail_server->sizeHint());
	grid->addMultiCellWidget(mail_server, 1, 1, 1, 2);

	label = new QLabel(i18n("Port:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mail_port = new QLineEdit(gb);
	mail_port->setFixedWidth(50);
	mail_port->setMinimumHeight(mail_port->sizeHint().height());
	grid->addWidget(mail_port, 2, 1);

	grid->setColStretch(2, 1);

	grid->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Identity"), page);

	grid = new QGridLayout(gb, 5, 2, 12);

	grid->addRowSpacing(0, fontMetrics().lineSpacing());

	label = new QLabel(i18n("Name:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mail_name = new QLineEdit(gb);
	mail_name->setMinimumSize(mail_name->sizeHint());
	grid->addWidget(mail_name, 1, 1);

	label = new QLabel(i18n("Organization:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mail_org = new QLineEdit(gb);
	mail_org->setMinimumSize(mail_org->sizeHint());
	grid->addWidget(mail_org, 2, 1);

	label = new QLabel(i18n("Email address:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 3, 0);

	mail_email = new QLineEdit(gb);
	mail_email->setMinimumSize(mail_email->sizeHint());
	grid->addWidget(mail_email, 3, 1);

	label = new QLabel(i18n("Reply-to address:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 4, 0);

	mail_reply = new QLineEdit(gb);
	mail_reply->setMinimumSize(mail_reply->sizeHint());
	grid->addWidget(mail_reply, 4, 1);

	grid->setColStretch(1, 1);

	grid->activate();

	vbox->addWidget(gb);
	vbox->addStretch(1);

	vbox->activate();

	reset();
}


Configure::~Configure()
{
}


void Configure::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(gen_tooltips, i18n("Show tooltips in dialogs"));
		QToolTip::add(gen_loadnew, i18n("Download only new facsimiles from modem"));
		QToolTip::add(gen_deletefax, i18n("Delete facsimiles from modem after download"));
		QToolTip::add(mod_device, i18n("Device of the modem"));
		QToolTip::add(mod_speed, i18n("Speed for serial communication"));
		QToolTip::add(prt_lmargin, i18n("Left margin on paper"));
		QToolTip::add(prt_rmargin, i18n("Right margin on paper"));
		QToolTip::add(prt_tmargin, i18n("Top margin on paper"));
		QToolTip::add(prt_bmargin, i18n("Bottom margin on paper"));
		QToolTip::add(mail_server, i18n("Name of the SMTP server"));
		QToolTip::add(mail_port, i18n("Port of the SMTP server"));
		QToolTip::add(mail_name, i18n("Your full name"));
		QToolTip::add(mail_org, i18n("Your organization"));
		QToolTip::add(mail_email, i18n("Your email address"));
		QToolTip::add(mail_reply, i18n("The address replies are sent to"));
	}
	else {
		QToolTip::remove(gen_tooltips);
		QToolTip::remove(gen_loadnew);
		QToolTip::remove(gen_deletefax);
		QToolTip::remove(mod_device);
		QToolTip::remove(mod_speed);
		QToolTip::remove(prt_lmargin);
		QToolTip::remove(prt_rmargin);
		QToolTip::remove(prt_tmargin);
		QToolTip::remove(prt_bmargin);
		QToolTip::remove(mail_server);
		QToolTip::remove(mail_port);
		QToolTip::remove(mail_name);
		QToolTip::remove(mail_org);
		QToolTip::remove(mail_email);
		QToolTip::remove(mail_reply);
	}
}	


void Configure::slotDefault()
{
	QString s;
	KConfig mailconf("emaildefaults", true);

	gen_tooltips->setChecked(PREF_GEN_TOOLTIPS);
	gen_loadnew->setChecked(PREF_GEN_LOADNEW);
	gen_deletefax->setChecked(PREF_GEN_DELETEFAX);

	mod_device->setText(PREF_MOD_DEVICE);
	mod_speed->setCurrentItem(PREF_MOD_SPEED);

	prt_lmargin->setValue(PREF_PRT_LMARGIN);
	prt_rmargin->setValue(PREF_PRT_RMARGIN);
	prt_tmargin->setValue(PREF_PRT_TMARGIN);
	prt_bmargin->setValue(PREF_PRT_BMARGIN);

	mailconf.setGroup("ServerInfo");
	mail_server->setText(mailconf.readEntry("Outgoing", PREF_MAIL_SERVER));
	s.setNum(PREF_MAIL_PORT);
	mail_port->setText(s);
	mailconf.setGroup("UserInfo");
	mail_name->setText(mailconf.readEntry("FullName", PREF_MAIL_NAME));
	mail_org->setText(mailconf.readEntry("Organization", PREF_MAIL_ORG));
	mail_email->setText(mailconf.readEntry("EmailAddress", PREF_MAIL_EMAIL));
	mail_reply->setText(mailconf.readEntry("ReplyAddr", PREF_MAIL_REPLY));
}


void Configure::slotOk()
{
	slotApply();
	hide();
}


void Configure::slotApply()
{
	config->setGroup(ID_PREF_GROUP_GENERAL);
	config->writeEntry(ID_PREF_GEN_TOOLTIPS, gen_tooltips->isChecked());
	config->writeEntry(ID_PREF_GEN_LOADNEW, gen_loadnew->isChecked());
	config->writeEntry(ID_PREF_GEN_DELETEFAX, gen_deletefax->isChecked());
	
	config->setGroup(ID_PREF_GROUP_MODEM);
	config->writeEntry(ID_PREF_MOD_DEVICE, mod_device->text());
	config->writeEntry(ID_PREF_MOD_SPEED, mod_speed->currentItem());

	config->setGroup(ID_PREF_GROUP_PRINT);
	config->writeEntry(ID_PREF_PRT_LMARGIN, prt_lmargin->value());
	config->writeEntry(ID_PREF_PRT_RMARGIN, prt_rmargin->value());
	config->writeEntry(ID_PREF_PRT_TMARGIN, prt_tmargin->value());
	config->writeEntry(ID_PREF_PRT_BMARGIN, prt_bmargin->value());

	config->setGroup(ID_PREF_GROUP_MAIL);
	config->writeEntry(ID_PREF_MAIL_SERVER, mail_server->text());
	config->writeEntry(ID_PREF_MAIL_PORT, mail_port->text().toInt());
	config->writeEntry(ID_PREF_MAIL_NAME, mail_name->text());
	config->writeEntry(ID_PREF_MAIL_ORG, mail_org->text());
	config->writeEntry(ID_PREF_MAIL_EMAIL, mail_email->text());
	config->writeEntry(ID_PREF_MAIL_REPLY, mail_reply->text());

	config->sync();

	emit newConfig();
}


void Configure::slotCancel()
{
	reset();
	hide();
}


void Configure::reset()
{
	QString s;
	int speed;
	KConfig mailconf("emaildefaults", true);

	config->setGroup(ID_PREF_GROUP_GENERAL);
	gen_tooltips->setChecked(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));
	gen_loadnew->setChecked(config->readBoolEntry(ID_PREF_GEN_LOADNEW, PREF_GEN_LOADNEW));
	gen_deletefax->setChecked(config->readBoolEntry(ID_PREF_GEN_DELETEFAX, PREF_GEN_DELETEFAX));

	config->setGroup(ID_PREF_GROUP_MODEM);
	mod_device->setText(config->readEntry(ID_PREF_MOD_DEVICE, PREF_MOD_DEVICE));
	speed = config->readNumEntry(ID_PREF_MOD_SPEED, PREF_MOD_SPEED);
	if ((speed < 0) || (speed > 4))
		speed = PREF_MOD_SPEED;
	mod_speed->setCurrentItem(speed);

	config->setGroup(ID_PREF_GROUP_PRINT);
	prt_lmargin->setValue(config->readNumEntry(ID_PREF_PRT_LMARGIN, PREF_PRT_LMARGIN));
	prt_rmargin->setValue(config->readNumEntry(ID_PREF_PRT_RMARGIN, PREF_PRT_RMARGIN));
	prt_tmargin->setValue(config->readNumEntry(ID_PREF_PRT_TMARGIN, PREF_PRT_TMARGIN));
	prt_bmargin->setValue(config->readNumEntry(ID_PREF_PRT_BMARGIN, PREF_PRT_BMARGIN));

	config->setGroup(ID_PREF_GROUP_MAIL);
	mailconf.setGroup("ServerInfo");
	mail_server->setText(config->readEntry(ID_PREF_MAIL_SERVER, mailconf.readEntry("Outgoing", PREF_MAIL_SERVER)));
	s.setNum(config->readNumEntry(ID_PREF_MAIL_PORT, PREF_MAIL_PORT));
	mail_port->setText(s);
	mailconf.setGroup("UserInfo");
	mail_name->setText(config->readEntry(ID_PREF_MAIL_NAME, mailconf.readEntry("FullName", PREF_MAIL_NAME)));
	mail_org->setText(config->readEntry(ID_PREF_MAIL_ORG, mailconf.readEntry("Organization", PREF_MAIL_ORG)));
	mail_email->setText(config->readEntry(ID_PREF_MAIL_EMAIL, mailconf.readEntry("EmailAddress", PREF_MAIL_EMAIL)));
	mail_reply->setText(config->readEntry(ID_PREF_MAIL_REPLY, mailconf.readEntry("ReplyAddr", PREF_MAIL_REPLY)));
}
