/*
 * KMLOCfg
 *
 * A utility to configure modems of the ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifndef MODEM_H
#define MODEM_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <termios.h>

#include <qobject.h>
#include <qstring.h>
#include <qtimer.h>
#include <qsocknot.h>
#include <qfile.h>




#ifdef __linux__
#	define MODEM_SER_DEVNUM	5
#	define MODEM_USB_DEVNUM	4
#elif defined(__FreeBSD__)
#	define MODEM_SER_DEVNUM	4
#	define MODEM_USB_DEVNUM	2
#elif defined(__NetBSD__)
#	define MODEM_SER_DEVNUM	8
#	define MODEM_USB_DEVNUM	2
#elif defined(__svr4__)
#	define MODEM_SER_DEVNUM	4
#	define MODEM_USB_DEVNUM	2
#else
#	define MODEM_SER_DEVNUM	2
#	define MODEM_USB_DEVNUM	2
#endif


extern const char *modem_ser_devices[MODEM_SER_DEVNUM];
extern const char *modem_usb_devices[MODEM_USB_DEVNUM];



class Modem: public QObject
{
	Q_OBJECT
public:
	Modem(QObject *parent = 0, const char *name = 0);
	virtual ~Modem();

	void setDevice(bool usb, int device);
	void setSpeed(int speed);
	void setLineParams(int data, char parity, int stop);

	bool open();
	void close();

	void flush();

	bool lockDevice();
	void unlockDevice();

	bool dsrOn();
	bool ctsOn();

	void writeChar(const char c);
	void writeChar(const char c, int msec);
	void writeLine(const QCString &line);
	void writeLine(const QCString &line, int msec);
	void writeBlock(const char *block, uint length);
	void writeBlock(const char *block, uint length, int msec);

	void timerStart(int msec);
	void timerStop();

	void receiveXModem(const QString &fname, int size, bool crc);
	void sendXModem(const QString &fname);
	void abortXModem();


private slots:
	void timerDone();

	void readChar(int);
	void readXChar(int);

private:
	void init();
	void xinit();

	uchar calcChecksum();
	ushort calcCRC();

	bool is_locked;
	struct termios init_tty;

	speed_t m_speed;
	int m_data;
	char m_parity;
	int m_stop;

	const char *fdev;
	int fd;
	QTimer *timer;
	QSocketNotifier *sn;

	uchar buffer[1024];
	int bufpos;

	int xstate;
	QFile xfile;
	int xfile_size;
	bool xcrc;
	uchar xblock_no;
	int xblock_size;

signals:
	void gotLine(const QCString &);
	void xmodemStatus(int);
	void xmodemDone(bool);
	void timeout();
};


inline void Modem::timerStart(int msec)
{ timer->start(msec, true); }


inline void Modem::timerStop()
{ timer->stop(); }



#endif // MODEM_H
