/*
   SPDX-FileCopyrightText: 2017-2024 Laurent Montel <montel@kde.org>

   SPDX-License-Identifier: LGPL-2.0-or-later
*/

#include "message.h"
#include "ruqola_debug.h"
#include <KLocalizedString>
#include <QCborValue>
#include <QDateTime>
#include <QJsonArray>
#include <QJsonDocument>
using namespace Qt::Literals::StringLiterals;
Message::Message()
{
}

Message::~Message() = default;

void Message::parseMessage(const QJsonObject &o, bool restApi, EmojiManager *emojiManager)
{
    const QByteArray roomId = o.value("rid"_L1).toString().toLatin1();

    // t ? I can't find it.
    const QString type = o.value("t"_L1).toString();

    mMessageId = o.value("_id"_L1).toString().toLatin1();
    mRoomId = roomId;
    mText = o.value("msg"_L1).toString();
    if (restApi) {
        mUpdatedAt = Utils::parseIsoDate(QStringLiteral("_updatedAt"), o);
        setEditedAt(Utils::parseIsoDate(QStringLiteral("editedAt"), o));
        setTimeStamp(Utils::parseIsoDate(QStringLiteral("ts"), o));
        if (o.contains(QStringLiteral("tlm"))) {
            setThreadLastMessage(Utils::parseIsoDate(QStringLiteral("tlm"), o));
        }
        if (o.contains(QStringLiteral("dlm"))) {
            setDiscussionLastMessage(Utils::parseIsoDate(QStringLiteral("dlm"), o));
        }
    } else {
        setTimeStamp(Utils::parseDate(QStringLiteral("ts"), o));
        mUpdatedAt = Utils::parseDate(QStringLiteral("_updatedAt"), o);
        setEditedAt(Utils::parseDate(QStringLiteral("editedAt"), o));
        // Verify if a day we will use not restapi for it.
        if (o.contains(QStringLiteral("tlm"))) {
            setThreadLastMessage(Utils::parseDate(QStringLiteral("tlm"), o));
        }
        // Verify if a day we will use not restapi for it.
        if (o.contains(QStringLiteral("dlm"))) {
            setDiscussionLastMessage(Utils::parseDate(QStringLiteral("dlm"), o));
        }
    }

    QList<QByteArray> lst;
    const QJsonArray replieArray = o.value("replies"_L1).toArray();
    const auto nbReplieArrayCount{replieArray.count()};
    lst.reserve(nbReplieArrayCount);
    for (auto i = 0; i < nbReplieArrayCount; ++i) {
        lst.append(replieArray.at(i).toVariant().toString().toLatin1());
    }
    mReplies = lst;

    const auto userObject = o.value("u"_L1).toObject();
    mUsername = userObject.value("username"_L1).toString();
    mName = userObject.value("name"_L1).toString();
    mUserId = userObject.value("_id"_L1).toString().toLatin1();
    mEditedByUsername = o.value("editedBy"_L1).toObject().value("username"_L1).toString();
    mAlias = o.value("alias"_L1).toString();
    mAvatar = o.value("avatar"_L1).toString();
    assignMessageStateValue(Groupable, o.value("groupable"_L1).toBool(/*true*/ false)); // Laurent, disable for the moment groupable
    assignMessageStateValue(ParsedUrl, o.value("parseUrls"_L1).toBool());
    mRole = o.value("role"_L1).toString();
    if (o.contains("tcount"_L1)) {
        setThreadCount(o.value("tcount"_L1).toInt());
    }
    if (o.contains("tcount"_L1)) {
        setDiscussionCount(o.value("tcount"_L1).toInt());
    }
    if (o.contains("drid"_L1)) {
        setDiscussionRoomId(o.value("drid"_L1).toString().toLatin1());
    }
    if (o.contains("tmid"_L1)) {
        setThreadMessageId(o.value("tmid"_L1).toString().toLatin1());
    }
    mEmoji = o.value("emoji"_L1).toString();
    mMessageStarred.parse(o);

    MessagePinned pinned;
    pinned.parse(o);
    if (pinned.isValid()) {
        setMessagePinned(pinned);
    }

    // Translation
    MessageTranslation translation;
    translation.parse(o);
    if (!translation.isEmpty()) {
        setMessageTranslation(translation);
    }
    assignMessageStateValue(Private, o.value("private"_L1).toBool(false));

    mMessageType = Message::MessageType::NormalText;
    if (!type.isEmpty()) {
        if (type == "videoconf"_L1) {
            mMessageType = VideoConference;
            // qDebug() << " VIDEO " << o;
        } else {
            mSystemMessageType = SystemMessageTypeUtil::systemMessageTypeFromString(type);
            mMessageType = System;
        }
    }
    parseBlocks(o.value("blocks"_L1).toArray());
    parseMentions(o.value("mentions"_L1).toArray());

    parseAttachment(o.value("attachments"_L1).toArray());
    parseUrls(o.value("urls"_L1).toArray());
    parseReactions(o.value("reactions"_L1).toObject(), emojiManager);
    parseChannels(o.value("channels"_L1).toArray());
    // TODO unread element
}

void Message::parseReactions(const QJsonObject &reacts, EmojiManager *emojiManager)
{
    if (!reacts.isEmpty()) {
        if (!mReactions) {
            mReactions = new Reactions;
        } else {
            mReactions.reset(new Reactions);
        }
        mReactions->parseReactions(reacts, emojiManager);
    }
}

bool Message::isEditingMode() const
{
    return messageStateValue(Edited);
}

void Message::setIsEditingMode(bool isEditingMode)
{
    assignMessageStateValue(Edited, isEditingMode);
}

bool Message::showIgnoredMessage() const
{
    return messageStateValue(Ignored);
}

void Message::setShowIgnoredMessage(bool showIgnoredMessage)
{
    assignMessageStateValue(Ignored, showIgnoredMessage);
}

bool Message::pendingMessage() const
{
    return messageStateValue(Pending);
}

void Message::setPendingMessage(bool pendingMessage)
{
    assignMessageStateValue(Pending, pendingMessage);
}

QString Message::emoji() const
{
    return mEmoji;
}

void Message::setEmoji(const QString &emoji)
{
    mEmoji = emoji;
}

QList<QByteArray> Message::replies() const
{
    return mReplies;
}

void Message::setReplies(const QList<QByteArray> &replies)
{
    mReplies = replies;
}

QString Message::name() const
{
    return mName;
}

void Message::setName(const QString &name)
{
    mName = name;
}

bool Message::isAutoTranslated() const
{
    if (mMessageTranslation) {
        return !mMessageTranslation->isEmpty();
    }
    return false;
}

bool Message::showTranslatedMessage() const
{
    return messageStateValue(Translated);
}

void Message::setShowTranslatedMessage(bool showOriginalMessage)
{
    assignMessageStateValue(Translated, showOriginalMessage);
}

const MessageTranslation *Message::messageTranslation() const
{
    if (mMessageTranslation) {
        return mMessageTranslation.data();
    }
    return nullptr;
}

void Message::setMessageTranslation(const MessageTranslation &messageTranslation)
{
    if (!mMessageTranslation) {
        mMessageTranslation = new MessageTranslation(messageTranslation);
    } else {
        mMessageTranslation.reset(new MessageTranslation(messageTranslation));
    }
}

QString Message::displayTime() const
{
    return mDisplayTime;
}

QByteArray Message::threadMessageId() const
{
    if (mMessageExtra) {
        return mMessageExtra->threadMessageId();
    }
    return {};
}

void Message::setThreadMessageId(const QByteArray &threadMessageId)
{
    messageExtra()->setThreadMessageId(threadMessageId);
}

QByteArray Message::discussionRoomId() const
{
    if (mMessageExtra) {
        return mMessageExtra->discussionRoomId();
    }
    return {};
}

void Message::setDiscussionRoomId(const QByteArray &discussionRoomId)
{
    messageExtra()->setDiscussionRoomId(discussionRoomId);
}

int Message::discussionCount() const
{
    if (mMessageExtra) {
        return mMessageExtra->discussionCount();
    }
    return {};
}

void Message::setDiscussionCount(int discussionCount)
{
    messageExtra()->setDiscussionCount(discussionCount);
}

qint64 Message::discussionLastMessage() const
{
    if (mMessageExtra) {
        return mMessageExtra->discussionLastMessage();
    }
    return -1;
}

void Message::setDiscussionLastMessage(qint64 discussionLastMessage)
{
    messageExtra()->setDiscussionLastMessage(discussionLastMessage);
}

qint64 Message::threadLastMessage() const
{
    if (mMessageExtra) {
        return mMessageExtra->threadLastMessage();
    }
    return -1;
}

void Message::setThreadLastMessage(qint64 threadLastMessage)
{
    messageExtra()->setThreadLastMessage(threadLastMessage);
}

int Message::threadCount() const
{
    if (mMessageExtra) {
        return mMessageExtra->threadCount();
    }
    return 0;
}

void Message::setThreadCount(int threadCount)
{
    messageExtra()->setThreadCount(threadCount);
}

MessageStarred Message::messageStarred() const
{
    return mMessageStarred;
}

void Message::setMessageStarred(MessageStarred messageStarred)
{
    mMessageStarred = messageStarred;
}

const MessagePinned *Message::messagePinned() const
{
    if (mMessagePinned) {
        return mMessagePinned.data();
    }
    return nullptr;
}

void Message::setMessagePinned(const MessagePinned &messagePinned)
{
    if (!mMessagePinned) {
        mMessagePinned = new MessagePinned(messagePinned);
    } else {
        mMessagePinned.reset(new MessagePinned(messagePinned));
    }
}

bool Message::unread() const
{
    return messageStateValue(Unread);
}

void Message::setUnread(bool unread)
{
    assignMessageStateValue(Unread, unread);
}

QString Message::role() const
{
    return mRole;
}

void Message::setRole(const QString &role)
{
    mRole = role;
}

void Message::parseChannels(const QJsonArray &channels)
{
    mChannels.clear();
    for (int i = 0, total = channels.size(); i < total; ++i) {
        const QJsonObject mention = channels.at(i).toObject();
        mChannels.insert(mention.value("name"_L1).toString(), mention.value("_id"_L1).toString().toLatin1());
    }
}

QList<Block> Message::blocks() const
{
    return mBlocks;
}

void Message::setBlocks(const QList<Block> &newBlocks)
{
    mBlocks = newBlocks;
}

QString Message::originalMessageOrAttachmentDescription() const
{
    if (attachments().empty()) {
        return text();
    }
    return attachments().constFirst().description();
}

MessageExtra *Message::messageExtra()
{
    if (!mMessageExtra) {
        mMessageExtra = new MessageExtra;
    }
    return mMessageExtra;
}

QString Message::localTranslation() const
{
    if (!mMessageExtra) {
        return {};
    }
    return mMessageExtra->localTranslation();
}

void Message::setLocalTranslation(const QString &newLocalTranslation)
{
    messageExtra()->setLocalTranslation(newLocalTranslation);
}

bool Message::hoverHighlight() const
{
    return messageStateValue(HoverHighlight);
}

void Message::setHoverHighlight(bool newShowReactionIcon)
{
    assignMessageStateValue(HoverHighlight, newShowReactionIcon);
}

const QMap<QString, QByteArray> &Message::channels() const
{
    return mChannels;
}

void Message::setChannels(const QMap<QString, QByteArray> &newChannels)
{
    mChannels = newChannels;
}

void Message::parseBlocks(const QJsonArray &blocks)
{
    mBlocks.clear();
    for (int i = 0, total = blocks.count(); i < total; ++i) {
        const QJsonObject blockObject = blocks.at(i).toObject();
        Block b;
        b.parseBlock(blockObject);
        if (b.isValid()) {
            mBlocks.append(std::move(b));
        }
    }
}

bool Message::privateMessage() const
{
    return messageStateValue(Private);
}

void Message::setPrivateMessage(bool newPrivateMessage)
{
    assignMessageStateValue(Private, newPrivateMessage);
}

const ModerationMessage *Message::moderationMessage() const
{
    if (mModerationMessage) {
        return mModerationMessage.data();
    }
    return nullptr;
}

void Message::setModerationMessage(const ModerationMessage &newModerationMessage)
{
    if (!mModerationMessage) {
        mModerationMessage = new ModerationMessage(newModerationMessage);
    } else {
        mModerationMessage.reset(new ModerationMessage(newModerationMessage));
    }
}

void Message::setVideoConferenceInfo(const VideoConferenceInfo &info)
{
    auto it = std::find_if(mBlocks.cbegin(), mBlocks.cend(), [info](const auto &block) {
        return block.blockId() == info.blockId();
    });
    if (it != mBlocks.cend()) {
        mBlocks.removeAll(*it);
        Block b(*it);
        b.setVideoConferenceInfo(info);
        mBlocks.append(b);
    }
}

void Message::parseMentions(const QJsonArray &mentions)
{
    mMentions.clear();
    for (int i = 0; i < mentions.size(); i++) {
        const QJsonObject mention = mentions.at(i).toObject();
        mMentions.insert(mention.value("username"_L1).toString(), mention.value("_id"_L1).toString().toLatin1());
    }
}

void Message::parseUrls(const QJsonArray &urls)
{
    mUrls.clear();
    if (!urls.isEmpty()) {
        qCDebug(RUQOLA_LOG) << " void Message::urls(const QJsonObject &attachments)" << urls;
        for (int i = 0; i < urls.size(); i++) {
            const QJsonObject url = urls.at(i).toObject();
            MessageUrl messageUrl;
            messageUrl.setUrlId(Message::generateUniqueId(mMessageId, i));
            messageUrl.parseUrl(url);
            if (!messageUrl.isEmpty()) {
                mUrls.append(messageUrl);
            }
        }
    }
}

const Reactions *Message::reactions() const
{
    if (mReactions) {
        return mReactions.data();
    }
    return nullptr;
}

void Message::setReactions(const Reactions &reactions)
{
    if (!mReactions) {
        mReactions = new Reactions(reactions);
    } else {
        mReactions.reset(new Reactions(reactions));
    }
}

bool Message::isPinned() const
{
    if (mMessagePinned) {
        return mMessagePinned->pinned();
    }
    return false;
}

bool Message::isStarred() const
{
    return mMessageStarred.isStarred();
}

void Message::setIsStarred(bool starred)
{
    mMessageStarred.setIsStarred(starred);
}

QMap<QString, QByteArray> Message::mentions() const
{
    return mMentions;
}

void Message::setMentions(const QMap<QString, QByteArray> &mentions)
{
    mMentions = mentions;
}

void Message::parseAttachment(const QJsonArray &attachments)
{
    mAttachments.clear();
    if (!attachments.isEmpty()) {
        // qDebug() << " void Message::parseAttachment(const QJsonObject &attachments)"<<attachments;
        for (int i = 0; i < attachments.size(); i++) {
            const QJsonObject attachment = attachments.at(i).toObject();
            MessageAttachment messageAttachement;
            messageAttachement.parseAttachment(attachment);
            messageAttachement.setAttachmentId(Message::generateUniqueId(messageId(), i));
            if (messageAttachement.isValid()) {
                mAttachments.append(messageAttachement);
            }
        }
    }
}

bool Message::operator==(const Message &other) const
{
    bool result = (mMessageId == other.messageId()) && (mRoomId == other.roomId()) && (mText == other.text()) && (mTimeStamp == other.timeStamp())
        && (mUsername == other.username()) && (mName == other.name()) && (mUserId == other.userId()) && (mUpdatedAt == other.updatedAt())
        && (mEditedAt == other.editedAt()) && (mEditedByUsername == other.editedByUsername()) && (mAlias == other.alias()) && (mAvatar == other.avatar())
        && (mSystemMessageType == other.systemMessageType()) && (groupable() == other.groupable()) && (parseUrls() == other.parseUrls())
        && (mUrls == other.urls()) && (mAttachments == other.attachments()) && (mMentions == other.mentions()) && (mRole == other.role())
        && (unread() == other.unread()) && (mMessageStarred == other.messageStarred()) && (threadCount() == other.threadCount())
        && (threadLastMessage() == other.threadLastMessage()) && (discussionCount() == other.discussionCount())
        && (discussionLastMessage() == other.discussionLastMessage()) && (discussionRoomId() == other.discussionRoomId())
        && (threadMessageId() == other.threadMessageId()) && (showTranslatedMessage() == other.showTranslatedMessage()) && (mReplies == other.replies())
        && (mEmoji == other.emoji()) && (pendingMessage() == other.pendingMessage()) && (showIgnoredMessage() == other.showIgnoredMessage())
        && (mChannels == other.channels()) && (localTranslation() == other.localTranslation()) && (mBlocks == other.blocks())
        && (mDisplayTime == other.mDisplayTime) && (privateMessage() == other.privateMessage());
    if (!result) {
        return false;
    }
    // compare reactions
    if (messagePinned() && other.messagePinned()) {
        if (*messagePinned() == (*other.messagePinned())) {
            result = true;
        } else {
            return false;
        }
    } else if (!messagePinned() && !other.messagePinned()) {
        result = true;
    } else {
        return false;
    }

    // compare reactions
    if (reactions() && other.reactions()) {
        if (*reactions() == (*other.reactions())) {
            result = true;
        } else {
            return false;
        }
    } else if (!reactions() && !other.reactions()) {
        result = true;
    } else {
        return false;
    }
    // compare messageTranslation
    if (messageTranslation() && other.messageTranslation()) {
        if (*messageTranslation() == (*other.messageTranslation())) {
            result = true;
        } else {
            return false;
        }
    } else if (!messageTranslation() && !other.messageTranslation()) {
        result = true;
    } else {
        return false;
    }
    // Compare moderationMessage
    if (moderationMessage() && other.moderationMessage()) {
        if (*moderationMessage() == (*other.moderationMessage())) {
            result = true;
        } else {
            return false;
        }
    } else if (!moderationMessage() && !other.moderationMessage()) {
        result = true;
    } else {
        return false;
    }
    return result;
}

bool Message::operator<(const Message &other) const
{
    return mTimeStamp < other.mTimeStamp;
}

SystemMessageTypeUtil::SystemMessageType Message::systemMessageType() const
{
    return mSystemMessageType;
}

void Message::setSystemMessageType(const SystemMessageTypeUtil::SystemMessageType &systemMessageType)
{
    mSystemMessageType = systemMessageType;
}

Message::MessageType Message::messageType() const
{
    return mMessageType;
}

QString Message::systemMessageText() const
{
    switch (mSystemMessageType) {
    case SystemMessageTypeUtil::SystemMessageType::UserJoined:
        return i18n("%1 has joined the channel", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserLeft:
        return i18n("%1 has left the channel", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserLeftTeam:
        return i18n("%1 left this Team", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomTopicChanged:
        if (mText.isEmpty()) {
            return i18n("Topic was cleared by: %1", mUsername);
        } else {
            return i18n("%2 changed topic to: <i>%1</i>", mText, mUsername);
        }
    case SystemMessageTypeUtil::SystemMessageType::UserAdded:
        return i18n("%2 added %1 to the conversation", mText, mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomNameChanged:
        return i18n("%2 changed room name to <a href=\"ruqola:/room/%1\">#%1</a>", mText, mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserRemoved:
        return i18n("%2 removed user %1", mText, mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomDescriptionChanged:
        if (mText.isEmpty()) {
            return i18n("Description was cleared by %1", mUsername);
        } else {
            return i18n("%2 changed room description to %1", mText, mUsername);
        }
    case SystemMessageTypeUtil::SystemMessageType::RoomAnnoucementChanged:
        if (mText.isEmpty()) {
            return i18n("Announcement was cleared by %1", mUsername);
        } else {
            return i18n("%2 changed room announcement to %1", mText, mUsername);
        }
    case SystemMessageTypeUtil::SystemMessageType::RoomPrivacyChanged:
        return i18n("%2 changed room privacy to %1", mText, mUsername);
    case SystemMessageTypeUtil::SystemMessageType::JitsiCallStarted:
        return QStringLiteral("<a href=\"ruqola:/jitsicall/\">") + i18n("Click to join to video") + QStringLiteral("</a>");
    case SystemMessageTypeUtil::SystemMessageType::MessageDeleted:
        // TODO encrypted message
        return i18n("Message Deleted");
    case SystemMessageTypeUtil::SystemMessageType::Pinned:
        return i18n("Message Pinned");
    case SystemMessageTypeUtil::SystemMessageType::EncryptedMessage:
        return i18n("Encrypted Message");
    case SystemMessageTypeUtil::SystemMessageType::UserUnmuted:
        return i18n("%1 was unmuted by %2", mText, mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserMuted:
        return i18n("%1 was muted by %2", mText, mUsername);
    case SystemMessageTypeUtil::SystemMessageType::SubscriptionRoleAdded:
        return i18n("Role \'%3\' was added to %1 by %2", mText, mUsername, mRole);
    case SystemMessageTypeUtil::SystemMessageType::SubscriptionRoleRemoved:
        return i18n("Role \'%3\' was removed to %1 by %2", mText, mUsername, mRole);
    case SystemMessageTypeUtil::SystemMessageType::MessageE2E:
        // TODO need to unencrypt it
        // return i18n("Encrypted message: %1", mText);
        return i18n("This message is end-to-end encrypted. To view it, you must enter your encryption key in your account settings.");
    case SystemMessageTypeUtil::SystemMessageType::DiscussionCreated:
        return i18n("Discussion created about \"%1\"", mText);
    case SystemMessageTypeUtil::SystemMessageType::UserJoinedConversation:
        return i18n("%1 has joined the conversation", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomArchived:
        return i18n("This room has been archived by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomUnarchived:
        return i18n("This room has been unarchived by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::Rtc:
        qCWarning(RUQOLA_LOG) << "Need to implement : " << mSystemMessageType << " mText " << mText;
        return i18n("Unknown action!");
    case SystemMessageTypeUtil::SystemMessageType::Welcome:
        // TODO verify
        qCWarning(RUQOLA_LOG) << "Need to implement : " << mSystemMessageType << " mText " << mText;
        return i18n("Welcome %1!", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomAvatarChanged:
        return i18n("Room avatar changed by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomE2eEnabled:
        return i18n("This room's encryption has been enabled by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomE2eDisabled:
        return i18n("This room's encryption has been disabled by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomSetReadOnly:
        return i18n("Room set as Read Only by  %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomRemoveReadOnly:
        return i18n("Room added writing permission by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::AddedUserToTeam:
        return i18n("%1 added @%2 to this Team", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::RemovedUserFromTeam:
        return i18n("%1 removed @%2 from this Team", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::UserConvertedToTeam:
        return i18n("%1 converted #%2 to a Team", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::UserConvertedToChannel:
        return i18n("%1 converted #%2 to a Channel", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::UserRemovedRoomFromTeam:
        return i18n("%1 removed #%2 from this Team", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::UserDeletedRoomFromTeam:
        return i18n("%1 deleted #%2", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::UserAddedRoomToTeam:
        return i18n("%1 added #%2 to this Team", mUsername, mText);
    case SystemMessageTypeUtil::SystemMessageType::RoomAllowedReacting:
        return i18n("Room allowed reacting by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::RoomDisallowedReacting:
        return i18n("Room disallowed reacting by %1", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserJoinedTeam:
        return i18n("%1 joined this Team", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserJoinedOtr:
        return i18n("%1 has joined OTR chat.", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserKeyRefreshedSuccessfully:
        return i18n("%1 key refreshed successfully", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::UserRequesterOtrKeyRefresh:
        return i18n("%1 has requested key refresh.", mUsername);
    case SystemMessageTypeUtil::SystemMessageType::VideoConf:
        return i18n("Conference Call");
    case SystemMessageTypeUtil::SystemMessageType::Unknown:
        qCWarning(RUQOLA_LOG) << "Unknown type for message: type: " << mSystemMessageType << " mText " << mText;
        break;
    }
    return i18n("Unknown action!");
}

void Message::setMessageType(MessageType messageType)
{
    mMessageType = messageType;
}

QList<MessageAttachment> Message::attachments() const
{
    return mAttachments;
}

void Message::setAttachments(const QList<MessageAttachment> &attachments)
{
    mAttachments = attachments;
}

QList<MessageUrl> Message::urls() const
{
    return mUrls;
}

void Message::setUrls(const QList<MessageUrl> &urls)
{
    mUrls = urls;
}

QString Message::alias() const
{
    return mAlias;
}

void Message::setAlias(const QString &alias)
{
    mAlias = alias;
}

QString Message::editedByUsername() const
{
    return mEditedByUsername;
}

void Message::setEditedByUsername(const QString &editedByUsername)
{
    mEditedByUsername = editedByUsername;
}

bool Message::wasEdited() const
{
    return !mEditedByUsername.isEmpty();
}

qint64 Message::editedAt() const
{
    return mEditedAt;
}

void Message::setEditedAt(qint64 editedAt)
{
    if (mEditedAt != editedAt) {
        mEditedAt = editedAt;
    }
}

qint64 Message::updatedAt() const
{
    return mUpdatedAt;
}

void Message::setUpdatedAt(qint64 updatedAt)
{
    mUpdatedAt = updatedAt;
}

QByteArray Message::userId() const
{
    return mUserId;
}

void Message::setUserId(const QByteArray &userId)
{
    mUserId = userId;
}

QString Message::username() const
{
    return mUsername;
}

void Message::setUsername(const QString &username)
{
    mUsername = username;
}

qint64 Message::timeStamp() const
{
    return mTimeStamp;
}

void Message::setTimeStamp(qint64 timeStamp)
{
    if (mTimeStamp != timeStamp) {
        mTimeStamp = timeStamp;
        mDisplayTime = QDateTime::fromMSecsSinceEpoch(mTimeStamp).time().toString(QStringLiteral("hh:mm"));
    }
}

QString Message::text() const
{
    return mText;
}

void Message::setText(const QString &text)
{
    mText = text;
}

QByteArray Message::messageId() const
{
    return mMessageId;
}

void Message::setMessageId(const QByteArray &messageId)
{
    mMessageId = messageId;
}

QByteArray Message::roomId() const
{
    return mRoomId;
}

void Message::setRoomId(const QByteArray &roomId)
{
    mRoomId = roomId;
}

QString Message::avatar() const
{
    return mAvatar;
}

void Message::setAvatar(const QString &avatar)
{
    mAvatar = avatar;
}

bool Message::parseUrls() const
{
    return messageStateValue(ParsedUrl);
}

void Message::setParseUrls(bool parseUrls)
{
    assignMessageStateValue(ParsedUrl, parseUrls);
}

bool Message::groupable() const
{
    return messageStateValue(Groupable);
}

void Message::setGroupable(bool groupable)
{
    assignMessageStateValue(Groupable, groupable);
}

QByteArray Message::generateUniqueId(const QByteArray &messageId, int index)
{
    return messageId + QByteArray("_") + QByteArray::number(index);
}

Utils::AvatarInfo Message::avatarInfo() const
{
    Utils::AvatarInfo info; // Optimize ???
    info.avatarType = Utils::AvatarType::User;
    info.identifier = mUsername;
    return info;
}

Message Message::deserialize(const QJsonObject &o, EmojiManager *emojiManager)
{
    Message message;
    if (o.contains("tcount"_L1)) {
        message.setThreadCount(o["tcount"_L1].toInt());
    }
    if (o.contains("tmid"_L1)) {
        message.setThreadMessageId(o["tmid"_L1].toString().toLatin1());
    }
    if (o.contains("dcount"_L1)) {
        message.setDiscussionCount(o["dcount"_L1].toInt());
    }

    if (o.contains("drid"_L1)) {
        message.setDiscussionRoomId(o.value("drid"_L1).toString().toLatin1());
    }

    message.assignMessageStateValue(Private, o["private"_L1].toBool(false));
    if (o.contains("tlm"_L1)) {
        message.setThreadLastMessage(static_cast<qint64>(o["tlm"_L1].toDouble()));
    }
    if (o.contains("dlm"_L1)) {
        message.setDiscussionLastMessage(static_cast<qint64>(o["dlm"_L1].toDouble()));
    }

    message.mMessageId = o["messageID"_L1].toString().toLatin1();
    message.mRoomId = o["roomID"_L1].toString().toLatin1();
    message.mText = o["message"_L1].toString();
    message.setTimeStamp(static_cast<qint64>(o["timestamp"_L1].toDouble()));
    message.mUsername = o["username"_L1].toString();
    message.mName = o["name"_L1].toString();
    message.mUserId = o["userID"_L1].toString().toLatin1();
    message.mUpdatedAt = static_cast<qint64>(o["updatedAt"_L1].toDouble());
    message.setEditedAt(static_cast<qint64>(o["editedAt"_L1].toDouble()));
    message.mEditedByUsername = o["editedByUsername"_L1].toString();
    message.mAlias = o["alias"_L1].toString();
    message.mAvatar = o["avatar"_L1].toString();
    message.assignMessageStateValue(Message::MessageState::Groupable, o["groupable"_L1].toBool());
    message.assignMessageStateValue(Message::MessageState::ParsedUrl, o["parseUrls"_L1].toBool());
    message.mMessageStarred.setIsStarred(o["starred"_L1].toBool());

    if (o.contains("pinnedMessage"_L1)) {
        MessagePinned *pinned = MessagePinned::deserialize(o["pinnedMessage"_L1].toObject());
        message.setMessagePinned(*pinned);
        delete pinned;
    }

    message.mRole = o["role"_L1].toString();
    message.mSystemMessageType = SystemMessageTypeUtil::systemMessageTypeFromString(o["type"_L1].toString());
    message.mEmoji = o["emoji"_L1].toString();
    message.mMessageType = o["messageType"_L1].toVariant().value<MessageType>();
    const QJsonArray attachmentsArray = o.value("attachments"_L1).toArray();
    for (int i = 0; i < attachmentsArray.count(); ++i) {
        const QJsonObject attachment = attachmentsArray.at(i).toObject();
        MessageAttachment att = MessageAttachment::deserialize(attachment);
        att.setAttachmentId(Message::generateUniqueId(message.messageId(), i));
        if (att.isValid()) {
            message.mAttachments.append(std::move(att));
        }
    }
    const QJsonArray urlsArray = o.value("urls"_L1).toArray();
    for (int i = 0; i < urlsArray.count(); ++i) {
        const QJsonObject urlObj = urlsArray.at(i).toObject();
        MessageUrl url = MessageUrl::deserialize(urlObj);
        url.setUrlId(Message::generateUniqueId(message.messageId(), i));
        if (!url.isEmpty()) {
            message.mUrls.append(std::move(url));
        }
    }
    if (o.contains("reactions"_L1)) {
        const QJsonObject reactionsArray = o.value("reactions"_L1).toObject();
        Reactions *reaction = Reactions::deserialize(reactionsArray);
        message.setReactions(*reaction);
        delete reaction;
    }

    const QJsonArray repliesArray = o.value("replies"_L1).toArray();
    QList<QByteArray> replies;
    replies.reserve(repliesArray.count());
    for (int i = 0, total = repliesArray.count(); i < total; ++i) {
        replies.append(repliesArray.at(i).toString().toLatin1());
    }
    message.setReplies(replies);

    QMap<QString, QByteArray> mentions;
    const QJsonArray mentionsArray = o.value("mentions"_L1).toArray();
    for (int i = 0, total = mentionsArray.count(); i < total; ++i) {
        const QJsonObject mention = mentionsArray.at(i).toObject();
        mentions.insert(mention.value("username"_L1).toString(), mention.value("_id"_L1).toString().toLatin1());
    }
    message.setMentions(mentions);

    QMap<QString, QByteArray> channels;
    const QJsonArray channelsArray = o.value("channels"_L1).toArray();
    for (int i = 0, total = channelsArray.count(); i < total; ++i) {
        const QJsonObject channel = channelsArray.at(i).toObject();
        channels.insert(channel.value("channel"_L1).toString(), channel.value("_id"_L1).toString().toLatin1());
    }
    message.setChannels(channels);

    const QJsonArray blocksArray = o.value("blocks"_L1).toArray();
    for (int i = 0, total = blocksArray.count(); i < total; ++i) {
        const Block block = Block::deserialize(blocksArray.at(i).toObject());
        message.mBlocks.append(std::move(block));
    }

    if (o.contains("localTransation"_L1)) {
        message.setLocalTranslation(o["localTransation"_L1].toString());
    }

    if (o.contains("messageTranslation"_L1)) {
        MessageTranslation *translation = MessageTranslation::deserialize(o["messageTranslation"_L1].toArray());
        message.setMessageTranslation(*translation);
        delete translation;
    }

    return message;
}

QByteArray Message::serialize(const Message &message, bool toBinary)
{
    QJsonDocument d;
    QJsonObject o;

    o["messageID"_L1] = QString::fromLatin1(message.mMessageId);
    o["roomID"_L1] = QString::fromLatin1(message.mRoomId);
    o["message"_L1] = message.mText;
    o["timestamp"_L1] = message.mTimeStamp;
    o["username"_L1] = message.mUsername;
    if (!message.mName.isEmpty()) {
        o["name"_L1] = message.mName;
    }
    o["userID"_L1] = QString::fromLatin1(message.mUserId);
    o["updatedAt"_L1] = message.mUpdatedAt;
    o["editedAt"_L1] = message.mEditedAt;
    if (message.threadLastMessage() > -1) {
        o["tlm"_L1] = message.threadLastMessage();
    }

    o["editedByUsername"_L1] = message.mEditedByUsername;
    o["alias"_L1] = message.mAlias;
    o["avatar"_L1] = message.mAvatar;
    o["groupable"_L1] = message.messageStateValue(Message::MessageState::Groupable);
    o["parseUrls"_L1] = message.parseUrls();
    o["starred"_L1] = message.mMessageStarred.isStarred();

    if (message.mMessagePinned) {
        o["pinnedMessage"_L1] = MessagePinned::serialize(*message.mMessagePinned);
    }

    if (!message.mRole.isEmpty()) {
        o["role"_L1] = message.mRole;
    }
    if (!message.mEmoji.isEmpty()) {
        o["emoji"_L1] = message.mEmoji;
    }

    o["type"_L1] = SystemMessageTypeUtil::systemMessageTypeStringFromEnum(message.mSystemMessageType);
    o["messageType"_L1] = QJsonValue::fromVariant(QVariant::fromValue<Message::MessageType>(message.mMessageType));

    // Attachments
    if (!message.mAttachments.isEmpty()) {
        QJsonArray array;
        const int nbAttachment = message.mAttachments.count();
        for (int i = 0; i < nbAttachment; ++i) {
            array.append(MessageAttachment::serialize(message.mAttachments.at(i)));
        }
        o["attachments"_L1] = array;
    }

    // Mentions
    if (!message.mentions().isEmpty()) {
        QMapIterator<QString, QByteArray> i(message.mentions());
        QJsonArray array;
        while (i.hasNext()) {
            i.next();
            QJsonObject mention;
            mention.insert("_id"_L1, QString::fromLatin1(i.value()));
            mention.insert("username"_L1, i.key());
            array.append(std::move(mention));
        }
        o["mentions"_L1] = array;
    }

    // Channels
    if (!message.channels().isEmpty()) {
        QMapIterator<QString, QByteArray> j(message.channels());
        QJsonArray array;
        while (j.hasNext()) {
            j.next();
            QJsonObject channel;
            channel.insert("_id"_L1, QString::fromLatin1(j.value()));
            channel.insert("channel"_L1, j.key());
            array.append(std::move(channel));
        }
        o["channels"_L1] = array;
    }
    // Urls
    if (!message.mUrls.isEmpty()) {
        QJsonArray array;
        const int nbUrls = message.mUrls.count();
        for (int i = 0; i < nbUrls; ++i) {
            array.append(MessageUrl::serialize(message.mUrls.at(i)));
        }
        o["urls"_L1] = array;
    }

    if (message.reactions() && !message.reactions()->isEmpty()) {
        o["reactions"_L1] = Reactions::serialize(*message.reactions());
    }

    if (message.threadCount() > 0) {
        o["tcount"_L1] = message.threadCount();
        o["tlm"_L1] = message.threadLastMessage();
    }

    if (message.discussionCount() > 0) {
        o["dcount"_L1] = message.discussionCount();
        o["dlm"_L1] = message.discussionLastMessage();
    }
    if (!message.discussionRoomId().isEmpty()) {
        o["drid"_L1] = QString::fromLatin1(message.discussionRoomId());
    }

    if (!message.threadMessageId().isEmpty()) {
        o["tmid"_L1] = QString::fromLatin1(message.threadMessageId());
    }
    if (!message.mReplies.isEmpty()) {
        QStringList serialize;
        for (const QByteArray &i : std::as_const(message.mReplies)) {
            serialize << QString::fromLatin1(i);
        }
        o["replies"_L1] = QJsonArray::fromStringList(serialize);
    }

    if (!message.mBlocks.isEmpty()) {
        QJsonArray blockArray;
        const int nBlocks = message.mBlocks.count();
        for (int i = 0; i < nBlocks; ++i) {
            blockArray.append(Block::serialize(message.mBlocks.at(i)));
        }
        o["blocks"_L1] = blockArray;
    }
    if (!message.localTranslation().isEmpty()) {
        o["localTransation"_L1] = message.localTranslation();
    }
    if (message.messageTranslation() && !message.messageTranslation()->isEmpty()) {
        o["messageTranslation"_L1] = MessageTranslation::serialize(*message.messageTranslation());
    }
    if (message.messageStateValue(Private)) {
        o["private"_L1] = true;
    }

    if (toBinary) {
        return QCborValue::fromJsonValue(o).toCbor();
    }

    d.setObject(o);
    return d.toJson(QJsonDocument::Indented);
}

QDebug operator<<(QDebug d, const Message &t)
{
    d.space() << "mMessageId:" << t.messageId();
    d.space() << "mText:" << t.text();
    d.space() << "mTimeStamp:" << t.timeStamp();
    d.space() << "mUsername:" << t.username();
    d.space() << "mName:" << t.name();
    d.space() << "mUserId:" << t.userId();
    d.space() << "mUpdatedAt:" << t.updatedAt();
    d.space() << "mEditedAt:" << t.editedAt();
    d.space() << "mEditedByUsername:" << t.editedByUsername();
    d.space() << "mAlias:" << t.alias();
    d.space() << "mSystemMessageType:" << t.systemMessageType();
    d.space() << "mRoomId:" << t.roomId();
    d.space() << "mAvatar:" << t.avatar();
    d.space() << "mGroupable:" << t.groupable();
    d.space() << "mParseUrls:" << t.parseUrls();
    for (int i = 0, total = t.attachments().count(); i < total; ++i) {
        d.space() << "Attachment:" << t.attachments().at(i);
    }
    for (int i = 0, total = t.urls().count(); i < total; ++i) {
        d.space() << "Urls:" << t.urls().at(i);
    }
    d.space() << "Mentions:" << t.mentions();
    d.space() << "mMessageType:" << t.messageType();
    d.space() << "mRole:" << t.role();
    if (t.reactions()) {
        d.space() << "mReaction:" << *t.reactions();
    }
    d.space() << "mUnread:" << t.unread();
    d.space() << "starred" << t.messageStarred();
    if (t.messagePinned()) {
        d.space() << "pinned" << *t.messagePinned();
    }
    d.space() << "threadcount" << t.threadCount();
    d.space() << "threadlastmessage" << t.threadLastMessage();
    d.space() << "discussionlastmessage" << t.discussionLastMessage();
    d.space() << "discussionRoomId" << t.discussionRoomId();
    d.space() << "threadMessageId" << t.threadMessageId();
    if (t.messageTranslation()) {
        d.space() << "messagetranslation" << *t.messageTranslation();
    }
    d.space() << "mShowOriginalMessage" << t.showTranslatedMessage();
    d.space() << "mReplies" << t.replies();
    d.space() << "mEmoji" << t.emoji();
    d.space() << "mPendingMessage" << t.pendingMessage();
    d.space() << "mShowIgnoredMessage" << t.showIgnoredMessage();
    d.space() << "mChannels" << t.channels();
    d.space() << "mLocalTranslation" << t.localTranslation();
    d.space() << "mDiscussionRoomId" << t.discussionRoomId();
    d.space() << "mDiscussionCount" << t.discussionCount();
    d.space() << "mDiscussionLastMessage" << t.discussionLastMessage();
    if (t.moderationMessage()) {
        d.space() << "moderationMessage" << *t.moderationMessage();
    }

    for (int i = 0, total = t.blocks().count(); i < total; ++i) {
        d.space() << "block:" << t.blocks().at(i);
    }

    d.space() << "mPrivateMessage" << t.privateMessage();
    return d;
}

QString Message::dateTime() const
{
    const QDateTime currentDate = QDateTime::fromMSecsSinceEpoch(timeStamp());
    return currentDate.toString();
}

bool Message::messageStateValue(MessageState type) const
{
    return mMessageStates & type;
}

Message::MessageStates Message::messageStates() const
{
    return mMessageStates;
}

void Message::setMessageStates(const MessageStates &newMessageStates)
{
    mMessageStates = newMessageStates;
}

void Message::assignMessageStateValue(MessageState type, bool status)
{
    if (status) {
        mMessageStates |= type;
    } else {
        mMessageStates &= ~type;
    }
}

#include "moc_message.cpp"
