// SPDX-FileCopyrightText: 2020 Tobias Fella <tobias.fella@kde.org>
// SPDX-License-Identifier: GPL-2.0-or-later

import QtQuick
import QtQuick.Controls as QQC2
import QtQuick.Layouts
import QtQuick.Window
import Qt.labs.platform

import org.kde.kirigami as Kirigami
import org.kde.kirigamiaddons.formcard as FormCard
import org.kde.kirigamiaddons.labs.components as KirigamiComponents

import org.kde.neochat

FormCard.FormCardPage {
    id: root

    title: i18n("Accounts")

    FormCard.FormHeader {
        title: i18n("Accounts")
    }
    FormCard.FormCard {
        Repeater {
            model: AccountRegistry
            delegate: FormCard.AbstractFormDelegate {
                id: accountDelegate
                required property NeoChatConnection connection
                Layout.fillWidth: true
                onClicked: applicationWindow().pageStack.layers.push(Qt.createComponent('org.kde.neochat.settings', 'AccountEditorPage'), {
                    connection: accountDelegate.connection
                }, {
                    title: i18n("Account editor")
                })

                contentItem: RowLayout {
                    KirigamiComponents.Avatar {
                        name: accountDelegate.connection.localUser.displayName
                        source: accountDelegate.connection.localUser.avatarMediaId ? ("image://mxc/" + accountDelegate.connection.localUser.avatarMediaId) : ""

                        Layout.rightMargin: Kirigami.Units.largeSpacing
                        implicitWidth: Kirigami.Units.iconSizes.medium
                        implicitHeight: Kirigami.Units.iconSizes.medium
                    }

                    ColumnLayout {
                        Layout.fillWidth: true
                        spacing: Kirigami.Units.smallSpacing

                        QQC2.Label {
                            Layout.fillWidth: true
                            text: accountDelegate.connection.localUser.displayName
                            textFormat: Text.PlainText
                            elide: Text.ElideRight
                            wrapMode: Text.Wrap
                            maximumLineCount: 2
                            color: Kirigami.Theme.textColor
                        }

                        QQC2.Label {
                            Layout.fillWidth: true
                            text: accountDelegate.connection.localUserId
                            color: Kirigami.Theme.disabledTextColor
                            font: Kirigami.Theme.smallFont
                            elide: Text.ElideRight
                        }
                    }

                    QQC2.ToolButton {
                        text: i18n("Logout")
                        icon.name: "im-kick-user"
                        onClicked: confirmLogoutDialogComponent.createObject(applicationWindow().overlay).open()
                    }

                    Component {
                        id: confirmLogoutDialogComponent
                        ConfirmLogoutDialog {
                            connection: accountDelegate.connection
                            onAccepted: {
                                if (AccountRegistry.accountCount === 1) {
                                    root.Window.window.close();
                                }
                            }
                        }
                    }

                    FormCard.FormArrow {
                        Layout.alignment: Qt.AlignRight | Qt.AlignVCenter
                        direction: Qt.RightArrow
                    }
                }
            }
        }
        FormCard.FormDelegateSeparator {
            below: addAccountDelegate
        }

        FormCard.FormButtonDelegate {
            id: addAccountDelegate
            text: i18n("Add Account")
            icon.name: "list-add"
            onClicked: applicationWindow().pageStack.layers.push(Qt.createComponent('org.kde.neochat.login', 'WelcomePage'))
        }
    }

    property Connections connections: Connections {
        target: Controller
        function onConnectionAdded() {
            if (pageStack.layers.depth > 2) {
                pageStack.layers.pop();
            }
        }
    }
}
