// SPDX-FileCopyrightText: 2024 James Graham <james.h.graham@protonmail.com>
// SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only OR LicenseRef-KDE-Accepted-GPL

#pragma once

#include <QAbstractListModel>
#include <QQmlEngine>

#include <Quotient/room.h>

#include "enums/messagecomponenttype.h"
#include "eventhandler.h"
#include "itinerarymodel.h"
#include "linkpreviewer.h"
#include "neochatroom.h"

struct MessageComponent {
    MessageComponentType::Type type;
    QString content;
    QVariantMap attributes;

    int operator==(const MessageComponent &right) const
    {
        return type == right.type && content == right.content && attributes == right.attributes;
    }
};

/**
 * @class MessageContentModel
 *
 * A model to visualise the components of a single RoomMessageEvent.
 */
class MessageContentModel : public QAbstractListModel
{
    Q_OBJECT
    QML_ELEMENT
    QML_UNCREATABLE("")

public:
    /**
     * @brief Defines the model roles.
     */
    enum Roles {
        DisplayRole = Qt::DisplayRole, /**< The display text for the message. */
        ComponentTypeRole, /**< The type of component to visualise the message. */
        ComponentAttributesRole, /**< The attributes of the component. */
        EventIdRole, /**< The matrix event ID of the event. */
        AuthorRole, /**< The author of the event. */
        MediaInfoRole, /**< The media info for the event. */
        FileTransferInfoRole, /**< FileTransferInfo for any downloading files. */
        ItineraryModelRole, /**< The itinerary model for a file. */
        LatitudeRole, /**< Latitude for a location event. */
        LongitudeRole, /**< Longitude for a location event. */
        AssetRole, /**< Type of location event, e.g. self pin of the user location. */
        PollHandlerRole, /**< The PollHandler for the event, if any. */

        IsReplyRole, /**< Is the message a reply to another event. */
        ReplyComponentType, /**< The type of component to visualise the reply message. */
        ReplyEventIdRole, /**< The matrix ID of the message that was replied to. */
        ReplyAuthorRole, /**< The author of the event that was replied to. */
        ReplyDisplayRole, /**< The body of the message that was replied to. */
        ReplyMediaInfoRole, /**< The media info of the message that was replied to. */

        LinkPreviewerRole, /**< The link preview details. */
    };
    Q_ENUM(Roles)

    explicit MessageContentModel(const Quotient::RoomEvent *event, NeoChatRoom *room);

    /**
     * @brief Get the given role value at the given index.
     *
     * @sa QAbstractItemModel::data
     */
    [[nodiscard]] QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const override;

    /**
     * @brief Number of rows in the model.
     *
     * @sa  QAbstractItemModel::rowCount
     */
    [[nodiscard]] int rowCount(const QModelIndex &parent = QModelIndex()) const override;

    /**
     * @brief Returns a mapping from Role enum values to role names.
     *
     * @sa Roles, QAbstractItemModel::roleNames()
     */
    [[nodiscard]] QHash<int, QByteArray> roleNames() const override;

private:
    QPointer<NeoChatRoom> m_room;
    const Quotient::RoomEvent *m_event = nullptr;

    QList<MessageComponent> m_components;
    void updateComponents(bool isEditing = false);

    QPointer<LinkPreviewer> m_linkPreviewer;
    ItineraryModel *m_itineraryModel = nullptr;

    QList<MessageComponent> componentsForType(MessageComponentType::Type type);

    void updateLinkPreviewer();
    void updateItineraryModel();
    bool m_emptyItinerary = false;

    Quotient::FileTransferInfo fileInfo() const;
};
