/*
 *  alarmlistdelegate.cpp  -  handles editing and display of alarm list
 *  Program:  kalarm
 *  SPDX-FileCopyrightText: 2007-2021 David Jarvie <djarvie@kde.org>
 *
 *  SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "alarmlistdelegate.h"

#include "functions.h"
#include "resources/resourcedatamodelbase.h"
#include "resources/eventmodel.h"
#include "kalarm_debug.h"

#include <KAlarmCal/KACalendar>

#include <KColorScheme>

#include <QPainter>


void AlarmListDelegate::paint(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index) const
{
    QStyleOptionViewItem opt = option;
    if (index.isValid())
    {
        if (opt.state & QStyle::State_Selected
        &&  !index.data(ResourceDataModelBase::EnabledRole).toBool())
        {
            // Make the text colour for selected disabled alarms
            // distinguishable from enabled alarms.
            KColorScheme::adjustForeground(opt.palette, KColorScheme::InactiveText, QPalette::HighlightedText, KColorScheme::Selection);
        }
        switch (index.column())
        {
            case AlarmListModel::TimeColumn:
            {
                const QString str = index.data(ResourceDataModelBase::TimeDisplayRole).toString();
                // Need to pad out spacing to align times without leading zeroes
                int i = str.indexOf(QLatin1Char('~'));    // look for indicator of a leading zero to be omitted
                if (i >= 0)
                {
                    painter->save();
                    opt.displayAlignment = Qt::AlignRight;
                    const QVariant value = index.data(Qt::ForegroundRole);
                    if (value.isValid())
                        opt.palette.setColor(QPalette::Text, value.value<QColor>());
                    QRect displayRect;
                    {
                        QString str0 = str;
                        str0[i] = QLatin1Char('0');
                        QRect r = opt.rect;
                        r.setWidth(INT_MAX/256);
                        displayRect = textRectangle(painter, r, opt.font, str0);
                        displayRect = QStyle::alignedRect(opt.direction, opt.displayAlignment,
                                                          displayRect.size().boundedTo(opt.rect.size()),
                                                          opt.rect);
                    }
                    const QString date = str.left(i);
                    const QString time = str.mid(i + 1);
                    if (i > 0)
                    {
                        opt.displayAlignment = Qt::AlignLeft;
                        drawDisplay(painter, opt, displayRect, date);
                        opt.displayAlignment = Qt::AlignRight;
                    }
                    drawDisplay(painter, opt, displayRect, time);
                    painter->restore();
                    return;
                }
                break;
            }
            case AlarmListModel::ColourColumn:
            {
                const KAEvent event = static_cast<const EventListModel*>(index.model())->event(index);
                if (event.isValid()  &&  event.commandError() != KAEvent::CMD_NO_ERROR)
                {
                    opt.font.setBold(true);
                    opt.font.setStyleHint(QFont::Serif);
                    opt.font.setPointSize(opt.rect.height() - 2);
                }
                break;
            }
            default:
                break;
        }
    }
    QItemDelegate::paint(painter, opt, index);
}

QSize AlarmListDelegate::sizeHint(const QStyleOptionViewItem& option, const QModelIndex& index) const
{
    if (index.isValid())
    {
        switch (index.column())
        {
            case AlarmListModel::ColourColumn:
            {
                int h = option.fontMetrics.lineSpacing();
                return {h * 3 / 4, h};
            }
        }
    }
    return QItemDelegate::sizeHint(option, index);
}

void AlarmListDelegate::edit(KAEvent& event, EventListView* view)
{
    KAlarm::editAlarm(event, static_cast<AlarmListView*>(view));   // edit alarm (view-only mode if archived or read-only)
}

// vim: et sw=4:
