/***************************************************************************
                                kstplugin.cpp
                             -------------------
    begin                : May 15 2003
    copyright            : (C) 2003 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "plugincollection.h"
#include "kstplugin.h"
#include "kstdoc.h"
#include "kstdatacollection.h"
#include "kstplugindialog_i.h"
#include <kdebug.h>
#include <kmessagebox.h>
#include <klocale.h>

#include <stdlib.h>


static void countScalarsAndVectors(const QValueList<Plugin::Data::IOValue>& table, unsigned& scalars, unsigned& vectors)
{
  scalars = 0;
  vectors = 0;

  for (QValueList<Plugin::Data::IOValue>::ConstIterator it = table.begin(); it != table.end(); ++it) {
        switch ((*it)._type) {
          case Plugin::Data::IOValue::FloatType:
            scalars++;
            break;
          case Plugin::Data::IOValue::TableType:
            if ((*it)._subType == Plugin::Data::IOValue::FloatSubType) {
              vectors++;
            }
            break;
          default:
            break;
        }
  }
}


KstPlugin::KstPlugin() : KstDataObject() {
  commonConstructor();
}


KstPlugin::KstPlugin(QDomElement &e) : KstDataObject(e) {
  QString pluginName;

  commonConstructor();

  QDomNode n = e.firstChild();
  while(!n.isNull()) {
    QDomElement e = n.toElement();
    if(!e.isNull()) {
      if (e.tagName() == "tag") {
        _tag = e.text();
      } else if (e.tagName() == "name") {
        pluginName = e.text();
      } else if (e.tagName() == "ivector") {
        _inputVectorLoadQueue.append(e.text());
      } else if (e.tagName() == "iscalar") {
        _inputScalarLoadQueue.append(e.text());
      } else if (e.tagName() == "ovector") {
        _outputVectors.append(new KstVector(e.text()));
      } else if (e.tagName() == "oscalar") {
        _outputScalars.append(new KstScalar(e.text()));
      }
    }
    n = n.nextSibling();
  }

  _plugin = PluginCollection::self()->plugin(pluginName);

  if (!_plugin.data()) {
    KMessageBox::sorry(0L, i18n("Unable to load plugin %1 for \"%2\".").arg(pluginName).arg(_tag));
  } else {
    countScalarsAndVectors(_plugin->data()._inputs, _inScalarCnt, _inArrayCnt);
    countScalarsAndVectors(_plugin->data()._outputs, _outScalarCnt, _outArrayCnt);

    for (unsigned i = _outputVectors.count(); i < _outArrayCnt; i++) {
      _outputVectors.append(new KstVector(tagName() + " unnamed vector"));
    }

    for (unsigned i = _outputScalars.count(); i < _outScalarCnt; i++) {
      _outputScalars.append(new KstScalar(tagName() + " unnamed scalar"));
    }
  }
}


void KstPlugin::commonConstructor() {
  _typeString = i18n("Plugin");
  _plugin = 0L;
  _inScalarCnt = 0;
  _outScalarCnt = 0;
  _inArrayCnt = 0;
  _outArrayCnt = 0;
  //kdDebug() << "Creating KSTPlugin: " << long(this) << endl;
}


KstPlugin::~KstPlugin() {
  //kdDebug() << "Destroying KSTPlugin: " << long(this) << endl;
}


KstObject::UpdateType KstPlugin::update(int update_counter) {
  if (!isValid()) {
    return NO_CHANGE;
  }

  if (KstObject::checkUpdateCounter(update_counter))
    return NO_CHANGE;

  int *inArrayLens = 0L, *outArrayLens = 0L;
  double *inScalars = 0L, *outScalars = 0L;
  double **inVectors = 0L, **outVectors = 0L;

  if (_inArrayCnt > 0) {
    inArrayLens = new int[_inArrayCnt];
    inVectors = new double*[_inArrayCnt];
  }

  if (_outArrayCnt > 0) {
    outArrayLens = new int[_outArrayCnt];
    outVectors = new double*[_outArrayCnt];
  }

  if (_inScalarCnt > 0) {
    inScalars = new double[_inScalarCnt];
  }

  if (_outScalarCnt > 0) {
    outScalars = new double[_outScalarCnt];
  }

  for (unsigned i = 0; i < _inArrayCnt; i++) {
    if (!_inputVectors[i].data()) {
      kdWarning() << "Plugin [" << tagName() << "]: Input vector " << i << " is invalid.  Cannot continue." << endl;
      return NO_CHANGE;
    }
    inVectors[i] = _inputVectors[i]->value();
    inArrayLens[i] = _inputVectors[i]->sampleCount();
  }

  for (unsigned i = 0; i < _inScalarCnt; i++) {
    inScalars[i] = _inputScalars[i]->value();
  }

  for (unsigned i = 0; i < _outArrayCnt; i++) {
    if (!_outputVectors[i].data()) {
      kdWarning() << "Plugin [" << tagName() << "]: Output vector " << i << " is invalid.  Cannot continue." << endl;
      return NO_CHANGE;
    }
    outVectors[i] = _outputVectors[i]->value();
    outArrayLens[i] = 1;
  }

  int rc = _plugin->call(inVectors,  inArrayLens,  inScalars,
                         outVectors, outArrayLens, outScalars);

  if (rc != 0) {
    // Error
  }

  for (unsigned i = 0; i < _outArrayCnt; i++) {
    vectorRealloced(_outputVectors[i], outVectors[i], outArrayLens[i]);
  }

  for (unsigned i = 0; i < _outScalarCnt; i++) {
    _outputScalars[i]->setValue(outScalars[i]);
  }

  delete[] inVectors;
  delete[] outVectors;
  delete[] outArrayLens;
  delete[] inArrayLens;
  delete[] outScalars;
  delete[] inScalars;

  return UPDATE;
}


void KstPlugin::save(QTextStream &ts) {
  if (_plugin.data() != 0L) {
    ts << " <plugin>" << endl;
    ts << "   <tag>" << _tag << "</tag>" << endl;
    ts << "   <name>" << _plugin->data()._name << "</name>" << endl;
    for (unsigned i = 0; i < _inputVectors.count(); i++) {
      ts << "   <ivector>" << _inputVectors[i]->tagName()
         << "</ivector>" << endl;
    }
    for (unsigned i = 0; i < _inputScalars.count(); i++) {
      ts << "   <iscalar>" << _inputScalars[i]->tagName()
         << "</iscalar>" << endl;
    }
    for (unsigned i = 0; i < _outputVectors.count(); i++) {
      ts << "   <ovector>" << _outputVectors[i]->tagName()
         << "</ovector>" << endl;
    }
    for (unsigned i = 0; i < _outputScalars.count(); i++) {
      ts << "   <oscalar>" << _outputScalars[i]->tagName()
         << "</oscalar>" << endl;
    }
    ts << " </plugin>" << endl;
  }
}


bool KstPlugin::slaveVectorsUsed() const {
  return true;
}


bool KstPlugin::isValid() const {
  return _inputVectors.count() == _inArrayCnt &&
         _inputScalars.count() == _inScalarCnt &&
         _plugin.data() != 0L;
}


QString KstPlugin::propertyString() const {
  if (!isValid()) {
    return i18n("Invalid plugin.");
  }
  return plugin()->data()._name;
}


bool KstPlugin::setPlugin(KSharedPtr<Plugin> plugin) {
  if (plugin == _plugin) {
    return true;
  }

  if (!plugin.data()) {
    _inputVectors.clear();
    _inputScalars.clear();
    _outputVectors.clear();
    _outputScalars.clear();
    _plugin = 0L;
    return true;
  }

  countScalarsAndVectors(plugin->data()._inputs, _inScalarCnt, _inArrayCnt);

  if (_inputVectors.count() != _inArrayCnt ||
      _inputScalars.count() != _inScalarCnt) {
    _plugin = 0L;
    return false;
  }

  _outScalarCnt = 0;
  _outArrayCnt = 0;
  _outputVectors.clear();
  _outputScalars.clear();

  const QValueList<Plugin::Data::IOValue>& otable = plugin->data()._outputs;
  for (QValueList<Plugin::Data::IOValue>::ConstIterator it = otable.begin();
                                                         it != otable.end();
                                                                        ++it) {
    if ((*it)._type == Plugin::Data::IOValue::TableType) {
      _outputVectors.append(new KstVector);
      _outArrayCnt++;
    } else if ((*it)._type == Plugin::Data::IOValue::FloatType) {
      _outputScalars.append(new KstScalar);
      _outScalarCnt++;
    }
  }

  _plugin = plugin;
  return true;
}


KSharedPtr<Plugin> KstPlugin::plugin() const {
  return _plugin;
}


void KstPlugin::_showDialog() {
  KstPluginDialogI::globalInstance()->show_I(tagName());
}

// vim: ts=2 sw=2 et
