/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <errno.h>

#include <qglobal.h>
#include <qcstring.h>

#include "modem.h"



#ifdef __linux__
#	define LOCK_DIR "/var/lock"
#else
#	ifdef BSD
#		define LOCK_DIR "/var/spool/lock"
#	else
#		define LOCK_DIR "/var/spool/locks"
#	endif
#endif



#ifndef CSOH
#define CSOH  01
#endif

#ifndef CSTX
#define CSTX  02
#endif

#ifndef CEOT
#define CEOT  04
#endif

#ifndef CACK
#define CACK  06
#endif

#ifndef CNAK
#define CNAK 025
#endif

#ifndef CCAN
#define CCAN 030
#endif

#ifndef CSUSP
#define CSUSP 032
#endif


#define XMOD_NONE             0

#define XMOD_REC_SENTC1       1
#define XMOD_REC_SENTC2       2
#define XMOD_REC_SENTC3       3
#define XMOD_REC_SENTC4       4
#define XMOD_REC_SENTNAK1     5
#define XMOD_REC_SENTNAK2     6
#define XMOD_REC_SENTNAK3     7
#define XMOD_REC_SENTNAK4     8
#define XMOD_REC_SENTNAK5     9
#define XMOD_REC_SENTNAK6    10
#define XMOD_REC_WAIT4BLOCK  11
#define XMOD_REC_WAIT4CBLOCK 12
#define XMOD_REC_DATA        13
#define XMOD_REC_WAIT4CRCHI  14
#define XMOD_REC_WAIT4CRCLO  15

#define XMOD_SEND_WAIT4REC   16
#define XMOD_SEND_SENTEOT    17


const char *modem_ser_devices[] = {
#ifdef __linux__
	"/dev/modem",
	"/dev/ttyS0",
	"/dev/ttyS1",
	"/dev/ttyS2",
	"/dev/ttyS3"
#elif defined(__FreeBSD__)
	"/dev/cuaa0",
	"/dev/cuaa1",
	"/dev/cuaa2",
	"/dev/cuaa3"
#elif defined(__NetBSD__)
	"/dev/tty00",		/* "normal" modem lines */
	"/dev/tty01",
	"/dev/tty02",
	"/dev/tty03",
	"/dev/dty00",		/* Dial out devices */
	"/dev/dty01",
	"/dev/dty02",
	"/dev/dty03"
#elif defined(__svr4__)
	"/dev/cua/a",
	"/dev/cua/b",
	"/dev/ttya",
	"/dev/ttyb"
#else
	"/dev/ttyS0",
	"/dev/ttyS1"
#endif
	};


const char *modem_usb_devices[] = {
#ifdef __linux__
	"/dev/usb/ttyACM0",
	"/dev/usb/ttyACM1",
	"/dev/usb/ttyACM2",
	"/dev/usb/ttyACM3"
#elif defined(__FreeBSD__)
	"/dev/umodem0",
	"/dev/umodem1"
#elif defined(__NetBSD__)
	"/dev/umodem0",
	"/dev/umodem1"
#elif defined(__svr4__)
	"/dev/umodem0",		/* Don't know, if this is right */
	"/dev/umodem1"
#else
	"/dev/umodem0",
	"/dev/umodem1"
#endif
	};



Modem::Modem(QObject *parent, const char *name): QObject(parent, name)
{
	timer = new QTimer(this, "modemtimer");
	Q_CHECK_PTR(timer);
	connect(timer, SIGNAL(timeout()), SLOT(timerDone()));

	init();
	xinit();
}


Modem::~Modem()
{
	close();
}


void Modem::setDevice(bool usb, int device)
{
	if (usb) {
		if (device >= MODEM_USB_DEVNUM)
			device = 0;
		fdev = modem_usb_devices[device];
	} else {
		if (device >= MODEM_SER_DEVNUM)
			device = 0;
		fdev = modem_ser_devices[device];
	}
}


void Modem::setSpeed(int speed)
{
	switch (speed) {
		case 0:
			m_speed = 0;
			break;
		case 300:
			m_speed = B300;
			break;
		case 600:
			m_speed = B600;
			break;
		case 1200:
			m_speed = B1200;
			break;
		case 2400:
			m_speed = B2400;
			break;
		case 4800:
			m_speed = B4800;
			break;
		case 9600:
			m_speed = B9600;
			break;
		case 19200:
			m_speed = B19200;
			break;
		case 38400:
			m_speed = B38400;
			break;
#ifdef B57600
		case 57600:
			m_speed = B57600;
			break;
#endif
#ifdef B115200
		case 115200:
			m_speed = B115200;
			break;
#endif
#ifdef B230400
		case 230400:
			m_speed = B230400;
			break;
#endif
		default:
#ifdef MODEM_DEBUG
			qDebug("Modem: setSpeed(): fallback to default speed.");
#endif
			m_speed = B38400;
	}
}


void Modem::setLineParams(int data, char parity, int stop)
{
	m_data = data;
	m_parity = parity;
	m_stop = stop;
}


bool Modem::open()
{
	struct termios tty;

	close();

	if (!lockDevice()) {
#ifdef MODEM_DEBUG
		qDebug("Modem: open(): Cannot lock device.");
#endif
		return false;
	}

	if ((fd = ::open(fdev, O_RDWR | O_NOCTTY | O_NONBLOCK | O_NDELAY)) == -1) {
#ifdef MODEM_DEBUG
		qDebug("Modem: open(): Cannot open device.");
#endif
		return false;
	}
	tcdrain(fd);
	tcflush(fd, TCIOFLUSH);
	if (tcgetattr(fd, &init_tty) == -1) {
#ifdef MODEM_DEBUG
		qDebug("Modem: open(): tcgetattr() failed.");
#endif
		::close(fd);
		fd = 0;
		return false;
	}
	
	tty = init_tty;
	tty.c_cflag |= CLOCAL | CREAD;
	tty.c_cflag &= ~(PARENB | CSTOPB | CSIZE);
	switch (m_data) {
		case 5:
			tty.c_cflag |= CS5;
			break;
		case 6:
			tty.c_cflag |= CS6;
			break;
		case 7:
			tty.c_cflag |= CS7;
			break;
		default:
			tty.c_cflag |= CS8;
	}
	if (m_parity == 'E')
		tty.c_cflag |= PARENB;
	else if (m_parity == 'O')
		tty.c_cflag |= PARENB | PARODD;
	if (m_stop == 2)
		tty.c_cflag |= CSTOPB;
#ifdef CRTSCTS
	tty.c_cflag |= CRTSCTS;
#endif
  tty.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG); // Raw input
  tty.c_oflag &= ~OPOST;													// Raw output
	tty.c_iflag = IGNBRK;
	if (m_parity == 'N')
		tty.c_iflag |= IGNPAR;
	else
		tty.c_iflag |= (INPCK | ISTRIP);
	tty.c_cc[VMIN] = 0;
	tty.c_cc[VTIME] = 0;
/*
	memset(&tty, 0, sizeof(tty));
	tty.c_cflag = CS8 | CREAD | CLOCAL | CRTSCTS;
	tty.c_iflag = IGNPAR | IGNBRK;
*/
	if (m_speed) {
		cfsetospeed(&tty, m_speed);
		cfsetispeed(&tty, m_speed);
	}
	tcdrain(fd);
	if (tcsetattr(fd, TCSANOW, &tty) == -1) {
#ifdef MODEM_DEBUG
		qDebug("Modem: open(): tcsetattr() failed.");
#endif
		::close(fd);
		fd = 0;
		return false;
	}

	sn = new QSocketNotifier(fd, QSocketNotifier::Read, this, "modemsocketnotifier");
	Q_CHECK_PTR(sn);
	connect(sn, SIGNAL(activated(int)), SLOT(readChar(int)));

	return true;
}


void Modem::close()
{
	timer->stop();

	if (xstate != XMOD_NONE)
		abortXModem();

	delete sn;
	sn = 0;

	if (fd) {
		tcflush(fd, TCIOFLUSH);
		tcsetattr(fd, TCSANOW, &init_tty);
		::close(fd);
		fd = 0;
	}

	unlockDevice();
}


void Modem::flush()
{
	timerStop();
	if (fd) {
		char buf[128];
		while (read(fd, (void *)buf, 128) > 0) {
			tcflush(fd, TCIOFLUSH);
			usleep(100000);
		}
		bufpos = 0;
	}
}


bool Modem::lockDevice()
{
	char fname[1024];
	char content[256];
	ssize_t count;
	pid_t pid;
	int lfd;
	struct passwd *pw;

	if (is_locked)
		return true;

	sprintf(fname, "%s/LCK..%s", LOCK_DIR, strrchr(fdev, '/') + 1);
	if (!access(fname, F_OK)) {
		if ((lfd = ::open(fname, O_RDONLY)) < 0) {
#ifdef MODEM_DEBUG
			qDebug("Modem: lockDevice(): Cannot open existing lock file.");
#endif
			return false;
		}

		count = read(lfd, content, 79);
		if (count < 0) {
#ifdef MODEM_DEBUG
			qDebug("Modem: lockDevice(): Cannot read existing lock file.");
#endif
			::close(lfd);
			return false;
		}
		content[count] = 0;
		::close(lfd);

		count = sscanf(content, "%d", &pid);
		if ((count != 1) || (pid <= 0)) {
#ifdef MODEM_DEBUG
			qDebug("Modem: lockDevice(): Cannot get PID out of existing lock file.");
#endif
			return false;
		}

		if (!kill((pid_t)pid, 0)) {
#ifdef MODEM_DEBUG
			qDebug("Modem: lockDevice(): Process of existing lock file is still running.");
#endif
			return false;
		}

		if (errno != ESRCH) {
#ifdef MODEM_DEBUG
			qDebug("Modem: lockDevice(): Cannot emit signal to PID of existing lock file.");
#endif
			return false;
		}
	}
	
	if ((lfd = creat(fname, 0644)) == -1) {
#ifdef MODEM_DEBUG
		qDebug("Modem: lockDevice(): Cannot create lock file.");
#endif
		return false;
	}

	pid = (int)getpid();
	pw = getpwuid(getuid());
	sprintf(content, "%08d %s %s", pid, "kmlofax", pw->pw_name);
	write(lfd, content, qstrlen(content));
	::close(lfd);

	is_locked = true;

	return true;
}


void Modem::unlockDevice()
{
	char fname[1024];

	if (is_locked) {
		sprintf(fname, "%s/LCK..%s", LOCK_DIR, strrchr(fdev, '/') + 1);
		unlink(fname);
		is_locked = false;
	}
}


bool Modem::dsrOn()
{
	int flags;

	if (!fd) {
#ifdef MODEM_DEBUG
		qDebug("Modem: dsrOn(): File not open.");
#endif
		return false;
	}

	if (ioctl(fd, TIOCMGET, &flags) == -1) {
#ifdef MODEM_DEBUG
		qDebug("Modem: dsrOn(): ioctl() failed.");
#endif
		return false;
	}

	return (flags & TIOCM_DSR) != 0;
}


bool Modem::ctsOn()
{
	int flags;

	if (!fd) {
#ifdef MODEM_DEBUG
		qDebug("Modem: ctsOn(): File not open.");
#endif
		return false;
	}

	if (ioctl(fd, TIOCMGET, &flags) == -1) {
#ifdef MODEM_DEBUG
		qDebug("Modem: ctsOn(): ioctl() failed.");
#endif
		return false;
	}

	return (flags & TIOCM_CTS) != 0;
}


void Modem::writeChar(const char c)
{
	int result;

	do {
		result = write(fd, (const void *)&c, 1);
		if ((result < 0) && (errno != EAGAIN))
			return;
	} while (result != 1);
}


void Modem::writeChar(const char c, int msec)
{
	writeChar(c);
	timerStart(msec);
}


void Modem::writeLine(const QCString &line)
{
	writeBlock(line.data(), line.length());
	writeChar('\r');
}


void Modem::writeLine(const QCString &line, int msec)
{
	writeLine(line);
	timerStart(msec);
}


void Modem::writeBlock(const char *block, uint length)
{
	int result;
	uint count = 0;
	size_t bs;

	usleep(10000);
	while (count < length) {
		bs = (size_t)(length-count);
		if (bs > 128)
			bs = 128;
		result = write(fd, (const void *)&block[count], bs);
		if (result < 0) {
			if (errno != EAGAIN)
				return;
		} else
			count += (uint)result;
	}
}


void Modem::writeBlock(const char *block, uint length, int msec)
{
	writeBlock(block, length);
	timerStart(msec);
}


void Modem::receiveXModem(const QString &fname, int size, bool crc)
{
	xfile.setName(fname);
	if (!xfile.open(IO_ReadWrite | IO_Truncate))
		return;
  xfile_size = size;
		
	disconnect(sn, 0, this, 0);
	connect(sn, SIGNAL(activated(int)), SLOT(readXChar(int)));

	xcrc = crc;

	if (xcrc) {
		writeChar('C', 3000);
		xstate = XMOD_REC_SENTC1;
	} else {
		writeChar(CNAK, 10000);
		xstate = XMOD_REC_SENTNAK1;
	}
	xblock_no = 1;
}


void Modem::sendXModem(const QString &fname)
{
	xfile.setName(fname);
	if (!xfile.open(IO_ReadOnly))
		return;
  xfile_size = xfile.size();
			
	disconnect(sn, 0, this, 0);
	connect(sn, SIGNAL(activated(int)), SLOT(readXChar(int)));

	xcrc = false;
	xblock_size = 0;

	xstate = XMOD_SEND_WAIT4REC;
	timerStart(10000);

	xblock_no = 1;
}


void Modem::abortXModem()
{
	int i;

	timerStop();
	for (i = 0; i < 8; i++)
		writeChar(CCAN);
	usleep(250000);
	flush();
	writeLine("");
	usleep(100000);
	flush();
	
	xinit();
	emit xmodemDone(false);
}


void Modem::timerDone()
{
#ifdef MODEM_DEBUG
	qDebug("Modem: timeout, xstate = %d.", xstate);
#endif
	switch (xstate) {
		case XMOD_NONE:							// non-XModem mode
			emit timeout();
			break;

		case XMOD_REC_SENTC1:				// 1st 'C' sent
		case XMOD_REC_SENTC2:				// 2nd 'C' sent
		case XMOD_REC_SENTC3:				// 3rd 'C' sent
			writeChar('C', 1000);			// Should be 3000 in original XModem
			xstate++;
			break;

		case XMOD_REC_SENTC4:				// 4th 'C' sent
			xcrc = false;

		case XMOD_REC_SENTNAK1:			// 1st <NAK> sent
		case XMOD_REC_SENTNAK2:			// 2nd <NAK> sent
		case XMOD_REC_SENTNAK3:			// 3rd <NAK> sent
		case XMOD_REC_SENTNAK4:			// 4th <NAK> sent
		case XMOD_REC_SENTNAK5:			// 5th <NAK> sent
			writeChar(CNAK, 1000);		// Should be 10000 in original XModem
			xstate++;
			break;

		case XMOD_REC_SENTNAK6:			// 6th <NAK> sent
			xinit();
			emit xmodemDone(false);
			break;

		case XMOD_REC_WAIT4BLOCK:		// pending XModem block
		case XMOD_REC_WAIT4CBLOCK:
		case XMOD_REC_DATA:
		case XMOD_REC_WAIT4CRCHI:
		case XMOD_REC_WAIT4CRCLO:
			writeChar(CNAK, 1000);		// Should be 10000 in original XModem
			xstate = XMOD_REC_SENTNAK1;
			break;
		
		case XMOD_SEND_WAIT4REC:
			xinit();
			emit xmodemDone(false);
			break;
		case XMOD_SEND_SENTEOT:
			writeChar(CEOT);
			xinit();
			emit xmodemDone(true);
			break;

		default:
			break;
	}
}


void Modem::readChar(int)
{
	uchar c;

	while (read(fd, (void *)&c, 1) == 1) {
		if (c == '\n') {
			buffer[bufpos] = 0;
			bufpos = 0;
			emit gotLine((const char *)buffer);
			break;
		}	else if ((bufpos < 1024) && (c != '\r'))
			buffer[bufpos++] = c;
	}
}


void Modem::readXChar(int)
{
	uchar c;
	static uchar crc_hi, block, cblock;
	ushort crc;
	int diff;
	
	while (read(fd, (void *)&c, 1) == 1) {
		switch (xstate) {
			case XMOD_REC_SENTC1:						// 1st 'C' sent
			case XMOD_REC_SENTC2:						// 2nd 'C' sent
			case XMOD_REC_SENTC3:						// 3rd 'C' sent
			case XMOD_REC_SENTC4:						// 4th 'C' sent
			case XMOD_REC_SENTNAK1:					// 1st <NAK> sent
			case XMOD_REC_SENTNAK2:					// 2nd <NAK> sent
			case XMOD_REC_SENTNAK3:					// 3rd <NAK> sent
			case XMOD_REC_SENTNAK4:					// 4th <NAK> sent
			case XMOD_REC_SENTNAK5:					// 5th <NAK> sent
			case XMOD_REC_SENTNAK6:					// 6th <NAK> sent
				if (c == CSOH) {
					timerStart(1000);
					xblock_size = 128;
					xstate = XMOD_REC_WAIT4BLOCK;
				} else if (c == CSTX) {
					timerStart(1000);
					xblock_size = 1024;
					xstate = XMOD_REC_WAIT4BLOCK;
				} else if (c == CEOT) {
					timerStop();
					xfile.close();
					writeChar(CACK);
					xinit();
					emit xmodemDone(true);
				} else
					timerStart(1000);
				break;

			case XMOD_REC_WAIT4BLOCK:				// <SOH> or <STX> received
				timerStart(1000);
				block = c;
				xstate = XMOD_REC_WAIT4CBLOCK;
				break;

			case XMOD_REC_WAIT4CBLOCK:			// block number received
				timerStart(1000);
				cblock = c;
				xstate = XMOD_REC_DATA;
				bufpos = 0;
				break;

			case XMOD_REC_DATA:							// complement block number received
				timerStart(1000);
				buffer[bufpos++] = c;
				if (bufpos == xblock_size) {
					bufpos = 0;
					xstate = xcrc ? XMOD_REC_WAIT4CRCHI : XMOD_REC_WAIT4CRCLO;
				}
				break;

			case XMOD_REC_WAIT4CRCHI:				// data block received
				timerStart(1000);
				crc_hi = c;
				xstate = XMOD_REC_WAIT4CRCLO;
				break;

			case XMOD_REC_WAIT4CRCLO:				// crc high-byte received
				timerStart(10000);
				xstate = XMOD_REC_SENTC4;
				if ((uchar)(block ^ cblock) != 0xff) {
					writeChar(CNAK);
					break;
				}
				if (block+1 == xblock_no) {
					writeChar(CACK);
					break;
				}
				if (block != xblock_no) {
					abortXModem();
					break;
				}
				if (xcrc) {
					if (((ushort)crc_hi << 8 | (ushort)c) != calcCRC()) {
						writeChar(CNAK);
						break;
					}
				} else {
					if (c != calcChecksum()) {
						writeChar(CNAK);
						break;
					}
				}
				writeChar(CACK);
				if (xfile_size != 0) {
					diff = xfile_size - xfile.at();
					if (diff > xblock_size)
						diff = xblock_size;
				} else
					diff =  xblock_size;
				xfile.writeBlock((char *)buffer, (uint)diff);
				emit xmodemStatus(xfile.at());
				xblock_no++;
				break;

			case XMOD_SEND_WAIT4REC:				// Waiting for receiver
				if (c == 'C')
					xcrc = true;
				else if (c == CACK) {
#ifdef MODEM_DEBUG
					qDebug("Modem: Xmodem - ACK for block %d.", xblock_no);
#endif
					xblock_no++;
					emit xmodemStatus(xfile.at());
				} else if (c == CNAK) {
#ifdef MODEM_DEBUG
					qDebug("Modem: Xmodem - NAK for block %d.", xblock_no);
#endif
					xfile.at(xfile.at() - xblock_size);
				} else {
					timerStart(10000);
					break;
				}
				if (xfile.atEnd()) {
					xfile.close();
					writeChar(CEOT, 1000);
					xstate = XMOD_SEND_SENTEOT;
					break;
				}
				
				timerStart(10000);
				
				if (xfile.size() - xfile.at() > 896) {
					xblock_size = 1024;
					writeChar(CSTX);
				} else {
					xblock_size = 128;
					writeChar(CSOH);
				}
#ifdef MODEM_DEBUG
				qDebug("Modem: Xmodem - Sending block %d.", xblock_no);
#endif
				writeChar(xblock_no);
				writeChar(~xblock_no);
				memset((void *)buffer, CSUSP, xblock_size);
				xfile.readBlock((char *)buffer, (uint)xblock_size);
				writeBlock((const char *)buffer, xblock_size);
				if (xcrc) {
					crc = calcCRC();
					writeChar((uchar)(crc >> 8));
					writeChar((uchar)(crc & 0xff));
				} else
					writeChar(calcChecksum());
				
				break;
			case XMOD_SEND_SENTEOT:				// EOT sent
				if (c == CACK) {
					timerStop();
					xinit();
					emit xmodemDone(true);
				}
				break;
			default:
				break;
		}
	}
}


void Modem::init()
{
	is_locked = false;

	m_speed = B38400;
	m_data = 8;
	m_parity = 'N';
	m_stop = 1;

	fdev = 0;
	fd = 0;
	sn = 0;

	bufpos = 0;
}


void Modem::xinit()
{
	bufpos = 0;

	xstate = XMOD_NONE;
	if (xfile.isOpen())
		xfile.close();
	xfile_size = 0;
	xcrc = false;
	xblock_no = 0;
	xblock_size = 0;

	if (sn) {
		disconnect(sn, 0, this, 0);
		connect(sn, SIGNAL(activated(int)), SLOT(readChar(int)));
	}
}


uchar Modem::calcChecksum()
{
	int i;
	uchar c = 0;

	for (i=0; i < xblock_size; i++)
		c += buffer[i];

	return c;
}


ushort Modem::calcCRC()
{
	int i, j;
	ushort c = 0;
	
	for (i=0; i < xblock_size; i++) {
		c ^= (ushort)buffer[i] << 8;
		for (j=0; j < 8; j++)
			if (c & 0x8000)
				c = c << 1 ^ 0x1021;
			else
				c <<= 1;
	}

	return c;
}
