/***************************************************************************
                          kpldoc.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <qdir.h>
#include <klocale.h>
#include <kglobal.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include "kpldoc.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "arrayitem.h"
#include "funitem.h"
#include "legenditem.h"
#include "utils.h"

KplDoc::KplDoc(QObject* parent) :
 QObject(parent), fType(Unknown), chisq(0.0)
{
  memset(pLast, 0, sizeof(pLast));
  memset(pErr, 0, sizeof(pErr));
  itd.setAutoDelete(true);
  itb.setAutoDelete(true);
}

KplDoc::~KplDoc()
{
  freeAllItems();
  itb.clear();
}

void KplDoc::newDocument()
{
  docURL.setFileName(i18n("Untitled"));
  freeAllItems();
  aut.nyLeg = 0;
}

bool KplDoc::saveDocument(const KURL& url, bool abs)
{
  FILE* f;
  if (url.isLocalFile())
    f = fopen(url.path(), "w");
  else
    f = fopen(QDir::homeDirPath() + "/kpltmp.tmp", "w");
  if (f) {
    fprintf(f, "[Items]\nnitems=%i\n", itd.count());
    for (unsigned int i = 0; i < itd.count(); i++) {
      fprintf(f, "\n[Item %i]\n", i);
      itd.at(i)->writePlo(f, url, abs);
    }
    fclose(f);
    if (!url.isLocalFile()) {
      job = KIO::file_copy(KURL(QDir::homeDirPath() + "/kpltmp.tmp"), url, -1,
                           true);
      connectIOfinished();
    }
    setModified(false);
    fType = Plot;
  } else
    KMessageBox::sorry((QWidget *) parent(),
                       i18n("Writing to this file not possible."));
  return (f);
}

void KplDoc::setModified(bool m)
{
  modified = m;
}

bool KplDoc::isModified() const
{
  return modified;
}

void KplDoc::setCurrentDir(const KURL& url)
{
  QFileInfo fi(url.path());
  KURL u = url;
  u.setPath(fi.dirPath(true));
  m_dir = u.url(+1);
}

const QString& KplDoc::currentDir() const
{
  return m_dir;
}

void KplDoc::setURL(const KURL& url)
{
  docURL = url;
  setCurrentDir(url);
  if (url.isLocalFile()) {
    QFileInfo fi(url.path());
    m_time = fi.lastModified();
  } else
    m_time = QDateTime();
}

const KURL& KplDoc::URL() const
{
  return docURL;
}

const QDateTime& KplDoc::URLTime() const
{
  return m_time;
}

void KplDoc::freeAllItems()
{
  itd.clear();
  setModified(false);
}

void KplDoc::newItem(KplItem::ItemTypes ityp)
{
  switch (ityp) {
    case KplItem::Frame:
      itd.append(new FrameItem(&aut));
      break;
    case KplItem::Array:
      itd.append(new ArrayItem(&aut));
      break;
    case KplItem::Function:
      itd.append(new FunItem(&aut));
      break;
    case KplItem::ParFunction:
      itd.append(new ParFunItem(&aut));
      break;
    case KplItem::Legend:
      itd.append(new LegendItem(&aut));
  }
}

bool KplDoc::autoFile(double* xmi, double* xma, double* tic, int* mtic,
                      int* ndig, int* ie, double* fn, const double* x, int n,
                      double c, bool log)
{
  double xmin, xmax;
  if (Utils::minMaxFile(&xmin, &xmax, x, n))
    return true;
  if (aut.autoNorm)
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), ie, fn);
  else {
    *ie = 0;
    *fn = 1.0;
  }
  return Utils::autoSc(xmi, xma, tic, mtic, ndig, xmin, xmax, *fn, c, log);
}

bool KplDoc::autoScale(FrameItem* fd, ArrayItem* ad)
{
  if ((ad->x) && (aut.ixAuto < ad->ncols) && (aut.iyAuto < ad->ncols)) {
    if (autoFile(&fd->xmi, &fd->xma, &fd->xtic, &fd->mticx, &fd->ndigx,
                 &fd->iex, &ad->fx, ad->x[aut.ixAuto], ad->nrows,
                 0.2, aut.autoLogX))
      return true;
    if (autoFile(&fd->ymi, &fd->yma, &fd->ytic, &fd->mticy, &fd->ndigy,
                 &fd->iey, &ad->fy, ad->x[aut.iyAuto], ad->nrows,
                 0.2 * aut.xlAuto / aut.ylAuto, aut.autoLogY))
      return true;
    if (aut.autohPath)
      fd->sh = ad->url.isLocalFile() ? ad->url.path() : ad->url.url();
    int minCol = ad->ncols - 1;
    ad->ix = QMIN(aut.ixAuto, minCol);
    ad->iy = QMIN(aut.iyAuto, minCol);
    ad->ie = QMIN(aut.ieAuto, minCol);
    ad->symb = aut.autoSymb;
    ad->istart = 0;
    ad->n = ad->nrows;
    ad->errbars = aut.autoErr & (ad->ie < ad->ncols);
    return false;
  }
  return true;
}

void KplDoc::autoScaleFrame(bool autoNorm, int item, int* iex, int* iey,
                            double* fx, double* fy, double* xmi, double* xma,
                            double* ymi, double* yma, double* xtic, double* ytic,
                            int* mticx, int* mticy, int* ndigx, int* ndigy,
                            double c, bool logx, bool logy)
{
  if (autoNorm) {
    double fxt, fyt;
    int iext, ieyt;
    *iex = *iey = 999;
    for (KplItem* itm = itd.at(item + 1); itm; itm = itd.next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      iext = *iex;
      ieyt = *iey;
      itm->expoItem(&iext, &ieyt, &fxt, &fyt);
      if (iext < *iex) {
        *iex = iext;
        *fx = fxt;
      }
      if (ieyt < *iey) {
        *iey = ieyt;
        *fy = fyt;
      }
    }
  } else {
    *iex = *iey = 0;
    *fx = *fy = 1.0;
  }
  if ((*iex != 999) && (*iey != 999)) {
    double xmin = 1.0e300;
    double ymin = xmin;
    double xmax = -xmin;
    double ymax = xmax;
    for (KplItem *itm = itd.at(item + 1); itm; itm = itd.next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      *xmi = xmin;
      *ymi = ymin;
      *xma = xmax;
      *yma = ymax;
      itm->minMax(xmi, xma, ymi, yma);
      xmin = QMIN(xmin, *xmi);
      xmax = QMAX(xmax, *xma);
      ymin = QMIN(ymin, *ymi);
      ymax = QMAX(ymax, *yma);
    }
    Utils::autoSc(xmi, xma, xtic, mticx, ndigx, xmin, xmax, *fx, 0.2, logx);
    Utils::autoSc(ymi, yma, ytic, mticy, ndigy, ymin, ymax, *fy, 0.2 * c, logy);
  }
}

void KplDoc::autoFit(ArrayItem* ad, FunItem* fd)
{
  fd->pathy = aut.fitPath;
  fd->namey = aut.fitName;
  memcpy(fd->py, aut.pFit, KPL_NPMAX * sizeof(double));
  fd->symb = aut.fitSymb;
  fd->color = aut.fitColor;
  fd->tmin = ad->x[ad->ix][ad->istart];
  fd->tmax = ad->x[ad->ix][ad->istart + ad->n - 1];
  fd->fx = ad->fx;
  fd->fy = ad->fy;
  if (Utils::getFuncAddr(fd->pathy.path(), fd->namey, &fd->hmody, &fd->fkty))
    return;
  int mode = 0;
  if (aut.fitShowDlg)
    mode |= FitDlg::ShowDlg;
  if (aut.fitSavePar)
    mode |= FitDlg::SavePar;
  if (aut.fitFollow)
    mode |= FitDlg::Follow;
  QList<ArrayItem> arr;
  arr.append(ad);
  QList<FunItem> fun;
  fun.append(fd);
  KConfig* config = KGlobal::config();
  config->setGroup("DataErrors0");
  config->writeEntry("FitErrorColumn", aut.err.fitErrCol);
  if (!aut.err.fitErrCol) {
    config->writeEntry("ErrorModelPath", aut.err.errModPath);
    config->writeEntry("ErrorModelName", aut.err.errModName);
    QStrList s1;
    for (int i = 0; i < KPL_NPMAX; i++)
      s1.insert(i, QString::number(aut.err.pErrMod[i], aut.format, aut.prec));
    config->writeEntry("ErrorModelParam", s1, ' ');
  }
  FitDlg dlg((QWidget*) parent(), this, &arr, &fun, mode);
  dlg.slotFit();
  if (aut.fitShowDlg) {
    backupItems();
    dlg.exec();
  } else
    dlg.slotApply();
  return;
}

bool KplDoc::openDocument(const KURL& url, FileTypes type)
{
  fType = type;
  if (fType == Unknown) {
    QString s = url.fileName();
    fType = ((s.length() - s.findRev(".plo", -1, false)) == 4) ? Plot : Data;
  }
  bool success;
  if (fType == Data) {
    freeAllItems();
    newItem(KplItem::Frame);
    newItem(KplItem::Array);
    if (aut.autoFit)
      newItem(KplItem::Function);
    ((ArrayItem *) itd.at(1))->url = url;
    int nrows = ((ArrayItem *) itd.at(1))->readFile();
    success = nrows;
    if (success) {
      if (autoScale((FrameItem *) itd.at(0), (ArrayItem *) itd.at(1)))
        success = false;
      else {
        setURL(url);
        if (aut.autoFit)
          autoFit((ArrayItem *) itd.at(1), (FunItem *) itd.at(2));
        setModified();
      }
    }
  } else {
    QString tmpFile;
    success = KIO::NetAccess::download(url, tmpFile);
    if (success) {
      QFileInfo f2(tmpFile);
      success = f2.isReadable();
      if (success) {
        newDocument();
        KSimpleConfig plo(tmpFile, true);
        plo.setGroup("Items");
        if (int nItems = QMAX(plo.readNumEntry("nitems"), 0)) {
          const char *itemName[] = {"", "FRAMEITEM", "ARRAYITEM", "FUNITEM",
                                    "PARFUNITEM", "LEGENDITEM"};
          setURL(url);
          for (int i = 0; i < nItems; i++) {
            QString s;
            s.sprintf("Item %i", i);
            plo.setGroup(s);
            s = plo.readEntry("Type");
            int iType;
            for (iType = KplItem::Frame; iType <= KplItem::Legend; iType++)
              if (s == itemName[iType])
                break;
            switch (iType) {
              case KplItem::Frame:
                itd.append(new FrameItem(&plo, &aut));
                break;
              case KplItem::Array:
                itd.append(new ArrayItem(&plo, &aut, url));
                break;
              case KplItem::Function:
                itd.append(new FunItem(&plo, &aut, url));
                break;
              case KplItem::ParFunction:
                itd.append(new ParFunItem(&plo, &aut, url));
                break;
              case KplItem::Legend:
                itd.append(new LegendItem(&plo, &aut));
            }
          }
        } else
          success = false;
      }
      KIO::NetAccess::removeTempFile(tmpFile);
    }
  }
  if (success)
    backupItems();
  else {
    restoreItems();
    KMessageBox::error((QWidget *) parent(),
                       QString(i18n("while loading the file\n")) + url.path());
  }
  return success;
}

void KplDoc::connectIOfinished()
{
  connect(job, SIGNAL(result(KIO::Job *)), this, SLOT(slotIOfinished()));
}

bool KplDoc::restoreItems(bool undo)
{
  QList<KplItem>* itc = 0;
  if (undo) {
    if (undoAllowed())
      itc = itb.prev();
  } else
    if (redoAllowed())
      itc = itb.next();
  if (itc) {
    itd.clear();
    for (unsigned int i = 0; i < itc->count(); i++)
    itd.append(itc->at(i)->copy());
    emit modelChanged(true, true);
    return true;
  } else
    return false;
}

bool KplDoc::undoAllowed() const
{
  if (!itb.current())
    return false;
  return (itb.current() != itb.getFirst());
}

bool KplDoc::redoAllowed() const
{
  if (!itb.current())
    return false;
  return (itb.current() != itb.getLast());
}

void KplDoc::slotIOfinished()
{
  if (job->error()) {
    job->showErrorDialog();
    QFileInfo fi(job->destURL().path());
    if (fi.extension().lower() == "plo") {
      setModified();
      emit modelChanged(false, false);
    }
  }
  remove(QDir::homeDirPath() + "/kpltmp.tmp");
}

void KplDoc::backupItems(bool list)
{
  QList<KplItem>* itc = itb.current();
  if (itc) {
    if (itc == itb.getLast()) {
      if (itb.count() >= 20)
        itb.removeFirst();
    } else
      do
        itb.removeLast();
      while (itc != itb.current());
  }
  itb.append(new QList<KplItem>);
  for (unsigned int i = 0; i < itd.count(); i++)
    itb.current()->append(itd.at(i)->copy());
  emit modelChanged(true, list);
}

void KplDoc::deleteItem(int i)
{
  itd.remove(i);
  setModified();
  backupItems();
}

void KplDoc::moveItem(int is, int id)
{
  KplItem *item = itd.at(is);
  itd.setAutoDelete(false);
  itd.remove(is);
  itd.insert(id, item);
  itd.setAutoDelete(true);
  setModified();
  backupItems();
}
