/***************************************************************************
                          frameitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <ksimpleconfig.h>
#include <klocale.h>
#include "kplgraph.h"
#include "kplchecklistitem.h"
#include "frameitem.h"
#include "framedlg.h"
#include "utils.h"

FrameItem::FrameItem()
{
  init();
}

FrameItem::FrameItem(Kpl::AutoStruct* aut) :
 logx(aut->autoLogX), logy(aut->autoLogY), ndigx(-1), ndigy(-1),
 mticx(5), mticy(2), gridmode(aut->autoGrid), iex(0), iey(0),
 colFrame(aut->colFrame), colGrid(aut->colGrid), xf(aut->xf), yf(aut->yf),
 xw0(aut->x0Auto), xw(aut->xlAuto), yw0(aut->y0Auto), yw(aut->ylAuto),
 xmi(0.0), xma(aut->xlAuto), ymi(0.0), yma(aut->ylAuto), xtic(5.0), ytic(2.0),
 relsiz(1.0), sx(aut->sxAuto), sy(aut->syAuto), sh(aut->shAuto)
{
}

FrameItem::FrameItem(KSimpleConfig* plo, Kpl::AutoStruct* aut) : KplItem(plo)
{
  logx = plo->readBoolEntry("logx", false);
  logy = plo->readBoolEntry("logy", false);
  ndigx = plo->readNumEntry("ndigx", -1);
  ndigy = plo->readNumEntry("ndigy", -1);
  mticx = plo->readNumEntry("mticx", 5);
  mticy = plo->readNumEntry("mticy", 2);
  gridmode = plo->readNumEntry("gridmode", aut->autoGrid);
  iex = plo->readNumEntry("iex");
  iey = plo->readNumEntry("iey");
  char s[256];
  strcpy(s, plo->readEntry("colframe", ""));
  unsigned u;
  if (strlen(s)) {
    sscanf(s, "%x", &u);
    colFrame = Utils::rgbQt(u);
  } else
    colFrame = aut->colFrame;
  strcpy(s, plo->readEntry("colgrid", ""));
  if (strlen(s)) {
    sscanf(s, "%x", &u);
    colGrid = Utils::rgbQt(u);
  } else
    colGrid = aut->colGrid;
  xf = plo->readDoubleNumEntry("xf", aut->xf);
  yf = plo->readDoubleNumEntry("yf", aut->yf);
  xw0 = plo->readDoubleNumEntry("xw0", aut->x0Auto);
  xw = plo->readDoubleNumEntry("xw", aut->xlAuto);
  yw0 = plo->readDoubleNumEntry("yw0", aut->y0Auto);
  yw = plo->readDoubleNumEntry("yw", aut->ylAuto);
  xmi = plo->readDoubleNumEntry("xmi");
  xma = plo->readDoubleNumEntry("xma", xmi + 15.0);
  ymi = plo->readDoubleNumEntry("ymi");
  yma = plo->readDoubleNumEntry("yma", ymi + 10.0);
  xtic = plo->readDoubleNumEntry("xtic", 5.0);
  ytic = plo->readDoubleNumEntry("ytic", 2.0);
  relsiz = plo->readDoubleNumEntry("relsiz", 1.0);
  sx = plo->readEntry("sx", "");
  sy = plo->readEntry("sy", "");
  sh = plo->readEntry("sh", "");
}

FrameItem::FrameItem(bool act, bool lgx, bool lgy, int ndx, int ndy,
                     int mtx, int mty, int gm, int ex, int ey,
                     const QString& col_f, const QString& col_g,
                     double x0, double w, double y0, double h,
                     double xmin, double xmax, double ymin,
                     double ymax, double xt, double yt, double relSize,
                     const QString& sX, const QString& sY, const QString& sH,
                     Kpl::AutoStruct* aut) :
 logx(lgx), logy(lgy), ndigx(ndx), ndigy(ndy), mticx(mtx), mticy(mty),
 gridmode(gm), iex(ex), iey(ey), xf(aut->xf), yf(aut->yf),
 xw0(x0), xw(w), yw0(y0), yw(h), xmi(xmin), xma(xmax), ymi(ymin), yma(ymax),
 xtic(xt), ytic(yt), relsiz(relSize), sx(sX), sy(sY), sh(sH)
{
  active = act;
  unsigned u;
  sscanf(col_f.latin1(), "%x", &u);
  colFrame = Utils::rgbQt(u);
  sscanf(col_g.latin1(), "%x", &u);
  colGrid = Utils::rgbQt(u);
}

FrameItem::~FrameItem()
{
}

KplItem::ItemTypes FrameItem::iType() const
{
  return Frame;
}

void FrameItem::init()
{
  logx = logy = false;
  iex = iey = 0;
  xmi = ymi = 0.0;
  xtic = 5.0;
  ytic = 2.0;
  relsiz = 1.0;
  mticx = 5;
  mticy = 2;
  ndigx = ndigy = -1;
}

void FrameItem::draw(KplGraph* g) const
{
  if (xw && yw && active) {
    g->format(xf, yf);
    g->setRelSize(relsiz);
    g->Window(xw0, xw, yw0, yw);
    g->scale(xmi, xma, ymi, yma, logx, logy);
    g->setDig(ndigx, ndigy);
    g->setColFrame(colFrame);
    g->setColGrid(colGrid);
    g->raster(xtic, ytic, mticx, mticy, gridmode);
    g->letX(sx, iex);
    g->letY(sy, iey);
    g->letH(sh);
  }
}

void FrameItem::writePlo(FILE* f, const KURL& url, bool abs) const
{
  fprintf(f, "Type=FRAMEITEM\n");
  KplItem::writePlo(f, url, abs);
  if (logx)
    fprintf(f, "logx=true\n");
  if (logy)
    fprintf(f, "logy=true\n");
  fprintf(f, "ndigx=%i\nndigy=%i\nmticx=%i\nmticy=%i\ngridmode=%i\n"
          "iex=%i\niey=%i\ncolframe=%x\ncolgrid=%x\n"
          "xf=%g\nyf=%g\nxw0=%g\nxw=%g\nyw0=%g\nyw=%g\n"
          "xmi=%g\nxma=%g\nymi=%g\nyma=%g\nxtic=%g\nytic=%g\n"
          "relsiz=%g\nsx=%s\nsy=%s\nsh=%s\n",
          ndigx, ndigy, mticx, mticy, gridmode,
          iex, iey, Utils::rgbQt1(colFrame), Utils::rgbQt1(colGrid),
          xf, yf, xw0, xw, yw0, yw,
          xmi, xma, ymi, yma, xtic, ytic,
          relsiz, sx.utf8().data(), sy.utf8().data(), sh.utf8().data());
}

void FrameItem::setText(KplCheckListItem* it, bool*, bool*) const
{
  it->setText(1, i18n("Frame"));
  QString s1, s2;
  it->setText(2, s1.sprintf("x = %g cm, y = %g cm, ", xw0, yw0) + i18n("size")
              + s2.sprintf(" %g cm x %g cm", xw, yw));
}

int FrameItem::editItem(QWidget* parent, KplDoc* m, int i)
{
  FrameDlg dlg(parent, m, i);
  return dlg.exec();
}

KplItem* FrameItem::copy() const
{
  return new FrameItem(*this);
}
