#ifndef DATABASE_H_
#define DATABASE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif 

// Local includes
#include "gdbm.h"
#include "databaseresource.h"

#define DATA_TYPE datum

/** This is the class that will need to be rewritten when a new database is used.
 * This one is currently written for gdbm
 *
 * @author Mike Pilone
 */
class Database
{
public:
  Database();
  ~Database();

  /** Opens the database. Returns TRUE if successful, FALSE otherwise
   *
   * @param name Should be the name of the database file
   * @param flags Whatever flags that would need to get passed to the database opener, like
   * readwrite or readonly
   */
  bool openDB(char* name, dbOpenType flag = dbNotOpen);
  void closeDB();

  /** Inserts or replaces data in the database
   *
   * -1
   * The item was not stored in the database because the caller was not
   * an official writer or either `key' or `content' have a NULL dptr
   * field.  Both `key' and `content' must have the dptr field be a
   * non-NULL value.  Since a NULL dptr field is used by other
   * functions to indicate an error, a NULL field cannot be valid data.
   *
   * Or the database wasn't open.
   *
   * +1
   * The item was not stored because the argument `flag' was
   * GDBM_INSERT and the `key' was already in the database.
   *
   * 0
   * No error. `content' is keyed by `key'. The file on disk is updated
   * to reflect the structure of the new database before returning from
   * this function.
   */
  int storeData(DATA_TYPE key, DATA_TYPE content, insertionPolicy flag);

  DATA_TYPE fetchData(DATA_TYPE key);

  /** The ret value is -1 if the item is not present or the requester is a
   * reader or DB is not open.  The ret value is 0 if there was a successful 
   * delete.
   */
  int deleteData(DATA_TYPE key);

  /** Not recommended for use. This may cause problems because of the multiple
   * key design of the address book. This method may be removed later.
   */
  DATA_TYPE firstKey();

    /** Not recommended for use. This may cause problems because of the multiple
   * key design of the address book. This method may be removed later.
   */
  DATA_TYPE nextKey(DATA_TYPE key);

  /** If an error is detected, the return value is negative. The value 
   * zero is returned after a successful reorganization.
   */
  int reorganize();

  void sync();

  /** simply returns true or false, depending on whether the `key' exists, or
   * not.
   */
  int exists(DATA_TYPE key);

  char* strerror(int) {return "";};
  
  dbOpenType openDBType();

private:
  /** Stores whether the DB is open or not
   */
  dbOpenType fDBOpen;

  /** Stores the pointer to the open DB file
   */
  GDBM_FILE fDatabase;

};

#endif
