/***************************************************************************
                          messenger.cpp  -  description                              
                             -------------------                                         
    begin                : Mon Apr  5 09:53:56 MEST 1999
                                           
    copyright            : (C) 1999 by Torsten Uhlmann                         
    email                : TUhlmann@gmx.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include <qscrollview.h>
#include <qmessagebox.h>
#include <qstring.h>
#include <qstrlist.h>
#include <kprocess.h>
#include <kfiledialog.h>

#include "messenger.h"
#include "settingsdialog.h"
#include "ModemControl.h"

MessengerApp::MessengerApp()
{
  setCaption("Messenger " VERSION);

  ///////////////////////////////////////////////////////////////////
  // read the config file options
  readOptions();

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initMenuBar();
  initToolBar();
  initStatusBar();
  initView();
  initModem();

  ///////////////////////////////////////////////////////////////////
  // enable bars dependend on config file setups
  if (!configuration.bViewToolbar)
    enableToolBar(KToolBar::Hide,0);
  if (!configuration.bViewStatusbar)
    enableStatusBar(KStatusBar::Hide);

  menuBar()->setMenuBarPos(configuration.menu_bar_pos);
  toolBar()->setBarPos(configuration.tool_bar_pos);

  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar items at startup
	disableCommand(ID_FAX_POPUP_VIEW);
	disableCommand(ID_FAX_POPUP_SAVE);
	disableCommand(ID_FAX_POPUP_PRINT);
	disableCommand(ID_VOICE_POPUP_SAVE);
	disableCommand(ID_VOICE_POPUP_PLAY_INTERN);
	disableCommand(ID_VOICE_POPUP_PLAY_CARD);
	disableCommand(ID_GENERIC_POPUP_SAVE);

  _tabListIndex = -1;
  _faxIndex     = -1;
}

MessengerApp::~MessengerApp()
{
	delete modem;
}

void MessengerApp::enableCommand(int id_)
{
  ///////////////////////////////////////////////////////////////////
  // enable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_,true);
  toolBar()->setItemEnabled(id_,true);
}

void MessengerApp::disableCommand(int id_)
{
  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_,false);
  toolBar()->setItemEnabled(id_,false);
}


void MessengerApp::initMenuBar()
{

  ///////////////////////////////////////////////////////////////////
  // MENUBAR  

  ///////////////////////////////////////////////////////////////////
  // menuBar entry file_menu
  file_menu = new QPopupMenu();
  //file_menu->insertItem(Icon("mini/messengerapp.xpm"), i18n("New &Window"), ID_FILE_NEW_WINDOW );
  //file_menu->insertItem(Icon("fileprint.xpm"), i18n("&Print"), ID_FILE_PRINT );
  file_menu->insertItem(i18n("E&xit"), ID_FILE_QUIT );
  file_menu->setAccel(CTRL+Key_Q, ID_FILE_QUIT);


  ///////////////////////////////////////////////////////////////////
  // menuBar entry view_menu
  view_menu = new QPopupMenu();
  view_menu->setCheckable(true);
  view_menu->insertItem(i18n("Tool&bar"), ID_VIEW_TOOLBAR);
  view_menu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR );

  view_menu->setItemChecked(ID_VIEW_TOOLBAR, configuration.bViewToolbar);
  view_menu->setItemChecked(ID_VIEW_STATUSBAR, configuration.bViewStatusbar);

  ///////////////////////////////////////////////////////////////////
  // EDIT YOUR APPLICATION SPECIFIC MENUENTRIES HERE

  // modem_menu
  modem_menu = new QPopupMenu();
  modem_menu->insertItem(i18n("Get Memory &Information"), ID_MODEM_GET_INFO);
  modem_menu->insertItem(i18n("&Get Fax Memory"), ID_MODEM_GET_MEMORY);
  modem_menu->insertSeparator();
  modem_menu->insertItem(i18n("&Control Modem"), ID_MODEM_CONTROL);
  modem_menu->insertSeparator();
  modem_menu->insertItem(i18n("&Clear Fax Memory"), ID_MODEM_CLEAR_MEMORY);
  modem_menu->insertItem(i18n("&Reset Modem Clock"),ID_MODEM_RESET_CLOCK);
  //modem_menu->insertItem(i18n("Toggle &Fax Reception"),ID_MODEM_TOGGLE_FRECEPT);
  //modem_menu->insertItem(i18n("Toggle &Voice Reception"),ID_MODEM_TOGGLE_VRECEPT);
  //modem_menu->insertItem(i18n("Toggle Control &Monitor"),ID_MODEM_TOGGLE_MONITOR);
  //modem_menu->insertItem(i18n("&Ring Setup"),ID_MODEM_SETUP_RINGS);
  //modem_menu->insertItem(i18n("Setup Fax &Id String"),ID_MODEM_SETUP_FAXID);
  modem_menu->insertItem(i18n("Commit C&hanges"),ID_MODEM_COMMIT_SETUP);


  // modem_menu
  config_menu = new QPopupMenu();
  config_menu->insertItem(i18n("&Settings"),     ID_CONFIG_CONFIG);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry help_menu
  help_menu = new QPopupMenu();
  help_menu = kapp->getHelpMenu(true, i18n(IDS_APP_ABOUT));

  // create the fax popup menu
  fax_popup_menu = new QPopupMenu();
  fax_popup_menu->insertItem(i18n("&Save Fax as ..."),ID_FAX_POPUP_SAVE);
  fax_popup_menu->insertSeparator();
  fax_popup_menu->insertItem(i18n("&View Fax"),ID_FAX_POPUP_VIEW);
  fax_popup_menu->insertItem(i18n("&Print Fax"),ID_FAX_POPUP_PRINT);

  // create the voice popup menu
  voice_popup_menu = new QPopupMenu();
  voice_popup_menu->insertItem(i18n("&Save Message as ...")	,ID_VOICE_POPUP_SAVE);
  voice_popup_menu->insertSeparator();
  voice_popup_menu->insertItem(i18n("Play &internal")				,ID_VOICE_POPUP_PLAY_INTERN);
  voice_popup_menu->insertItem(i18n("Play via &Soundcard")	,ID_VOICE_POPUP_PLAY_CARD);

  // popup_menu key accelerators
  fax_popup_menu->setAccel(CTRL+Key_V, ID_FAX_POPUP_VIEW);
  fax_popup_menu->setAccel(CTRL+Key_S, ID_FAX_POPUP_SAVE);
  fax_popup_menu->setAccel(CTRL+Key_P, ID_FAX_POPUP_PRINT);

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // set menuBar() the current menuBar and the position due to config file
  menuBar()->insertItem(i18n("&File") 		, file_menu);
  menuBar()->insertItem(i18n("&View") 		, view_menu);
  menuBar()->insertItem(i18n("&Modem")		, modem_menu);
  menuBar()->insertItem(i18n("&Fax")  		, fax_popup_menu);
  menuBar()->insertItem(i18n("&Voice") 		, voice_popup_menu);
  menuBar()->insertItem(i18n("&Settings")	, config_menu);

  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC MENUENTRIES HERE


  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), help_menu);

  ///////////////////////////////////////////////////////////////////
  // CONNECT THE SUBMENU SLOTS WITH SIGNALS

  CONNECT_CMD(file_menu );
  CONNECT_CMD(view_menu );
  CONNECT_CMD(modem_menu);
  CONNECT_CMD(config_menu);
  CONNECT_CMD(fax_popup_menu);
  CONNECT_CMD(voice_popup_menu);
}

void MessengerApp::initToolBar()
{

  ///////////////////////////////////////////////////////////////////
  // TOOLBAR
  // set toolBar() the current toolBar and the position due to config file
  toolBar()->insertButton(Icon("exit.xpm")         	, ID_FILE_QUIT, true, i18n("Exit"));
  toolBar()->insertSeparator();
  toolBar()->insertButton(Icon("delete.xpm")       	, ID_MODEM_CLEAR_MEMORY, true, i18n("Clear Modem Memory"));
  toolBar()->insertButton(Icon("find.xpm")         	, ID_MODEM_GET_INFO, true, i18n("Get Memory Information"));
  toolBar()->insertButton(Icon("lassoo.xpm")       	, ID_MODEM_GET_MEMORY, true, i18n("Get Modem Memory"));
  toolBar()->insertButton(Icon("print_preview.xpm")	, ID_FAX_POPUP_VIEW, true, i18n("View the selected Fax"));
  toolBar()->insertButton(Icon("diashow.xpm")				, ID_FAX_POPUP_PRINT, true, i18n("Print the selected Fax"));
  toolBar()->insertButton(Icon("topic.xpm")					, ID_VOICE_POPUP_PLAY_INTERN, true, i18n("Play the selected message via internal speaker"));
  toolBar()->insertButton(Icon("next.xpm")					, ID_VOICE_POPUP_PLAY_CARD, true, i18n("Play the selected message via soundcard"));
  toolBar()->insertButton(Icon("filefloppy.xpm")   	, ID_GENERIC_POPUP_SAVE, true, i18n("Save the selected message"));
  toolBar()->insertSeparator();
  toolBar()->insertButton(Icon("help.xpm"), ID_HELP, SIGNAL(pressed()), kapp, SLOT(appHelpActivated()), true, i18n("Help"));

  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE -e.g. tool_bar_1:
  // add functionality for new created toolbars in:
  // enableCommand, disableCommand, in the menuBar() and an additional function slotViewToolbar_1
  // for that also create a bViewToolbar_1 and a KConfig entry (see Constructor).
  // Also update resource values and commands 


  ///////////////////////////////////////////////////////////////////
  // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars
  CONNECT_TOOLBAR(toolBar());

}

void MessengerApp::initStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  //STATUSBAR
  statusBar()->insertItem(IDS_DEFAULT, ID_STATUS_MSG );
  statusBar()->setInsertOrder(KStatusBar::RightToLeft);

}

void MessengerApp::initView()
{ 
  ////////////////////////////////////////////////////////////////////
  // set the main widget here
  KApplication *app=KApplication::getKApplication();
  view = new MessengerView(app,this);
  setView(view);
  connect( view, SIGNAL(popupMenu  ( int,int )), SLOT(slotOpenPopup   ( int,int )) );
  connect( view, SIGNAL(highlighted( int,int )), SLOT(slotSetFaxIndex ( int,int )) );
  //ON_STATUS_MSG(ID_POPUP_VIEW_FAX,    i18n("extracts the selected fax and shows it"))
  connect(this, SIGNAL(gotData(struct ModemGeneralInfo, ModemMessageList)),
					view, SLOT(slotGotData(struct ModemGeneralInfo, ModemMessageList)));
}

void MessengerApp::initModem()
{
  // Create a modem object and initialize it with serial line
  modem = new USRModem(configuration.serialLine.data(),
  										 configuration.baudrate.data(),
  										 configuration.lockfile.data());
  CHECK_PTR(modem);
  if (!modem->initSerialLine())
  {
    _modemInitialized = false;
    QMessageBox::information(0,i18n("Initializing Modem"),
      i18n("Could not initialize modem!\nPlease check modem device."),
      QMessageBox::Ok | QMessageBox::Default);
    slotDisableMost();
  } else {
    modem->initModem();
    _modemInitialized = true;
  }
}

bool MessengerApp::queryExit()
{

	if (!_modemInitialized)
		return true;
		
  int mode=KMsgBox::yesNoCancel(this, i18n("Standalone Mode"),
  			i18n("Do you want to switch in Standalone Mode?"));

  if (mode != 3)
  {
	  if(mode==1)
  	  modem->setStandaloneMode();
  	else
  		modem->unsetStandaloneMode();
  		
  	return true;
  }
 	return false;
}

void MessengerApp::saveOptions()
{

	//create separate object that handles configurable values
	
  KConfig *config = kapp->getConfig();

  config->setGroup("APPEARANCE");
  config->writeEntry("ShowToolbar",toolBar()->isVisible());
  config->writeEntry("ShowStatusbar",statusBar()->isVisible());
  config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
  config->writeEntry("ToolBar_Pos", (int)toolBar()->barPos());

  // write modem settings
  config->setGroup("MODEM");
  config->writeEntry("SerialLine",	configuration.serialLine);
  config->writeEntry("Baudrate", 		configuration.baudrate);
  config->writeEntry("Lockfile",		configuration.lockfile);

  // read SOUND settings
  config->setGroup("SOUND");
  config->writeEntry("AudioDevice"	,	configuration.audioDevice);

  // helper applications
  config->setGroup("HELPER");
  config->writeEntry("FaxViewer",				configuration.faxViewer);
  config->writeEntry("VoiceProcessor",	configuration.voiceProcessor);
}

void MessengerApp::readOptions()
{
  ///////////////////////////////////////////////////////////////////
  // read the config file entries
  KConfig *config = kapp->getConfig();

 
  config->setGroup("APPEARANCE");
  configuration.bViewToolbar = config->readBoolEntry("ShowToolbar", true);
  configuration.bViewStatusbar = config->readBoolEntry("ShowStatusbar", true);
  configuration.menu_bar_pos = (KMenuBar::menuPosition)config->readNumEntry("MenuBarPos", KMenuBar::Top);
  configuration.tool_bar_pos = (KToolBar::BarPosition)config->readNumEntry("ToolBar_Pos", KToolBar::Top);

  // read modem settings
  config->setGroup("MODEM");
  configuration.serialLine 	= config->readEntry("SerialLine"	, MODEMDEVICE);
  configuration.baudrate 		= config->readEntry("Baudrate"		, BAUDRATE);
  configuration.lockfile		= config->readEntry("Lockfile"		, LOCKFILE);

  // read SOUND settings
  config->setGroup("SOUND");
  configuration.audioDevice	= config->readEntry("AudioDevice", AUDIODEVICE);

  // read helper settings
  config->setGroup("HELPER");
  configuration.faxViewer 			= config->readEntry("FaxViewer"			, FAXVIEWER);
  configuration.voiceProcessor	= config->readEntry("VoiceProcessor", VOICEPROCESSOR);

}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void MessengerApp::slotFileQuit()
{ 

  ///////////////////////////////////////////////////////////////////
  // exits the Application
  if(this->queryExit())
    {
    	// should we turn on self mode?
      saveOptions();
      KTMainWindow::deleteAll();
      kapp->quit();
    }
  else
    slotStatusMsg(IDS_DEFAULT);
    return;
}

void MessengerApp::slotViewToolBar()
{
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
  configuration.bViewToolbar=!configuration.bViewToolbar;
  menuBar()->setItemChecked(ID_VIEW_TOOLBAR, configuration.bViewToolbar);
  enableToolBar(KToolBar::Toggle,0);
  slotStatusMsg(IDS_DEFAULT);

}

void MessengerApp::slotViewStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  configuration.bViewStatusbar=!configuration.bViewStatusbar;
  menuBar()->setItemChecked(ID_VIEW_STATUSBAR, configuration.bViewStatusbar);
  enableStatusBar();
  slotStatusMsg(IDS_DEFAULT);
}


void MessengerApp::slotStatusMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG );
}


void MessengerApp::slotStatusHelpMsg(const char *text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}

void MessengerApp::slotOpenPopup(int index, int column)
{
/*  _tabListIndex = index;
  QString str = view->text(index,0);
  _faxIndex = str.toInt();

  str = view->text(index,1);
  if (str == "Fax")
  	_msgType = Fax;
  else
  	if (str == "Voice")
  		_msgType = Voice;
  	else
  		_msgType = unknown;
*/
	slotSetFaxIndex(index,column);
			
  //debug("Message index: %i, Message Type: %i",_faxIndex,_msgType);
  switch (_msgType)
  {
  	case Fax: fax_popup_menu->exec( QCursor::pos() );
  		break;
  	case Voice: voice_popup_menu->exec( QCursor::pos() );
  		break;
  	default:
  		QMessageBox::information(0,i18n("Open Popup"),
   			i18n("Message type unknown!"),
   	  	QMessageBox::Ok | QMessageBox::Default);
   	 	return;

  }
}


void MessengerApp::commandCallback(int id_){
  switch (id_){
    ON_CMD(ID_FILE_QUIT,                slotFileQuit())

    ON_CMD(ID_VIEW_TOOLBAR,             slotViewToolBar())
    ON_CMD(ID_VIEW_STATUSBAR,           slotViewStatusBar())

    ON_CMD(ID_MODEM_GET_INFO,           slotGetMemoryInfo())
    ON_CMD(ID_MODEM_GET_MEMORY,         slotGetMemory())
    ON_CMD(ID_MODEM_CLEAR_MEMORY,       slotClearMemory())

    ON_CMD(ID_MODEM_RESET_CLOCK				,	slotResetClock())
    ON_CMD(ID_MODEM_COMMIT_SETUP			,	slotCommitSetup())

    ON_CMD(ID_CONFIG_CONFIG,       			slotSettings())
    ON_CMD(ID_MODEM_CONTROL,       			slotModemControl())

    ON_CMD(ID_FAX_POPUP_VIEW,           slotViewFax())
    ON_CMD(ID_FAX_POPUP_SAVE,           slotSaveFax())
    ON_CMD(ID_FAX_POPUP_PRINT,          slotPrintFax())

    ON_CMD(ID_GENERIC_POPUP_SAVE, 			slotSaveGeneric())

    ON_CMD(ID_VOICE_POPUP_SAVE, 				slotSaveVoice())
    ON_CMD(ID_VOICE_POPUP_PLAY_INTERN, 	slotPlayIntern())
    ON_CMD(ID_VOICE_POPUP_PLAY_CARD,  	slotPlaySoundcard())
  }
}

void MessengerApp::statusCallback(int id_){
  switch (id_){
    ON_STATUS_MSG(ID_FILE_QUIT,         			i18n("Exits the program"))

    ON_STATUS_MSG(ID_VIEW_TOOLBAR,      			i18n("Enables / disables the actual Toolbar"))
    ON_STATUS_MSG(ID_VIEW_STATUSBAR,    			i18n("Enables / disables the Statusbar"))

    ON_STATUS_MSG(ID_MODEM_GET_INFO,    			i18n("Get memory status from fax modem"))
    ON_STATUS_MSG(ID_MODEM_GET_MEMORY,  			i18n("Get used memory from fax modem"))
    ON_STATUS_MSG(ID_MODEM_CLEAR_MEMORY,			i18n("Clear memory of fax modem"))

    ON_STATUS_MSG(ID_MODEM_RESET_CLOCK			,	i18n("Reset modem clock"))
    ON_STATUS_MSG(ID_MODEM_COMMIT_SETUP			,	i18n("Commit changes"))

    ON_STATUS_MSG(ID_CONFIG_CONFIG,	    			i18n("Configure application settings"))
    ON_STATUS_MSG(ID_MODEM_CONTROL,	    			i18n("Configure Modem settings"))

    ON_STATUS_MSG(ID_FAX_POPUP_VIEW,    			i18n("View selected fax within Faxviewer"))
    ON_STATUS_MSG(ID_FAX_POPUP_SAVE,    			i18n("Save selected fax as G3 files"))
    ON_STATUS_MSG(ID_FAX_POPUP_PRINT,   			i18n("Print selected fax"))

    ON_STATUS_MSG(ID_GENERIC_POPUP_SAVE,			i18n("Save selected message"))

    ON_STATUS_MSG(ID_VOICE_POPUP_SAVE,				i18n("Save selected message as gsm file"))
    ON_STATUS_MSG(ID_VOICE_POPUP_PLAY_INTERN, i18n("Play selected message via modem"))
    ON_STATUS_MSG(ID_VOICE_POPUP_PLAY_CARD,		i18n("Play selected message via soundcard"))
  }
}

// OWN SLOT DEFINITIONS

void MessengerApp::slotGetMemoryInfo()
{
  struct ModemGeneralInfo moGeneral;
  ModemMessageList				moMsgList;

  slotStatusMsg(i18n("Get modem memory info..."));

  moMsgList.setAutoDelete(true);
  if (modem->getMemoryInfo(moGeneral, moMsgList))
  {
    emit gotData(moGeneral, moMsgList);
    //QMessageBox::information(0,i18n("Get Memory Info"),
    //  i18n("Got Information about stored faxes."),
    //  QMessageBox::Ok | QMessageBox::Default);
  } else {
    QMessageBox::information(0,i18n("Get Memory Info"),
      i18n("Could not get memory information\nProbably no faxes stored"),
      QMessageBox::Ok | QMessageBox::Default);
  }

  slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotGetFax(char *filename)
{
  slotStatusMsg(i18n("Extracting selected fax..."));

  if (_faxIndex > 0)
  {
    modem->getFax(_faxIndex, filename);
  }

  slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotGetVoice(char *filename)
{
  slotStatusMsg(i18n("Extracting selected message..."));

  if (_faxIndex > 0)
  {
    modem->getVoice(_faxIndex, filename);
  }

  slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotGetMemory()
{
  slotStatusMsg(i18n("Get modem memory ..."));

  if (!modem->getMemory())
  {
     QMessageBox::information(0,i18n("Get Memory"),
       i18n("Could not read fax memory!"),
       QMessageBox::Ok | QMessageBox::Default);
  } else {
    QMessageBox::information(0,i18n("Get Memory"),
      i18n("Fax Memory downloaded..."),
      QMessageBox::Ok | QMessageBox::Default);
  }

  slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotClearMemory()
{
  slotStatusMsg(i18n("Clear modem memory..."));

  if (modem->clearMemory())
  {
    view->clear();
    QMessageBox::information(0,i18n("Clearing fax memory"),
      i18n("Fax memory is now empty."),
      QMessageBox::Ok | QMessageBox::Default);
  } else {
    QMessageBox::information(0,i18n("Clearing fax memory"),
      i18n("Fax memory could not be cleared!"),
      QMessageBox::Ok | QMessageBox::Default);
  }

  slotStatusMsg(IDS_DEFAULT);
}

/** handles one fax
		return is no of pages written */
int MessengerApp::faxHandler()
{
  int  pagesWritten  = 0;
  char filename[255] = "";

  if (_faxIndex < 1)
  {
    QMessageBox::information(0,i18n("Get Fax"),
      i18n("You need to select the desired fax!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return 0;
  }

  if (_msgType != Fax)
  {
    QMessageBox::information(0,i18n("Get Fax"),
      i18n("You need to select a fax message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return 0;
  }

  if (!modem->getMemoryFileName(filename))
  {
    if (!modem->getMemory())
    {
      QMessageBox::information(0,i18n("Get Fax"),
        i18n("Could not read fax memory!"),
        QMessageBox::Ok | QMessageBox::Default);
		  slotStatusMsg(IDS_DEFAULT);
      return 0;
    }
  }
  slotGetFax(0);
  if (!modem->getFaxFileName(filename))
  {
    QMessageBox::information(0,i18n("Get Fax"),
      i18n("Could not extract the selected fax!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
    return 0;
  }

  debug("Fax stored in %s-<nr>",filename);

  if (!(pagesWritten = modem->getPagesWritten()))
  {
    slotStatusMsg(i18n("No pages written!"));

    QMessageBox::information(0,i18n("Get Fax"),
      i18n("There were no pages written!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
    return 0;
  }
  return pagesWritten;
}

/** handles one voice message
		return is ok or not ok */
bool MessengerApp::voiceHandler()
{
  char filename[255] = "";

  if (_faxIndex < 1)
  {
    QMessageBox::information(0,i18n("Get Message"),
      i18n("You need to select the desired message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return false;
  }

  if (_msgType != Voice)
  {
    QMessageBox::information(0,i18n("Get Message"),
      i18n("You need to select a voice message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return false;
  }

  if (!modem->getMemoryFileName(filename))
  {
    if (!modem->getMemory())
    {
      QMessageBox::information(0,i18n("Get Message"),
        i18n("Could not read modem memory!"),
        QMessageBox::Ok | QMessageBox::Default);
		  slotStatusMsg(IDS_DEFAULT);
      return false;
    }
  }
  slotGetVoice(0);
  if (!modem->getVoiceFileName(filename))
  {
    QMessageBox::information(0,i18n("Get Message"),
      i18n("Could not extract the selected message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
    return false;
  }

  debug("Message stored in %s.gsm",filename);

  return true;
}

void MessengerApp::slotViewFax()
{
  bool success       = false;
  int  pagesWritten  = 0;
  int  i             = 0;
  QString arg        = "";
  char filename[255] = "";

  slotStatusMsg(i18n("View the selected fax..."));

  if (((pagesWritten = faxHandler()) > 0) &&
  		(modem->getFaxFileName(filename)))
  {
  	// now spawn Faxviewer process with pages as parameters
  	slotStatusMsg(i18n("starting Faxviewer..."));
  	proc.clearArguments();
  	proc << configuration.faxViewer.data();
	  for (i=1; i <= pagesWritten; i++)
  	{
    	arg = "";
	    arg.sprintf("%s-%i.g3",filename,i);
  	  debug("Argument %i: %s",i,arg.data());
    	proc << arg.data();
	  }
  	success = proc.start();
	  if (success)
  	  slotStatusMsg(i18n("Faxviewer started..."));
	  else
  	{
    	slotStatusMsg(i18n("Could not start Faxviewer!"));

	    QMessageBox::information(0,i18n("Viewing Fax"),
  	    i18n("Could not start Faxviewer!"),
    	  QMessageBox::Ok | QMessageBox::Default);
	  }
  }

  slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotSaveFax()
{

  int  pagesWritten  = 0;
  char filename[255] = "";
  QString saveFileName;

  slotStatusMsg(i18n("Save the selected fax..."));

  if (_faxIndex < 1)
  {
    QMessageBox::information(0,i18n("Save Fax As..."),
      i18n("You need to select the desired fax!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

  if (_msgType != Fax)
  {
    QMessageBox::information(0,i18n("Save Fax As..."),
      i18n("You need to select a fax message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

  saveFileName = KFileDialog::getSaveFileName();

	if (!saveFileName.isEmpty())
  {
  	debug("files will be stored in %s",saveFileName.data());
	  if (!modem->getMemoryFileName(filename))
	  {
	    if (!modem->getMemory())
	    {
  	    QMessageBox::information(0,i18n("Save Fax As..."),
   	     i18n("Could not read fax memory!"),
   	     QMessageBox::Ok | QMessageBox::Default);
			  slotStatusMsg(IDS_DEFAULT);
   	   return;
	    }
  	}
	  slotGetFax(saveFileName.data());
  	if (!modem->getFaxFileName(filename))
	  {
  	  QMessageBox::information(0,i18n("Save Fax As..."),
    	  i18n("Could not extract the selected fax!"),
      	QMessageBox::Ok | QMessageBox::Default);
		  slotStatusMsg(IDS_DEFAULT);
  	  return;
	  }

  	debug("Fax stored in %s-<nr>",filename);

	  if (!(pagesWritten = modem->getPagesWritten()))
  	{
	    slotStatusMsg(i18n("No pages written!"));

  	  QMessageBox::information(0,i18n("Save Fax As..."),
    	  i18n("There were no pages written!"),
      	QMessageBox::Ok | QMessageBox::Default);
		  slotStatusMsg(IDS_DEFAULT);
  	  return;
	  }
  }
  slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotSetFaxIndex(int index, int column)
{
  _tabListIndex = index;
  QString str = view->text(index,0);
  _faxIndex = str.toInt();

  str = view->text(index,1);
  if (str == "Fax")
  	_msgType = Fax;
  else
  	if (str == "Voice")
  		_msgType = Voice;
  	else
  		_msgType = unknown;

  switch(_msgType)
  {
		case Fax:
			enableCommand(ID_FAX_POPUP_VIEW);
			enableCommand(ID_FAX_POPUP_SAVE);
			enableCommand(ID_FAX_POPUP_PRINT);
			disableCommand(ID_VOICE_POPUP_SAVE);
			disableCommand(ID_VOICE_POPUP_PLAY_INTERN);
			disableCommand(ID_VOICE_POPUP_PLAY_CARD);
			enableCommand(ID_GENERIC_POPUP_SAVE);
			break;
		case Voice:
			disableCommand(ID_FAX_POPUP_VIEW);
			disableCommand(ID_FAX_POPUP_SAVE);
			disableCommand(ID_FAX_POPUP_PRINT);
			enableCommand(ID_VOICE_POPUP_SAVE);
			enableCommand(ID_VOICE_POPUP_PLAY_INTERN);
			enableCommand(ID_VOICE_POPUP_PLAY_CARD);
			enableCommand(ID_GENERIC_POPUP_SAVE);
			break;
		default:
			disableCommand(ID_FAX_POPUP_VIEW);
			disableCommand(ID_FAX_POPUP_SAVE);
			disableCommand(ID_FAX_POPUP_PRINT);
			disableCommand(ID_VOICE_POPUP_SAVE);
			disableCommand(ID_VOICE_POPUP_PLAY_INTERN);
			disableCommand(ID_VOICE_POPUP_PLAY_CARD);
			disableCommand(ID_GENERIC_POPUP_SAVE);
	}
  debug("Message index: %i, Message Type: %i",_faxIndex,_msgType);
}

void MessengerApp::slotPrintFax()
{
  bool success       = false;
  int  pagesWritten  = 0;
  int  i             = 0;
  QString arg        = "";
  char filename[255] = "";
  KShellProcess	printer;

  slotStatusMsg(i18n("printing selected fax..."));

  if (((pagesWritten = faxHandler()) > 0) &&
  		(modem->getFaxFileName(filename)))
  {
  	// now spawn print process with pages as parameters
  	slotStatusMsg(i18n("printing to your default printer..."));
	  for (i=1; i <= pagesWritten; i++)
  	{
	  	printer.clearArguments();
    	arg = "";
	    arg.sprintf("g32pbm %s-%i.g3 | pnmtops -noturn | lpr", filename,i);
  	  debug("String %i: %s",i,arg.data());
  	  //system(arg.data());
    	printer << arg.data();
	  	success = printer.start(KProcess::Block);
	  	if (success)
	  	{
  	  	slotStatusMsg(i18n("Page sent to printer..."));
  	  	//sleep(2);
  	 	}
	  	else
  		{
    		slotStatusMsg(i18n("Could not print page!"));

	    	QMessageBox::information(0,i18n("Print Fax"),
  	    	i18n("Could not print page!"),
    	  	QMessageBox::Ok | QMessageBox::Default);
	  	}
	  }
  }

	slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotSaveGeneric()
{

  if (_faxIndex < 1)
  {
    QMessageBox::information(0,i18n("Save Message As..."),
      i18n("You need to select the desired message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

  switch(_msgType)
  {
  	case Fax: 	slotSaveFax(); 		break;
  	case Voice: slotSaveVoice(); 	break;
  	default:
  		QMessageBox::information(0,i18n("Save Message As..."),
      i18n("You have chosen an unsupported message type!"),
      QMessageBox::Ok | QMessageBox::Default);
  }
}

void MessengerApp::slotSaveVoice()
{
  char filename[255] = "";
  QString saveFileName;

	slotStatusMsg(i18n("Save selected message..."));

  if (_faxIndex < 1)
  {
    QMessageBox::information(0,i18n("Save Message As..."),
      i18n("You need to select the desired message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

  if (_msgType != Voice)
  {
    QMessageBox::information(0,i18n("Save Message As..."),
      i18n("You need to select a voice message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

  saveFileName = KFileDialog::getSaveFileName();

	if (!saveFileName.isEmpty())
  {
  	debug("file will be stored in %s",saveFileName.data());
	  if (!modem->getMemoryFileName(filename))
	  {
	    if (!modem->getMemory())
	    {
  	    QMessageBox::information(0,i18n("Save Message As..."),
   	     i18n("Could not read modem memory!"),
   	     QMessageBox::Ok | QMessageBox::Default);
			  slotStatusMsg(IDS_DEFAULT);
   	   return;
	    }
  	}
	  slotGetVoice(saveFileName.data());
  	if (!modem->getVoiceFileName(filename))
	  {
  	  QMessageBox::information(0,i18n("Save Message As..."),
    	  i18n("Could not extract the selected message!"),
      	QMessageBox::Ok | QMessageBox::Default);
		  slotStatusMsg(IDS_DEFAULT);
  	  return;
	  }

  	debug("Message stored in %s.gsm",filename);
  }
	slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotPlayIntern()
{
	slotStatusMsg(i18n("Play selected message via modem..."));

  if (_faxIndex < 1)
  {
    QMessageBox::information(0,i18n("Play Message"),
      i18n("You need to select the desired message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

  if (_msgType != Voice)
  {
    QMessageBox::information(0,i18n("Play Message"),
      i18n("You need to select a voice message!"),
      QMessageBox::Ok | QMessageBox::Default);
	  slotStatusMsg(IDS_DEFAULT);
  	return;
  }

 	modem->playVoiceIntern(_faxIndex);

	slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotPlaySoundcard()
{

  bool success       = false;
  QString arg        = "";
  char filename[255] = "";
  KShellProcess	sound;

	slotStatusMsg(i18n("Play selected message via soundcard..."));

  if ((voiceHandler()) &&	(modem->getVoiceFileName(filename)))
  {
  	// now spawn voiceProcessor process with file as parameter
  	slotStatusMsg(i18n("playing message..."));
  	sound.clearArguments();
   	arg = "";
    arg.sprintf("%s %s.gsm > %s",configuration.voiceProcessor.data(),
    						filename,configuration.audioDevice.data());
 	  debug("spawn: %s",arg.data());
 	  //system(arg.data());
   	sound << arg.data();
  	success = sound.start(KProcess::DontCare);
	  if (success)
  	  slotStatusMsg(i18n("playing..."));
	  else
  	{
    	slotStatusMsg(i18n("Could not start process!"));

	    QMessageBox::information(0,i18n("Playing Message"),
  	   i18n("Could not play the selected message!"),
    	  QMessageBox::Ok | QMessageBox::Default);
	  }
  }
	slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotResetClock()
{
	slotStatusMsg(i18n("Reset Clock..."));

  int mode=KMsgBox::yesNo(this, i18n("Reset Modem Clock"),
  			i18n("Do you want to reset the modem clock?"));

  if(mode==1)
  	modem->resetClock();
  	
	slotStatusMsg(IDS_DEFAULT);
}


void MessengerApp::slotCommitSetup()
{
	slotStatusMsg(i18n("Commit Changes..."));
	
  int mode=KMsgBox::yesNo(this, i18n("Commit Changes"),
  			i18n("Do you want to save changes?"));

  if(mode==1)
  	modem->commitSetup();
  	
	slotStatusMsg(IDS_DEFAULT);
}

int MessengerApp::setModemControlValues(struct ModemControlStruct mcontrol,
																				struct ModemControlStruct mcsave)
{
	int changes = 0;
	
	slotStatusMsg(i18n("Write changes back to the modem..."));
	
	if (mcontrol.faxReception != mcsave.faxReception)
	{
		changes++;
  	if (mcontrol.faxReception)
  		modem->setFaxReception();
  	else
  		modem->unsetFaxReception();
 	}

	if (mcontrol.voiceReception != mcsave.voiceReception)
	{
		changes++;
	  if (mcontrol.voiceReception)
  		modem->setVoiceReception();
  	else
  		modem->unsetVoiceReception();
	}

	if (mcontrol.controlMonitor != mcsave.controlMonitor)
	{
		changes++;
	  if (mcontrol.controlMonitor)
  		modem->setControlMonitor();
  	else
  		modem->unsetControlMonitor();
	}

	if (mcontrol.numberOfRings != mcsave.numberOfRings)
	{
		changes++;
		modem->setSetupRings(mcontrol.numberOfRings);
	}

	if (mcontrol.voiceDuration != mcsave.voiceDuration)
	{
		changes++;
		modem->setVoiceDuration(mcontrol.voiceDuration);
	}
		
	if (mcontrol.faxId != mcsave.faxId)
	{
		changes++;
		modem->setSetupFaxId(mcontrol.faxId);
	}
	
	slotStatusMsg(IDS_DEFAULT);
	return changes;
}

void MessengerApp::getModemControlValues(struct ModemControlStruct &mcontrol,
																				struct ModemControlStruct &mcsave)
{
	slotStatusMsg(i18n("Get modem settings..."));
	
	mcontrol.faxReception 		= modem->getFaxReception();
	if (mcontrol.faxReception)
	  debug("true");
	else
		debug("false");
		
	mcontrol.voiceReception 	= modem->getVoiceReception();
	if (mcontrol.voiceReception)
	  debug("true");
	else
		debug("false");
	
	mcontrol.controlMonitor	= modem->getControlMonitor();
	if (mcontrol.controlMonitor)
	  debug("true");
	else
		debug("false");
		
	mcontrol.numberOfRings		= modem->getSetupRings();
	mcontrol.voiceDuration		= modem->getVoiceDuration();
	mcontrol.faxId.setStr(modem->getSetupFaxId()->data());
	
	// store settings in mcsave
	mcsave.faxReception = mcontrol.faxReception;
	mcsave.voiceReception = mcontrol.voiceReception;
	mcsave.controlMonitor = mcontrol.controlMonitor;
	mcsave.numberOfRings	= mcontrol.numberOfRings;
	mcsave.voiceDuration	= mcontrol.voiceDuration;
	mcsave.faxId.setStr(mcontrol.faxId);
	
	slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotModemControl()
{

	struct ModemControlStruct mcontrol;
	struct ModemControlStruct mcsave;
	
  slotStatusMsg(i18n("configure Messenger Modem..."));

  getModemControlValues(mcontrol, mcsave);
	ModemControl *dialog = new ModemControl(&mcontrol);
	
	int ret = dialog->exec();
	if (ret == QDialog::Accepted)
	{
		debug("OK pressed");
		if (setModemControlValues(mcontrol, mcsave) > 0)
		{
    	QMessageBox::information(0,i18n("Changed Settings"),
      	i18n("New settings were written to the modem.\n"
      			"If you want to make them permanent, please\nchoose "
      			"\"Commit Setup\" from the Modem Menu."),
      	QMessageBox::Ok | QMessageBox::Default);
    }
	} else {
		debug("Cancel pressed");
	}
	slotStatusMsg(IDS_DEFAULT);
}

void MessengerApp::slotSettings()
{
	QString tmpStr;
	int idx = 0;
  slotStatusMsg(i18n("configure Messenger..."));
	SettingsDialog *dialog = new SettingsDialog(configuration);
	int ret = dialog->exec();
	if (ret == QDialog::Accepted)
	{
		debug("OK pressed");
		// do something with the new settings
		configuration.serialLine = dialog->serialDevice->text();
		configuration.faxViewer = dialog->faxViewer->text();
		configuration.baudrate = dialog->baudrate->currentText();
		tmpStr = dialog->serialDevice->text();
		idx = tmpStr.findRev('/');
		configuration.lockfile.sprintf("%s/LCK..%s",LOCKPATH,
									tmpStr.right(tmpStr.length() - idx - 1 ).data());
    QMessageBox::information(0,i18n("Changed Settings"),
      i18n("New settings will take effect after restart."),
      QMessageBox::Ok | QMessageBox::Default);
	} else {
		debug("Cancel pressed");
	}
	slotStatusMsg(IDS_DEFAULT);
}


void MessengerApp::slotDisableMost()
{
	disableCommand(ID_FAX_POPUP_VIEW);
	disableCommand(ID_FAX_POPUP_SAVE);
	disableCommand(ID_FAX_POPUP_PRINT);
	disableCommand(ID_VOICE_POPUP_SAVE);
	disableCommand(ID_VOICE_POPUP_PLAY_INTERN);
	disableCommand(ID_VOICE_POPUP_PLAY_CARD);
	disableCommand(ID_GENERIC_POPUP_SAVE);
	disableCommand(ID_MODEM_GET_INFO);
	disableCommand(ID_MODEM_GET_MEMORY);
	disableCommand(ID_MODEM_CLEAR_MEMORY);
	disableCommand(ID_MODEM_RESET_CLOCK);
	disableCommand(ID_MODEM_COMMIT_SETUP);
	disableCommand(ID_MODEM_CONTROL);
}