/**
 * This is the KTimeclockListItem class.  It is used for displaying items in a
 * @ref QListView.  I've derived from the @ref QListViewItem class, and simply
 * added in the extra functionality which was required (primarily just
 * formatting of text strings, and some constants for defining columns).
 *
 * @short   KTimeclockListItem class
 * @author  Graham TerMarsch (gtermars@home.com)
 * @version $Revision: 1.3 $
 * @see     QListViewItem
 */

#include "KTimeclockListItem.h"

QString KTimeclockListItem::ITEM_PROJECT = "Project";
QString KTimeclockListItem::ITEM_TASK    = "Task";

/**
 * Sets the description for the list item.
 */
void KTimeclockListItem::description (const char* desc)
{
    if (desc)
    {
        this->setText( COL_DESCRIPTION, desc );
        this->repaint();
    }
}

/**
 * Gets the description for the list item and returns it.
 */
const char* KTimeclockListItem::description ()
{
    return this->text( COL_DESCRIPTION );
}

/**
 * Sets the total time spent (in seconds)
 */
void KTimeclockListItem::totalTime (int seconds)
{
    QString tmp;
    tmp.setNum( seconds );
    this->setText( COL_TOTAL, tmp );
    this->repaint();
}

/**
 * Returns the total time spent (in seconds)
 */
int KTimeclockListItem::totalTime ()
{
    QString tmp;
    tmp = this->text( COL_TOTAL );
    return tmp.toInt();
}

/**
 * Sets the time spent in this session (in seconds)
 */
void KTimeclockListItem::sessionTime (int seconds)
{
    QString tmp;
    tmp.setNum( seconds );
    this->setText( COL_SESSION, tmp );
    this->repaint();
}

/**
 * Gets the time spent during this session (in seconds)
 */
int KTimeclockListItem::sessionTime ()
{
    QString tmp;
    tmp = this->text( COL_SESSION );
    return tmp.toInt();
}

/**
 * Sets the type for this list item
 */
void KTimeclockListItem::type( QString type )
{
    this->setText( COL_TYPE, type );
}

/**
 * Gets the type for this list item
 */
QString KTimeclockListItem::type ()
{
    return QString( this->text( COL_TYPE ) );
}

/**
 * Overridden repaint method.
 * This version of repaint will first take the values for the number of
 * seconds spent in this session and in total and update the text columns
 * for the item.  We'll then call the base class to do the repainting once
 * we've got this set up.
 */
void KTimeclockListItem::repaint ()
{
    // IF we're an actual task, be sure to draw the amount of time spent.
    if (this->type() == ITEM_TASK)
    {
        QString tmp;

        // First do the session time
        tmp = this->text( COL_SESSION );
        this->setText( COL_TEXTSESSION, timeString( tmp.toInt() ) );

        // Then do the total time
        tmp = this->text( COL_TOTAL );
        this->setText( COL_TEXTTOTAL, timeString( tmp.toInt() ) );
    }

    // Finally, repaint by calling our base class
    QListViewItem::repaint();
}

/**
 * Converts a given uint to a string in the format of 'hh:mm:ss'.  Inlined
 * method (so we don't have another .cpp file kicking around somewhere).
 *
 * @return  QString containing formatted time value
 * @param   timestamp   time to convert to formatted text string
 */
QString KTimeclockListItem::timeString (int timestamp)
{
    int hours, minutes;

    hours = timestamp / 3600;
    timestamp %= 3600;
    minutes = timestamp / 60;
    timestamp %= 60;

    QString rtnval;
    rtnval.sprintf( "%02d:%02d:%02d", hours, minutes, timestamp );
    return rtnval;
}

/**
 * Sets default values for our columns and provides us a pixmap depending
 * as to what type of item we are.  Internal method only.
 */
void KTimeclockListItem::setDefaults ( const char* type, const char* desc, int totalsecs )
{
    this->setText( COL_DESCRIPTION, desc );
    this->setText( COL_TYPE, type );

    if (ITEM_PROJECT == type)
    {
        this->setPixmap( COL_DESCRIPTION,
                         QPixmap( kapp->kde_icondir() + "/mini/folder.xpm" )
                       );
    }
    else if (ITEM_TASK == type)
    {
        QString tmp;
        tmp.setNum( totalsecs );
        this->setPixmap( COL_DESCRIPTION,
                         QPixmap( kapp->kde_icondir() + "/mini/document.xpm" )
                       );
        this->setText( COL_TEXTTOTAL, this->timeString( totalsecs ) );
        this->setText( COL_TOTAL, tmp );
    }
}
