// ----------------------------------------------------------------------------
// File:    KTimeclockData.h
// $Date: 1999/06/10 15:45:59 $
// $Revision: 1.6 $
// ----------------------------------------------------------------------------

#ifndef KTimeclockData_included
#define KTimeclockData_included

#include <qwidget.h>
#include <qpopupmenu.h>
#include "KTimeclockListItem.h"

/**
 * Tree-view display for managing time spent; data class.
 *
 * Base class for the tree-view display to allow for managing of time spent on
 * various tasks.  This class is responsible for creating the display, and for
 * holding all of the data items.  All of the functionality provided by the
 * class is implemented in KTimeclock.
 *
 * @author  Graham TerMarsch (gtermars@home.com)
 * @version $Revision: 1.6 $
 * @see     KTimeclock
 * @see     KTimeclockListItem
 */
class KTimeclockData : public QWidget
{
    Q_OBJECT

public:
    /**
     * Constructor.
     *
     * @param   parent  Parent widget
     * @param   name    Name of this widget
     */
    KTimeclockData( QWidget* parent = NULL, const char* name = NULL );

    /**
     * Destructor.
     */
    virtual ~KTimeclockData ();

public slots:
    /**
     * Adds a new project.  This function will pop-up a dialog box to ask for
     * all of the relevant information on the project and then add it to the
     * tree-view as needed.
     */
    virtual void AddProject ();

    /**
     * Adds a new sub-project.  This function will pop-up a dialog box to ask
     * for all of the relevant information on the sub-project and then add it
     * to the tree-view as needed.
     *
     * Checking is done to ensure that a project/sub-project is already
     * selected for the new sub-project to be a child of.  If no
     * project/sub-project is selected, an appropriate error message will be
     * displayed to the user.
     */
    virtual void AddSubProject ();

    /**
     * Adds a new task.  This function will pop-up a dialog box to ask for all
     * of the relevant information on the sub-project and then add it to the
     * tree-view as needed.
     *
     * Checking is done to ensure that a project/sub-project is already
     * selected for the new task to be a child of.  If no project/sub-project
     * is selected, an appropriate error message will be displayed to the user.
     */
    virtual void AddTask ();

    /**
     * Edits the currently selected item.  This function will pop-up an
     * appropriate dialog box to allow for editing of the selected item, and
     * will update the information shown in the tree-view as needed.  If no
     * item is currently selected, an appropriate error message will be
     * displayed to the user.
     */
    virtual void EditItem ();

    /**
     * Confirms and then deletes the currently selected item and _all_ child
     * items.  This function will pop-up a dialog box asking the user that
     * he/she does wish to delete the currently selected item and all of it's
     * child items.  If confirmed, the item and it's children are removed from
     * the tree-view.  If no item is currently selected, an appropriate error
     * message will be displayed to the user.
     */
    virtual void DeleteItem ();

    /**
     * Clears all of the time currently associated with a given item.  Note,
     * that this clearing is NOT recursive, and that we don't clear the time
     * for any child items.
     */
    virtual void ClearItem ();

    /**
     * Clears all of the time currently associated with ALL of the items.  This
     * update affects every single task that is present.
     */
    virtual void ClearAllItems ();

    /**
     * Currently does nothing.
     */
    virtual void Preferences ();

    /**
     * Starts the timer clock.
     */
    virtual void StartTimer ();

    /**
     * Stops the timer clock.
     */
    virtual void StopTimer ();

    /**
     * Saves all of the information currently held in the tree-view to our
     * data file.  This function returns zero if no errors occurred while
     * saving the information out to file, and non-zero if some error occurred.
     *
     * @return  Did any errors occur?
     */
    virtual int save();

    /**
     * Loads information from our data file and places it into the tree-view. 
     * This function returns zero if no errors occurred while loading
     * information from the file, and non-zero if some error occurred.
     *
     * @return  Did any errors occur?
     */
    virtual int load();

signals:
    /**
     * Timer has started.  When the timer has been started, this signal is
     * emitted.
     */
    void timerStarted ();

    /**
     * Timer has stopped.  When the timer has been stopped, this signal is
     * emitted.
     */
    void timerStopped ();

    /**
     * Timer state has changed.  When the timer is started or stopped, this
     * signal is emitted to signify that it's state has changed.  Any widget
     * which wishes to display the current state of the timer (e.g. enabled,
     * disabled) should attach to this signal.
     *
     * @param   state   Current state of timer.
     */
    void timerState( QString& state );

    /**
     * Selection has changed.  When a new item has been selected from the
     * tree-view, this signal is emitted.  Any widget which wishes to show the
     * currently selected item from the tree-view should attach to this signal.
     *
     * @param   item    Name of the newly selected item.
     */
    void itemSelected( QString& item );

protected slots:
    /**
     * Internal function.  Used to gather information required in order to emit
     * a signal stating that the current item in the tree-view has changed.
     *
     * @param   item    Handle to the widget containing the newly selected item.
     */
    virtual void SelectionChanged( QListViewItem* item );

    /**
     * Internal function.  Displays a pop-up menu allowing for the user to
     * edit/delete the currently selected item.
     *
     * @param   item    Handle to the currently selected item's widget.
     * @param   loc     Co-ordinates of where the mouse button was clicked
     * @param   col     Column which was clicked (unused)
     */
    virtual void ShowPopup( QListViewItem*, const QPoint&, int col );

protected:
    /**
     * Internal data item; handle to the widget which is our tree-view of the
     * projects/sub-projects/tasks.
     */
    QListView*  treeProjects;

    /**
     * Internal data item; ID number of the currently active timer.
     */
    int         timerID;

    /**
     * Internal data item; string representing the current state of the timer
     * (enabled or disabled).
     */
    QString     state;

    /**
     * Internal data item; handle to a popup menu widget which we'll display
     * when the user right-clicks on an item in the tree-view.
     */
    QPopupMenu* mnuPopup;

    /**
     * Internal function; called whenenever a timer event occurs.  We use this
     * function to update everything when another second has passed and our
     * timer tells us so.
     *
     * @param   event   Event as issued by our timer
     */
    virtual void timerEvent( QTimerEvent* event );
};

#endif // KTimeclockData_included
