/***************************************************************************
                          knd_gra.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include	"knd_gra.h"

/*  intToExp	: Convert value to xEy equivalent			*/
/*  value	: int		: Value in question			*/
/*  buff	: char *	: Text buffer				*/
/*  (returns)	: char *	: Text buffer				*/

static	char	*intToExp
	(	int	value,
		char	*buff
	)
{
	/* Note that this routine assumes that the value can be		*/
	/* expressed in the form xEy where x is an integer in the range	*/
	/* 1...9. If not, then it will not work properly, on account of	*/
	/* the integer truncation in the final division.		*/
	int	p10	;
	int	mul	;
	int	exp	;

	for (p10 = 1, exp = 0 ; p10 * 10 <= value ; p10 *= 10, exp += 1) ;
	mul	= value / p10 ;

	sprintf	(buff, "%dE%d", mul, exp) ;
	return	buff ;

}

/*  KNDGraphic								*/
/*  KNDGraphic	: Constructor for graphic display			*/
/*  _parent	: KNDView *	: Parent object				*/
/*  (returns)	: KNDGraphic	:					*/

KNDGraphic::KNDGraphic
	(	KNDView	*_parent
	)
	:
	QFrame	((QWidget *)_parent),
	g_load	(this),
	g_bar	(this),
	g_count	(this),
	lhmax	(this),
	lhmin	(this),
	pHist1	(this),
	pBar1	(this),
	pBar2	(this),
	pcmax	(this),
	pcmin	(this)
{
	going	  = false ;
	frozen	  = false ;

	loadHist  = NULL  ;
	loadBar1  = NULL  ;
	loadBar2  = NULL  ;
	pktHist	  = NULL  ;

	lhmax  .hide () ;
	lhmin  .hide () ;
	pHist1 .hide () ;
	pBar1  .hide () ;
	pBar2  .hide () ;
	pcmax  .hide () ;
	pcmin  .hide () ;
	g_load .hide () ;
	g_bar  .hide () ;
	g_count.hide () ;
}

/*  KNDGraphic								*/
/*  ~KNDGraphic	: Destructor for graphic display			*/
/*  (returns)	:		:					*/

KNDGraphic::~KNDGraphic ()
{
	if (loadHist != NULL) delete loadHist ;
	if (loadBar1 != NULL) delete loadBar1 ;
	if (loadBar2 != NULL) delete loadBar2 ;
	if (pktHist  != NULL) delete pktHist  ;
}

/*  KNDGraphic								*/
/*  setOptions	: Set display options					*/
/*  _v_hist	: int		: Time constant for load histogram	*/
/*  _v_bar1	: int		: Time constant for load bar 1		*/
/*  _v_bar2	: int		: Time constant for load bar 2		*/
/*  (returns)	: void		:					*/

void	KNDGraphic::setOptions
	(	int	_v_load,
		int	_v_bar1,
		int	_v_bar2
	)
{
#define	O_LOADHIST	(50)
#define	W_LOADHIST	(250)
#define	O_LOADBAR1	(O_LOADHIST+W_LOADHIST+35)
#define	W_LOADBAR1	(40)
#define	O_LOADBAR2	(O_LOADBAR1+W_LOADBAR1+15)
#define	W_LOADBAR2	(40)
#define	O_PKTHIST	(O_LOADBAR2+W_LOADBAR2+70)
#define	W_PKTHIST	(250)

	QRect	gRect	= geometry () ;
	char	text[32];

	g_load .setGeometry (O_LOADHIST-45,
			     5,
			     W_LOADHIST+60,
			     gRect.height()-10) ;
	g_bar  .setGeometry (O_LOADHIST-40+W_LOADHIST+60,
			     5,
			     O_PKTHIST -50-(O_LOADHIST-40+W_LOADHIST+60),
			     gRect.height()-10) ;
	g_count.setGeometry (O_PKTHIST -45,
			     5,
			     W_PKTHIST +60,
			     gRect.height()-10) ;
	g_load .setTitle    ("Load: bytes/sec"  ) ;
	g_bar  .setTitle    (" ") ;
	g_count.setTitle    ("Load: packets/sec") ;
	g_load .show	    () ;
	g_count.show	    () ;
	g_bar  .show	    () ;

	v_hist		= _v_load ;
	v_bar1		= _v_bar1 ;
	v_bar2		= _v_bar2 ;
	loadScale	= MINLOAD ;
	pktScale	= MINPCNT ;

	/* Destroy the existing load histograms and load bars if such	*/
	/* exist ....							*/
	if (loadHist != NULL) delete loadHist ;
	if (loadBar1 != NULL) delete loadBar1 ;
	if (loadBar2 != NULL) delete loadBar2 ;
	if (pktHist  != NULL) delete pktHist  ;

	/* Construct a new load histogram of suitable size, and place	*/
	/* the labels (scale minimum, maximum and width of the		*/
	/* histogram) around it. The minimum and width labels are known	*/
	/* so can be set; the maximum is set arbitrarily to start with	*/
	loadHist = new KNDLoadHist (this, O_LOADHIST,
					  25,
					  W_LOADHIST,
					  gRect.height() - 55,
					  v_hist) ;

	lhmax	.setGeometry (O_LOADHIST-35, 20,	 	  25, 15) ;
	lhmin	.setGeometry (O_LOADHIST-35, gRect.height() - 40, 25, 15) ;
	pHist1	.setGeometry (O_LOADHIST + W_LOADHIST/2 - 30,
				  	     gRect.height() - 25, 50, 15) ;
	lhmax	.show    ()  ;
	lhmin	.show    ()  ;
	pHist1	.show    ()  ;

	sprintf	(text, "%d secs", W_LOADHIST * v_hist) ;
	pHist1	.setText (text) ;
	lhmax	.setText (intToExp (loadScale, text)) ;
	lhmin	.setText ("0" ) ;

	/* Now create the load bars. These just have a label under each	*/
	/* to show the time constant.					*/
	loadBar1  = new KNDLoadBar  (this, O_LOADBAR1,
					   25,
					   W_LOADBAR1,
					   gRect.height() - 55,
					   v_bar1) ;

	sprintf	(text, "%d", v_bar1) ;

	pBar1	.setGeometry (O_LOADBAR1+15, gRect.height() - 25, 25, 15) ;
	pBar1	.show	     () ;
	pBar1	.setText     (text) ;

	loadBar2  = new KNDLoadBar  (this, O_LOADBAR2,
					   25,
					   W_LOADBAR2,
					   gRect.height() - 55,
					   v_bar2) ;
	sprintf	(text, "%d", v_bar2) ;

	pBar2	.setGeometry (O_LOADBAR2+15, gRect.height() - 25, 25, 15) ;
	pBar2	.show	     () ;
	pBar2	.setText     (text) ;

	/* Thirdly, create the packet count histogram and position its	*/
	/* labels.							*/
	pktHist	= new KNDPktHist  (this, O_PKTHIST,
					 25,
					 W_PKTHIST,
					 gRect.height() - 55,
					 v_hist) ;

	pcmax	.setGeometry (O_PKTHIST-35, 20,		         25, 15) ;
	pcmin	.setGeometry (O_PKTHIST-35, gRect.height() - 40, 25, 15) ;
	pcmax	.show        ()  ;
	pcmin	.show        ()  ;

	pcmax	.setText (intToExp (pktScale, text)) ;
	pcmin	.setText ("0" ) ;


	/* Repaint to get all the controls displayed correctly to start	*/
	/* with, then clear the going and frozen flags preparatory to	*/
	/* starting.							*/
	repaint () ;
	going	= false ;
	frozen	= false ;
}

/*  KNDGraphic								*/
/*  execute	: Set executing flag					*/
/*  _going	: bool		: Executing				*/
/*  (returns)	: void		:					*/

void	KNDGraphic::execute
	(	bool	_going
	)
{
	going	= _going ;
	frozen	= false	 ;
}

/*  KNDGraphic								*/
/*  timerTick	: Handle a one-second timer tick			*/
/*  second	: long		: Second				*/
/*  (returns)	: void		:					*/

void	KNDGraphic::timerTick
	(	long	second
	)
{
	if (going && (loadHist != NULL))
	{
		/* First stage is to pass the timer tick down to each	*/
		/* of the load objects, so that they can perform their	*/
		/* per-tick processing.					*/
		int	lScale	= loadHist->timerTick (second) ;
		int	pScale	= pktHist ->timerTick (second) ;
		char	text[32];

		if (lScale != loadScale)
			lhmax.setText (intToExp (loadScale = lScale, text)) ;
		if (pScale != pktScale )
			pcmax.setText (intToExp (pktScale  = pScale, text)) ;

		loadBar1->timerTick (second) ;
		loadBar2->timerTick (second) ;

		/* If this graphic display object is on view and we are	*/
		/* not frozen, then repaint the displays. If not, then	*/
		/* there is no point in doing anything.			*/
		if (onview && !frozen) paintAll () ;
	}
}

/*  KNDGraphic								*/
/*  paintAll	: Paint everything					*/
/*  (returns)	: void		:					*/

void	KNDGraphic::paintAll ()
{
	/* The scale determined by the load histogram is applied to	*/
	/* the bar displays. Since the former is scaled according to	*/
	/* the peak load within its total period, and since its total	*/
	/* period is greater than the time constant for the bar		*/
	/* displays, their values cannot exceed that of the former.	*/
	loadBar1 ->setScale (loadScale) ;
	loadBar2 ->setScale (loadScale) ;

	/* All ready to go, now redisplay all of the load displays.	*/
	loadHist->repaint () ;
	loadBar1->repaint () ;
	loadBar2->repaint () ;
	pktHist ->repaint () ;
}

/*  KNDGraphic								*/
/*  addPacket	: Add a new packet to the display			*/
/*  pktinfo	: PktInfo *	  : Packet information			*/
/*  (returns)	: void		  :					*/

void	KNDGraphic::addPacket
	(	PktInfo		*pktinfo
	)
{
	loadHist->addPacket (pktinfo) ;
	loadBar1->addPacket (pktinfo) ;
	loadBar2->addPacket (pktinfo) ;
	pktHist ->addPacket (pktinfo) ;
}

/*  KNDGraphic								*/
/*  setOnview	: Set whether on view or not				*/
/*  _onview	: bool		: New state				*/
/*  (returns)	: void		:					*/

void	KNDGraphic::setOnview
	(	bool	_onview
	)
{
	/* Note the state, and repaint if we are coming into view ...	*/
	if ((onview = _onview) && going) paintAll () ;
}

/*  KNDGraphic								*/
/*  freeze	: Set frozen or unfrozen state				*/
/*  _frozen	: bool		: New state				*/
/*  (returns)	: void		:					*/

void	KNDGraphic::freeze
	(	bool	_frozen
	)
{
	frozen	= _frozen ;
	if (loadHist != NULL) loadHist->freeze (frozen) ;
	if (loadBar1 != NULL) loadBar1->freeze (frozen) ;
	if (loadBar2 != NULL) loadBar2->freeze (frozen) ;
}
