/*
 *   kscan - a scanning program
 *   Copyright (C) 1998 Ivan Shvedunov
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <qimage.h>
extern "C" {
#include <sane.h>
}
#include <stdio.h>
#include <math.h>
#include <kapp.h>
#include "scan.h"
#include <dirent.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <qmsgbox.h>

SANE_Int options[NUMREQ];
SANE_Handle h;

char *required_options[NUMREQ] = {
	"mode",
	"brightness",
	"contrast",
	"resolution",
	"tl-x",
	"tl-y",
	"br-x",
	"br-y",
	"resolution-bind"
};

//Backend description
static bool sp;
static bool have_brightness,have_contrast;
static bool fixed_brightness,fixed_contrast;
static bool fixed_resolution;
static int max_brightness,max_contrast;
static bool have_color_mode,have_gray_mode;
static bool resolution_has_quant;
static bool brightness_has_quant,contrast_has_quant;
static int resolution_quant,brightness_quant,contrast_quant;
static char gray_string[101],color_string[101];
static bool emulate_brightness = FALSE,emulate_contrast = FALSE;

void testDir(const char *_name) //From karm.cpp
{
	DIR *dp;
	QString c = getenv( "HOME" );
	c += _name;
	dp = opendir( c.data() );
	if ( dp == NULL )
		::mkdir(c.data(),0755);
	else
		closedir( dp );
}

void ScanImage::grayify()
{
	int sz = p.pixels_per_line*p.lines;
	SANE_Byte *data2 = new SANE_Byte[sz];
	for(int i = 0,j = 0; j<sz; j++) {
		unsigned d = data[i++]*11;
		d += data[i++]*16;
		d += data[i++]*5;
		data2[j] = (SANE_Byte)(d>>5);
	}
	delete data;
	data = data2;
	p.bytes_per_line = p.pixels_per_line;
	p.depth = 8;
}

void ScanImage::colorify()
{
	int sz = p.pixels_per_line*p.lines,sz2 = sz*3;
	SANE_Byte *data2 = new SANE_Byte[sz2];
	for(int i = 0,j = 0; j<sz; j++) {
		int d = data[j];
		data2[i++] = d;
		data2[i++] = d;
		data2[i++] = d;
	}
	for(int i = 0; i<20; i++)
		printf("%d\n",data2[i]);
	delete data;
	data = data2;
	p.bytes_per_line = p.pixels_per_line*3;
	p.depth = 24;
}

bool ScanImage::brightnessEmulated()
{
	return emulate_brightness;
}

bool ScanImage::contrastEmulated()
{
	return emulate_contrast;
}

void ScanImage::forceBrightnessEmulation(bool f)
{
	emulate_brightness = f;
}

void ScanImage::forceContrastEmulation(bool f)
{
	emulate_contrast = f;
}

ScanImage::ScanImage(scan_type t,bool _preview = TRUE):
	QObject(),img(NULL),data(NULL),preview(_preview)
{
	cur_brightness = cur_contrast = 0;
	type = t;
	pp = FALSE;
	if(preview) {
		//First, try to read our data
		testDir("/.kde");
		testDir("/.kde/share");
		testDir("/.kde/share/config");
		testDir("/.kde/share/apps");
		testDir("/.kde/share/apps/kscan");
		QString fnm = getenv("HOME");
		fnm += "/.kde/share/apps/kscan/preview.dat";
		QFile *r;
		bool rootfile = FALSE;
		r = preview_data = new QFile(fnm.data());
		if(!preview_data->open(IO_ReadOnly)) {
			QString fnm2 = kapp->kde_datadir();
			fnm2 += "/kscan/preview.dat";
			r = new QFile(fnm2.data());
			if(!r->open(IO_ReadOnly)) {
				if(!sp) {
					QMessageBox::critical(NULL,
							      "Error",
							      "No scanner "
							      "and no "
							      "preview.dat");
					sane_close(h);
					sane_exit();
					exit(1);
				}
				r = NULL;
			} else rootfile = TRUE;
		}
		if(r) {
			QDataStream d(r);
			double a1,a2,a3,a4;
			int tt;
			d>>a1>>a2>>a3>>a4>>tt>>resolution;
			type = (scan_type)tt;
			d.readRawBytes((char *)&p,sizeof(p));
			t_tlx = SANE_FIX(a1);
			t_tly = SANE_FIX(a2);
			t_brx = SANE_FIX(a3);
			t_bry = SANE_FIX(a4);
			int sz = p.bytes_per_line*p.lines;
			printf("Reading %d bytes\n",sz);
			data = new SANE_Byte[sz];
			d.readRawBytes((char *)data,sz);
			r->close();
			if(rootfile) delete r;
			pp = TRUE;
			if(type==COLOR&&!have_color_mode) {
				grayify();
				type = GRAY;
				printf("grayified\n");
			} else if(type==GRAY&&!have_gray_mode) {
				colorify();
				type = COLOR;
			}
			if(type==COLOR)
				img = new QImage(p.pixels_per_line,p.lines,32);
			else {
				img = new QImage(p.pixels_per_line,p.lines,
						 8,256);
				for(int i = 0; i<256; i++)
					img->setColor(i,qRgb(i,i,i));
			}
			printf("*FORMAT* : %d\n"
			       "last_frame : %d\nlines : %d\ndepth : %d\n"
			       "pixels_per_line : %d\nbytes_per_line : %d\n",
			       p.format,p.last_frame,p.lines,p.depth,
                               p.pixels_per_line,
			       p.bytes_per_line);
			if(sp) setResolution(resolution);
		}
	}
	if(sp) {
		set_auto(TLX);
		set_auto(TLY);
		set_auto(BRX);
		set_auto(BRY);
		setMode(type==GRAY?"Gray":"Color");
	}
}

void ScanImage::save()
{
	if(preview&&data&&preview_data->open(IO_Truncate|IO_WriteOnly)) {
		SANE_Fixed tlx,tly,brx,bry;
		getArea(tlx,tly,brx,bry);
		QDataStream d(preview_data);
		d<<SANE_UNFIX(tlx)<<SANE_UNFIX(tly)<<SANE_UNFIX(brx)
		 <<SANE_UNFIX(bry);
		d<<type<<resolution;
		d.writeRawBytes((char *)&p,sizeof(p));
		d.writeRawBytes((char *)data,p.lines*p.bytes_per_line);
		preview_data->close();
	}
}

ScanImage::~ScanImage()
{
	if(img) delete img;
	if(data) delete data;
}

static bool checkNumericOption(int num,const SANE_Option_Descriptor *d,
			       bool enable_backend_emulation,
			       int &min,int &max,int &quant,
			       bool &fixed,bool &has_quant)
{
	if(!num||d->constraint_type!=SANE_CONSTRAINT_RANGE)
		return FALSE;
	fixed = FALSE;
	if(d->type==SANE_TYPE_FIXED) fixed = TRUE;
	else if(d->type!=SANE_TYPE_INT) return FALSE;
	min = d->constraint.range->min;
	max = d->constraint.range->max;
	has_quant = FALSE;
	if((quant = d->constraint.range->quant)!=0)
		has_quant = TRUE;
	//Now, check capabilities
	if(SANE_OPTION_IS_ACTIVE(d->cap)&&
	   SANE_OPTION_IS_SETTABLE(d->cap)&&
	   enable_backend_emulation||!(d->cap&SANE_CAP_EMULATED))
		return TRUE;
	return FALSE;
}

bool ScanImage::find_options()
{
	SANE_Int n;
	const SANE_Option_Descriptor *d,*dd[NUMREQ];
	printf("find_options()\n");
	sane_control_option(h,0,SANE_ACTION_GET_VALUE,&n,NULL);
	memset(options,0,sizeof(SANE_Int)*NUMREQ);
	for(int i = 1; i<n; i++) {
		d = sane_get_option_descriptor(h,i);
		if(!d) break;
		if(!d->name) continue;
		for(int j = 0; j<NUMREQ; j++)
			if(!strcmp(d->name,required_options[j])) {
				options[j] = i;
				dd[j] = d;
			}
	}
	if(!options[RESOLUTION]||
	   !options[TLX]||!options[TLY]||
	   !options[BRX]||!options[BRY]||
	   !options[MODE]) {
		printf("Inappropriate backend!!!\n");
		sane_close(h);
		return FALSE;
	}
	have_color_mode = FALSE;
	//Now, we should check if backend is good enough to work with KScan
	//and record some info about it
	//If brightness or/and contrast are not supproted we emulate them
	//FIXME : check gamma option
	//It would be fine if we can make use of every backend's
	//brightness/contrast options. But they are too different...
	int a,b;
	have_brightness = checkNumericOption(options[BRIGHTNESS],
					     dd[BRIGHTNESS],FALSE,a,b,
					     brightness_quant,
					     fixed_brightness,
					     brightness_has_quant);
	if(have_brightness) {
		a = abs(a);
		if(abs(a-b)>1||a<MIN_SUPPORTED_BRIGHTNESS)
			have_brightness = FALSE;
		else max_brightness = a<b?a:b;
	}
	have_contrast = checkNumericOption(options[CONTRAST],
					     dd[CONTRAST],FALSE,a,b,
					     contrast_quant,
					     fixed_contrast,
					     contrast_has_quant);
	if(have_contrast) {
		a = abs(a);
		if(abs(a-b)>1||a<MIN_SUPPORTED_CONTRAST)
			have_contrast = FALSE;
		else max_contrast = a<b?a:b;
	}
	if(!checkNumericOption(options[RESOLUTION],dd[RESOLUTION],TRUE,a,b,
			       resolution_quant,fixed_resolution,
			       resolution_has_quant))
		return FALSE;
	//FIXME: check resolution option properties
	//FIXME: check tlx/tly/brx/bry properties
	//FIXME: add an ability to express tlx/tly/brx/bry in pixels
	if(dd[TLX]->unit!=SANE_UNIT_MM) {
		printf("Sorry, tlx/tly/brx/bry must be expressed in mms\n");
		sane_close(h);
		return FALSE;
	}
	//FIXME: check MODE capabilities!
	if(dd[MODE]->constraint_type!=SANE_CONSTRAINT_STRING_LIST) {
		printf("Unsupported MODE constraints\n");
		sane_close(h);
		return FALSE;
	}
	have_gray_mode = have_color_mode = FALSE;
	for(char **sl = (char **)dd[MODE]->constraint.string_list;
	    *sl; sl++) {
		if(!stricmp(*sl,"Gray")) {
			strncpy(gray_string,*sl,100);
			have_gray_mode = TRUE;
		}
		if(!stricmp(*sl,"Color")) {
			strncpy(color_string,*sl,100);
			have_color_mode = TRUE;
		}
	}
	if(!have_gray_mode&&!have_color_mode) return FALSE;
	if(options[RESOLUTION_BIND]) {
		bool a = TRUE;
		sane_control_option(h,options[RESOLUTION_BIND],
				    SANE_ACTION_SET_VALUE,
				    (void *)&a,NULL);
	}
	return TRUE;
}

void ScanImage::getArea(SANE_Fixed &tlx,SANE_Fixed &tly,
			SANE_Fixed &brx,SANE_Fixed &bry)
{
	if(sp) {
		get_option(TLX,&tlx);
		get_option(TLY,&tly);
		get_option(BRX,&brx);
		get_option(BRY,&bry);
	} else tlx = t_tlx, tly = t_tly, brx = t_brx, bry = t_bry;
}

void ScanImage::setMode(char *mode)
{
	set_option(MODE,mode);
}

void ScanImage::setBrightness(int brightness)
{
	if(!sp||!have_brightness||emulate_brightness) {
		cur_brightness = brightness;
		return;
	}
	double a = (double)brightness*(double)max_brightness/
		(double)BRIGHTNESS_RANGE;
	brightness = fixed_brightness?SANE_FIX(a):(int)a;
	if(brightness_has_quant)
		brightness -= brightness%brightness_quant;
	set_option(BRIGHTNESS,&brightness);
}

void ScanImage::setType(scan_type t)
{
	if(t==GRAY&&!have_gray_mode||
	   t==COLOR&&!have_color_mode) return;
	type = t;
	setMode(type==GRAY?gray_string:color_string);
}

bool ScanImage::scannerPresent()
{
	return sp;
}

bool ScanImage::haveColorMode()
{
	return have_color_mode;
}

bool ScanImage::haveGrayMode()
{
	return have_gray_mode;
}

void ScanImage::setContrast(int contrast)
{
	if(!sp||!have_contrast||emulate_contrast) {
		cur_contrast = contrast;
		return;
	}
	double a = (double)contrast*(double)max_contrast/
		(double)CONTRAST_RANGE;
	contrast = fixed_contrast?SANE_FIX(a):(int)a;
	if(contrast_has_quant)
		contrast -= contrast%contrast_quant;
	set_option(CONTRAST,&contrast);
}

void ScanImage::setResolution(int dpi)
{
	printf("setting resolution to %d\n",dpi);
	resolution = dpi;
	if(fixed_resolution)
		dpi = SANE_FIX((double)dpi);
	if(resolution_has_quant)
		dpi -= dpi%resolution_quant;
	set_option(RESOLUTION,&dpi);
}

SANE_Status ScanImage::set_option(int opt,void *val)
{
	return sane_control_option(h,options[opt],
				   SANE_ACTION_SET_VALUE,val,NULL);
}

void ScanImage::get_option(int opt,void *val)
{
	sane_control_option(h,options[opt],SANE_ACTION_GET_VALUE,val,NULL);
}

void ScanImage::set_auto(int opt)
{
	sane_control_option(h,options[opt],SANE_ACTION_SET_AUTO,NULL,NULL);
}

void ScanImage::receive()
{
	int len;
	if(sane_read(h,dat,1000,&len) == SANE_STATUS_EOF) {
		done++;
		return;
	}
	dat += len;
	dlg->setProgress(dlg->progress()+len);
}

int ScanImage::acquire()
{
	setMode(type==GRAY?"Gray":"Color");
	sane_start(h);
	sane_get_parameters(h,&p);
	
	printf("format : %d\nlast_frame : %d\nlines : %d\ndepth : %d\n"
	       "pixels_per_line : %d\nbytes_per_line : %d\n",
	       p.format,p.last_frame,p.lines,p.depth,p.pixels_per_line,
	       p.bytes_per_line);
	
	if(img) delete img;
	if(type==COLOR)
		img = new QImage(p.pixels_per_line,p.lines,32);
	else {
		img = new QImage(p.pixels_per_line,p.lines,8,256);
		for(int i = 0; i<256; i++)
			img->setColor(i,qRgb(i,i,i));
	}
	maxlen = p.bytes_per_line*p.lines;
	if(data) delete data;
	data = new SANE_Byte[maxlen];
	dlg = new QProgressDialog("Acquiring RGB data...",
				  "Cancel",
				  maxlen,NULL,NULL,TRUE);
	dlg->setProgress(0);
	dlg->show();
	kapp->processEvents();
	dat = data;
	done = 0;
	while(!done) {
		receive();
		kapp->processEvents();
	}
	delete dlg;
	sane_cancel(h);
	return TRUE;
}

inline int adjust(int x,int g,int b,int c)
{
	//printf("x : %d, x/256 : %lg, g : %d, 100/g : %lg",
	//       x,(double)x/256.0,g,100.0/(double)g);
	x = (int)(256*pow((double)x/256.0,100.0/(double)g));
        x = ((65536/(128-c)-256)*(x-128)>>8)+128+b;
	if(x<0) x = 0; else if(x>255) x = 255;
	//printf(" -> %d\n",x);
	return x;
}

void ScanImage::adjust_and_convert(int gamma = 100,int brightness = 0,
				   int contrast = 0)
{
	//Convert raw scanned data to QImage Gray/RGB data
	if(!brightness&&(!have_brightness||emulate_brightness))
		brightness = cur_brightness;
	if(!contrast&&(!have_contrast||emulate_contrast))
		contrast = cur_contrast;
	printf("CONVERTING: brightness %d, contrast %d\n",brightness,contrast);
	SANE_Byte *dt = data;
	int convtab[256];
	if(!gamma) gamma++;
	int count[256];
	//int eqtab[256];
	memset(count,0,256*sizeof(int));
	brightness = (brightness<<8)/(128-contrast);
	for(int i = 0; i<256; i++)
		convtab[i] = adjust(i,gamma,brightness,contrast);

	/*
	  for(int i = 0; i<maxlen; i += 3) {
	        int     r = convtab[*dt++],
	        g = convtab[*dt++],
	        b = convtab[*dt++];
		//QColor c(r,g,b);
		//int h,s,v;
		//c.hsv(&h,&s,&v);
		//if(!(i%2000)) printf("v = %d\n",v);
		count[(r+g+b)/3]++;
	}
	dt = data;
        int sum = 0;
	int step = maxlen/(3*256);
	*eqtab = 0;
	for(int i = 1; i<256; i++) {
		eqtab[i] = sum/step;
		sum += count[i];
		printf("sum = %d, count[%d] = %d, eqtab[%d] = %d\n",
		       sum,i,count[i],i,eqtab[i]);
	}
	*/

	for(int i = 0; i<p.lines; i++) {
		uint *scanline = (uint *)img->scanLine(i);
		for(int j = 0; j<p.pixels_per_line; j++)
			if(type==COLOR) {
				int     r = convtab[*dt++],
					g = convtab[*dt++],
					b = convtab[*dt++];
				/*int I = (r+g+b)/3;
				  QColor c(r,g,b);
				  int h,s,v;
				  c.hsv(&h,&s,&v);
				  c.setHsv(h,s,eqtab[v]);
				  c.rgb(&r,&g,&b);*/
				//r = eqtab[r];
				//g = eqtab[g];
				//b = eqtab[b];
				scanline[j] = qRgb(r,g,b);
			} else ((unsigned char *)scanline)[j] = convtab[*dt++];
	}
}

bool ScanImage::InitScanner(char *backend)
{
	static int first_time = TRUE;
	sp = TRUE;
	printf("InitScanner()\n");
	if(first_time&&sane_init(NULL,NULL)||
	   sane_open(backend,&h)||
	   !find_options()) sp = FALSE;
	if(!sp) have_gray_mode = FALSE, have_color_mode = TRUE;
	first_time = FALSE;
	return sp;
}

void ScanImage::CloseScanner()
{
	sane_close(h);
	sane_exit();
	exit(0);
}

#include "scan.moc"
