#include <qstring.h>
#include <qregexp.h>
#include <psion.h>
#include <qdatetime.h>
#include "ppsocket.h"

const int PSION_PORT_DEFAULT = 7501;

KProtocolPsion::KProtocolPsion()
  : KProtocol(),
    filename(),
    entry()
{
  warning( "Creating psion protocol" );

  readingDir = false;
  allowHTML = false;
  psionSocket = new ppsocket();
  psionSocket->connect( 0, PSION_PORT_DEFAULT );
  psion = new rfsv32( psionSocket );
}

KProtocolPsion::~KProtocolPsion()
{
  delete psion;
  delete psionSocket;
}


//
// Utility methods
//
const QString &KProtocolPsion::urlToPath( const KURL &url )
{
  path = url.path();

#if 0
  // This works with the old KURL used in 1.1
  // Handle drive specifier (this handles KDE 1.1)
  path.replace( QRegExp( "^/c/" ), "c:/" );
  path.replace( QRegExp( "^/d/" ), "d:/" );
  path.replace( QRegExp( "^/z/" ), "z:/" );
#else
  // Handle drive specifier (this handles KDE 1.1.2)
  QString drive = url.host();
  path = drive;
  path += ":";
  path += url.path();
#endif

  warning( "psion path %s", path.data() );

  return path;
}

bool KProtocolPsion::hasQuery( const KURL &url )
{
  //  if ( url.searchPart() == 0 )
    return false;
    //  else
    //    return true;
}

//> ls
//dw-------          0 Wed Jan  1 08:10:02 1997 System
//dw-------          0 Wed Jan  1 08:10:02 1997 Documents
//dw-------          0 Tue Nov 10 16:46:44 1998 Python
//>                              
void KProtocolPsion::buildEntry( bufferStore &store )
{
  long dateLong = store.getDWord(0);
  entry.size = store.getDWord(4);
  long attr = store.getDWord(8);

  QDateTime date;
  date.setTime_t( dateLong );
  entry.date = date.toString();
  entry.name = store.getString( 12 );

  // Unix permission flags
  bool isDir;
  bool isReadOnly;

  if ( ( attr & rfsv32::PSI_ATTR_DIRECTORY )
       || ( attr & rfsv32::PSI_ATTR_VOLUME ) )
    isDir = true;
  else
    isDir = false;
  entry.isdir = isDir;

  if ( attr & rfsv32::PSI_ATTR_RONLY )
    isReadOnly = true;
  else
    isReadOnly = false;

  QString permissions;

  if ( isDir ) {
    permissions = "d";
    entry.name += "/";
  }
  else
    permissions = "-";

  if ( isReadOnly )
    permissions += "r--r--r--";
  else
    permissions += "rw-rw-rw-";

  entry.access = permissions;
  entry.owner = "rich";
  entry.group = "users";

  warning( "Psion Got entry: %s, perm %s", entry.name.data(), permissions.data() );

  //  cout << ((attr & rfsv32::PSI_ATTR_DIRECTORY) ? "d" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_RONLY) ? "-" : "w");
  //  cout << ((attr & rfsv32::PSI_ATTR_HIDDEN) ? "h" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_SYSTEM) ? "s" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_ARCHIVE) ? "a" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_VOLUME) ? "v" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_NORMAL) ? "n" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_TEMPORARY) ? "t" : "-");
  //  cout << ((attr & rfsv32::PSI_ATTR_COMPRESSED) ? "c" : "-");
}

int KProtocolPsion::openQuery( const KURL &url )
{
  warning( "psion: Query stuff not done yet" );

  return 0;
}

int KProtocolPsion::mapErrorCode( long psionErrCode )
{
  int err;

  if ( psionErrCode == rfsv32::PSI_ERR_NONE ) {
    // We shouldn't ever actually get here, but if we do...
    err = 0;
  }
  // General fuck ups
  else if ( err == rfsv32::PSI_ERR_NOT_FOUND ) {
    err = KIO_ERROR_FileDoesNotExist;
  }
  else if ( err == rfsv32::PSI_ERR_ALREADY_EXISTS ) {
    err = KIO_ERROR_FileExists;
  }
  else if ( err == rfsv32::PSI_ERR_PATH_NOT_FOUND ) {
    err = KIO_ERROR_FileDoesNotExist;
  }
  else if ( err == rfsv32::PSI_ERR_DISK_FULL ) {
    err = KIO_ERROR_DiskFull;
  }
  else {
    // Disasters
    // I would love to be able to handle connection errors better
    //  rfsv32::PSI_ERR_GENERAL:
    //  rfsv32::PSI_ERR_BAD_POWER:
    //  rfsv32::PSI_ERR_CANCEL:
    //  rfsv32::PSI_ERR_NO_MEMORY:
    //  rfsv32::PSI_ERR_NOT_SUPPORTED:
    //  rfsv32::PSI_ERR_ARGUMENT:
    //  rfsv32::PSI_ERR_TOTAL_LOSS_OF_PRECISION:
    //  rfsv32::PSI_ERR_BAD_HANDLE:
    //  rfsv32::PSI_ERR_OVERFLOW:
    //  rfsv32::PSI_ERR_UNDERFLOW:
    //  rfsv32::PSI_ERR_DIED:
    //  rfsv32::PSI_ERR_IN_USE:
    //  rfsv32::PSI_ERR_SERVER_TERMINATED:
    //  rfsv32::PSI_ERR_SERVER_BUSY:
    //  rfsv32::PSI_ERR_COMPLETION:
    //  rfsv32::PSI_ERR_NOT_READY:
    //  rfsv32::PSI_ERR_UNKNOWN:
    //  rfsv32::PSI_ERR_CORRUPT:
    //  rfsv32::PSI_ERR_ACCESS_DENIED:
    //  rfsv32::PSI_ERR_LOCKED:
    //  rfsv32::PSI_ERR_WRITE:
    //  rfsv32::PSI_ERR_DISMOUNTED:
    ///  rfsv32::PSI_ERR_EoF:
    //  rfsv32::PSI_ERR_BAD_DRIVER:
    //  rfsv32::PSI_ERR_BAD_NAME:
    //  rfsv32::PSI_ERR_COMMS_LINE_FAIL:
    //  rfsv32::PSI_ERR_COMMS_FRAME:
    //  rfsv32::PSI_ERR_COMMS_OVERRUN:
    //  rfsv32::PSI_ERR_COMMS_PARITY:
    //  rfsv32::PSI_ERR_TIMEOUT:
    //  rfsv32::PSI_ERR_COULD_NOT_CONNECT:
    //  rfsv32::PSI_ERR_COULD_NOT_DISCONNECT:
    //  rfsv32::PSI_ERR_DISCONNECTED:
    //  rfsv32::PSI_ERR_BAD_LIBRARY_ENTRY_POINT:
    //  rfsv32::PSI_ERR_BAD_DESCRIPTOR:
    //  rfsv32::PSI_ERR_ABORT:
    //  rfsv32::PSI_ERR_TOO_BIG:
    //  rfsv32::PSI_ERR_DIVIDE_BY_ZERO:
    err = KIO_ERROR_NotPossible;
  }

  return err;
}

int KProtocolPsion::throwError( long psionErrCode )
{
  static QString message;

  message = psion->opErr( psionErrCode  );

  warning( "Preparing error: %s", message.data() );

  int kerrCode = mapErrorCode( psionErrCode );
  //  int kerrCode = 18;

  QString s;
  s.sprintf( "Psion error: %s", message.data() );

  emit info( s.data() );

  Error( kerrCode, message, 0 );
  return FAIL;
}


// long handle of open psion file

int KProtocolPsion::Open(KURL *url, int mode)
{
  // Record the location of the file
  // Call rfsv32::open()
  // If opened then record the handle
  // Return status

warning( "psion: open" );

  int status = 0;
  filename = urlToPath( *url );

  if ( hasQuery( *url ) ) {
    // Will be used to provide access to the entended info available about the
    // Psion file system.
    openQuery( *url );
  }
  else {
    long psiStatus;

    if ( mode & KProtocol::READ ) {
      psiStatus = psion->fopen( rfsv32::PSI_OMODE_SHARE_READERS
				, filename, handle );
    }
    else if ( mode & KProtocol::WRITE ) {
      psiStatus = psion->fopen( rfsv32::PSI_OMODE_SHARE_EXCLUSIVE
				| rfsv32::PSI_OMODE_READ_WRITE
				, filename, handle );
    }

    if ( psiStatus )
      status = throwError( psiStatus );
  }

  return status;
}

long KProtocolPsion::Read(void *buffer, long nbytes)
{
  long status = psion->fread( handle, (char *) buffer, nbytes );

  if ( status < 0 ) {
    status = throwError( status );
  }
  else if ( status < nbytes ) {
    reachedEOF = true;
  }

  return status;
}

long KProtocolPsion::Write(void *buffer, long nbytes)
{
  long status = psion->fwrite( handle, (char *) buffer, nbytes );

  if ( status < 0 ) {
    status = throwError( status );
  }
  else if ( status < nbytes ) {
    reachedEOF = true;
  }

  return status;
}

int KProtocolPsion::Close()
{
  long status = psion->fclose( handle );

  if ( status < 0 ) {
    status = throwError( status );
  }

  return status;
}

int KProtocolPsion::atEOF()
{
  return reachedEOF;
}

long KProtocolPsion::Size()
{
  // Find out where we are now
  long current = psion->fseek( handle, 0, rfsv32::PSI_SEEK_CUR );
  long end;

  if ( current >= 0 ) {
    // Seek to the end
    end = psion->fseek( handle, 0, rfsv32::PSI_SEEK_END );

    if ( end >= 0 ) {
      // Restore the position
      long pos = psion->fseek( handle, current, rfsv32::PSI_SEEK_SET );
      if ( pos < 0 )
	end = throwError( pos );
    }
    else {
      end = throwError( end );
    }
  }
  else {
    end = throwError( current );
  }

  return end;
}

int KProtocolPsion::Delete(KURL *url)
{
  // If is dir
  //    check if empty
  //    if yes rmdir
  // else
  //    remove

  filename = urlToPath( *url );
  QString last = filename.right( 1 );
  int status = 0;

  if ( last == "/" ) {
    // handle dir
    warning( "psion: Remove directory not implemented\n" );
  }
  else {
    long result = psion->remove( filename );

    if ( result < 0 )
      status = throwError( result );
  }

  return status;
}

int KProtocolPsion::OpenDir(KURL *url)
{
  // Record the location of the dir
  // Call opendir
  // If opened record the handle
  // Return status

warning( "psion: opendir" );

  int status = 0;
  filename = urlToPath( *url );

  if ( hasQuery( *url ) ) {
    // Will be used to provide access to the entended info available about the
    // Psion file system.
    openQuery( *url );
  }

  if ( filename == "c:/" )
    return 0;
  else if ( filename ==  "d:/" )
    return 0;
  else if ( filename == "z:/" )
    return 0;

  // Check it really is a directory here
  long attr;
  long psiStatus = psion->fgetattr( filename, &attr );

  warning( "psion: opendir getattr attr %ld ,  psi %ld ", attr, psiStatus );

  if ( psiStatus < 0 ) {
      Error( KIO_ERROR_FileDoesNotExist, "The specified path does not exist", 0 );
      emit info( "The specified path does not exist" );

      status = FAIL;
  }
  else {
    if ( ( attr & (rfsv32::PSI_ATTR_DIRECTORY | rfsv32::PSI_ATTR_VOLUME ) ) ) {
      emit mimeType( "inode/directory" );
    }
    else {
      Error( KIO_ERROR_NotADirectory, "The specified path does not refer to a directory", 0 );
      emit info( "The specified path does not refer to a directory" );

      status = FAIL;
    }
  }

  warning( "psion: opendir returning %d", status );

  return status;
}

bool KProtocolPsion::isHTML()
{
  return false;
}

void KProtocolPsion::AllowHTML( bool allowed )
{
  allowHTML = allowed;
}

//void KProtocolPsion::EmitData( KIOSlaveIPC* )
//{
  // Not used unless emitting HTML
//}

KProtocolDirEntry *KProtocolPsion::ReadDir()
{
  long psiStatus = 0;

  // If it's the first time through
  if ( !readingDir ) {
    psiStatus = psion->dir( filename, &dirContents );
    if ( psiStatus >= 0 ) {
      readingDir = true;
    }
    else {
      // Could not read the dir
      readingDir = false;
      return 0;
    }
  }

  // If we should be returning a direntry
  if ( readingDir ) {
    if ( !dirContents.empty() ) {
      bufferStore dirStore = dirContents.popBuffer();
      buildEntry( dirStore );
    }
    else {
      readingDir = false;
      return 0;
    }
  }
 
  if ( psiStatus < 0 ) {
    throwError( psiStatus );

    return 0;
  }
  else {
    return &entry;
  }
}

int KProtocolPsion::CloseDir()
{
  return 0;
}

int KProtocolPsion::MkDir(KURL *url)
{
  int status = 0;
  QString filename = urlToPath( *url );

  long psiStatus;
  
  psiStatus = psion->mkdir( filename );

  if ( psiStatus )
    status = throwError( psiStatus );
  
  return status;
}

//int KProtocolPsion::GetPermissions( KURL &_u )
//{
  // Returns an int giving the permissions flags
//}

//void KProtocolPsion::SetPermissions( KURL &_u, int _perm )
//{
  // Sets the permissions flag to the specified int
//}

#include "psion.moc"
