/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <termios.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qradiobutton.h>
#include <qtooltip.h>
#include <qstring.h>

#include <kapp.h>

#include "preferences.h"
#include "global.h"



Preferences *prefs = 0;



Preferences::Preferences(QWidget *parent, const char *name) : QTabDialog (parent, name)
{
	QGridLayout *grid;
	QBoxLayout *vbox, *svbox, *hbox, *shbox;
	QLabel *label;
	QGroupBox *gb;
	QRadioButton *rb;

	config = kapp->getConfig();

	gotToolTips = false;

	page1 = new QWidget(this, "prefspage1");
	CHECK_PTR(page1);
	page2 = new QWidget(this, "prefspage2");
	CHECK_PTR(page2);
	page3 = new QWidget(this, "prefspage3");
	CHECK_PTR(page3);
	page4 = new QWidget(this, "prefspage4");
	CHECK_PTR(page4);

	addTab(page1, i18n("General"));
	addTab(page2, i18n("Modem"));
	addTab(page3, i18n("Print"));
	addTab(page4, i18n("Mail"));

	/* Page 1: General */
	vbox = new QVBoxLayout(page1, 12);

	gb = new QGroupBox(i18n("User Interface"), page1);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	gen_tooltips = new QCheckBox(i18n("Show tooltips"), gb);
	gen_tooltips->setMinimumSize(gen_tooltips->sizeHint());
	svbox->addWidget(gen_tooltips);
	svbox->activate();
	
	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Download"), page1);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	gen_loadnew = new QCheckBox(i18n("Download new facsimile only"), gb);
	gen_loadnew->setMinimumSize(gen_loadnew->sizeHint());
	svbox->addWidget(gen_loadnew);
	gen_deletefax = new QCheckBox(i18n("Delete facsimile from modem"), gb);
	gen_deletefax->setMinimumSize(gen_deletefax->sizeHint());
	svbox->addWidget(gen_deletefax);
	svbox->activate();

	vbox->addWidget(gb);

	vbox->addStretch(1);
	vbox->activate();

	/* Page 2: Device */
	vbox = new QVBoxLayout(page2, 12);

	gb = new QGroupBox(i18n("Modem Device"), page2);

	grid = new QGridLayout(gb, 3, 3, 12);

	grid->addRowSpacing(0, 2);

	label = new QLabel(i18n("Device:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mod_device = new QLineEdit(gb);
	mod_device->setMinimumSize(mod_device->sizeHint());
	grid->addMultiCellWidget(mod_device, 1, 1, 1, 2);

	label = new QLabel(i18n("Speed:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mod_speed = new QComboBox(false, gb);
	mod_speed->insertItem(" 19200 Bps ", 0);
	mod_speed->insertItem(" 38400 Bps ", 1);
#ifdef B57600
	mod_speed->insertItem(" 57600 Bps ", 2);
#endif
#ifdef B115200
	mod_speed->insertItem(" 115200 Bps ", 3);
#endif
#ifdef B230400
	mod_speed->insertItem(" 230400 Bps ", 4);
#endif
	mod_speed->setMinimumSize(mod_speed->sizeHint());
	grid->addWidget(mod_speed, 2, 1);

	grid->setColStretch(2, 1);

	grid->activate();

	vbox->addWidget(gb);
	vbox->addStretch(1);
	vbox->activate();


	/* Page 3: Print */
	vbox = new QVBoxLayout(page3, 12);

	gb = new QGroupBox(i18n("Print Command"), page3);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);

	prt_command = new QLineEdit(gb);
	prt_command->setMinimumWidth(prt_command->sizeHint().width());
	prt_command->setFixedHeight(prt_command->sizeHint().height());
	svbox->addWidget(prt_command);
	svbox->activate();
	
	vbox->addWidget(gb);

	hbox = new QHBoxLayout(8);
	vbox->addLayout(hbox);

	prt_paper = new QButtonGroup(i18n("Paper Format"), page3);
	svbox = new QVBoxLayout(prt_paper, 10);
	svbox->addSpacing(8);
	rb = new QRadioButton(i18n("A4 (210297 mm)"), prt_paper);
	rb->setMinimumSize(rb->sizeHint());
	rb->setChecked(true);
	svbox->addWidget(rb);
	rb = new QRadioButton(i18n("B4 (257364 mm)"), prt_paper);
	rb->setMinimumSize(rb->sizeHint());
	svbox->addWidget(rb);
	rb = new QRadioButton(i18n("B5 (182257 mm)"), prt_paper);
	rb->setMinimumSize(rb->sizeHint());
	svbox->addWidget(rb);
	rb = new QRadioButton(i18n("Letter (811 in.)"), prt_paper);
	rb->setMinimumSize(rb->sizeHint());
	svbox->addWidget(rb);
	rb = new QRadioButton(i18n("Legal (814 in.)"), prt_paper);
	rb->setMinimumSize(rb->sizeHint());
	svbox->addWidget(rb);
	rb = new QRadioButton(i18n("Executive (710 in.)"), prt_paper);
	rb->setMinimumSize(rb->sizeHint());
	svbox->addWidget(rb);
	svbox->activate();

	hbox->addWidget(prt_paper);

	gb = new QGroupBox(i18n("Paper Margins"), page3);
	grid = new QGridLayout(gb, 9, 2, 10);

	grid->addRowSpacing(0, 8);
	label = new QLabel(i18n("Left (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);
	prt_lmargin = new QSpinBox(gb);
	prt_lmargin->setMinimumSize(prt_lmargin->sizeHint());
	grid->addWidget(prt_lmargin, 1, 1);

	label = new QLabel(i18n("Right (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 3, 0);
	prt_rmargin = new QSpinBox(gb);
	prt_rmargin->setMinimumSize(prt_rmargin->sizeHint());
	grid->addWidget(prt_rmargin, 3, 1);

	label = new QLabel(i18n("Top (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 5, 0);
	prt_tmargin = new QSpinBox(gb);
	prt_tmargin->setMinimumSize(prt_tmargin->sizeHint());
	grid->addWidget(prt_tmargin, 5, 1);

	label = new QLabel(i18n("Bottom (mm):"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 7, 0);
	prt_bmargin = new QSpinBox(gb);
	prt_bmargin->setMinimumSize(prt_bmargin->sizeHint());
	grid->addWidget(prt_bmargin, 7, 1);

	grid->setColStretch(1, 1);
	grid->setRowStretch(0, 1);
	grid->setRowStretch(2, 1);
	grid->setRowStretch(4, 1);
	grid->setRowStretch(6, 1);
	grid->setRowStretch(8, 1);

	grid->activate();

	hbox->addWidget(gb);

	gb = new QGroupBox(i18n("Options"), page3);
	svbox = new QVBoxLayout(gb, 10);
	svbox->addSpacing(8);
	shbox = new QHBoxLayout(8);
	svbox->addLayout(shbox);

	prt_level2 = new QCheckBox(i18n("Language Level 2"), gb);
	prt_level2->setMinimumSize(prt_level2->sizeHint());
	shbox->addWidget(prt_level2);
	
	prt_interpolate = new QCheckBox(i18n("Interpolate"), gb);
	prt_interpolate->setMinimumSize(prt_interpolate->sizeHint());
	shbox->addWidget(prt_interpolate);
	svbox->activate();
	
	vbox->addWidget(gb);

	vbox->activate();

	/* Page 4: Mail */
	vbox = new QVBoxLayout(page4, 12);

	gb = new QGroupBox(i18n("SMTP Server"), page4);

	grid = new QGridLayout(gb, 3, 3, 12);

	grid->addRowSpacing(0, 2);

	label = new QLabel(i18n("Name:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mail_server = new QLineEdit(gb);
	mail_server->setMinimumSize(mail_server->sizeHint());
	grid->addMultiCellWidget(mail_server, 1, 1, 1, 2);

	label = new QLabel(i18n("Port:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mail_port = new QLineEdit(gb);
	mail_port->setFixedWidth(50);
	mail_port->setMinimumHeight(mail_port->sizeHint().height());
	grid->addWidget(mail_port, 2, 1);

	grid->setColStretch(2, 1);

	grid->activate();

	vbox->addWidget(gb);

	gb = new QGroupBox(i18n("Identity"), page4);

	grid = new QGridLayout(gb, 5, 2, 12);

	grid->addRowSpacing(0, 2);

	label = new QLabel(i18n("Name:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	mail_name = new QLineEdit(gb);
	mail_name->setMinimumSize(mail_name->sizeHint());
	grid->addWidget(mail_name, 1, 1);

	label = new QLabel(i18n("Organization:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	mail_org = new QLineEdit(gb);
	mail_org->setMinimumSize(mail_org->sizeHint());
	grid->addWidget(mail_org, 2, 1);

	label = new QLabel(i18n("Email address:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 3, 0);

	mail_email = new QLineEdit(gb);
	mail_email->setMinimumSize(mail_email->sizeHint());
	grid->addWidget(mail_email, 3, 1);

	label = new QLabel(i18n("Reply-to address:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 4, 0);

	mail_reply = new QLineEdit(gb);
	mail_reply->setMinimumSize(mail_reply->sizeHint());
	grid->addWidget(mail_reply, 4, 1);

	grid->setColStretch(1, 1);

	grid->activate();

	vbox->addWidget(gb);
	vbox->addStretch(1);

	vbox->activate();

	setOKButton(i18n("OK"));
	setDefaultButton(i18n("Default"));
	setApplyButton(i18n("Apply"));
	setCancelButton(i18n("Cancel"));

	reset();

	connect(this, SIGNAL(defaultButtonPressed()), SLOT(reset()));
	connect(this, SIGNAL(applyButtonPressed()), SLOT(apply()));

	setCaption(i18n("KMLOFax Preferences"));

	resize(sizeHint());
}


Preferences::~Preferences()
{
}


void Preferences::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(gen_tooltips, i18n("Show tooltips in dialogs"));
		QToolTip::add(gen_loadnew, i18n("Download only new facsimile from modem"));
		QToolTip::add(gen_deletefax, i18n("Delete facsimile from modem after download"));
		QToolTip::add(mod_device, i18n("Device of the modem"));
		QToolTip::add(mod_speed, i18n("Speed for serial communication"));
		QToolTip::add(prt_command, i18n("Command to invoke for printing"));
		QToolTip::add(prt_paper, i18n("Paper format used by the printer"));
		QToolTip::add(prt_lmargin, i18n("Left margin on paper"));
		QToolTip::add(prt_rmargin, i18n("Right margin on paper"));
		QToolTip::add(prt_tmargin, i18n("Top margin on paper"));
		QToolTip::add(prt_bmargin, i18n("Bottom margin on paper"));
		QToolTip::add(prt_level2, i18n("Use PostScript Language Level 2"));
		QToolTip::add(prt_interpolate, i18n("Interpolate for smoother output"));
		QToolTip::add(mail_server, i18n("Name of the SMTP server"));
		QToolTip::add(mail_port, i18n("Port of the SMTP server"));
		QToolTip::add(mail_name, i18n("Your full name"));
		QToolTip::add(mail_org, i18n("Your organization"));
		QToolTip::add(mail_email, i18n("Your email address"));
		QToolTip::add(mail_reply, i18n("The address replies are sent to"));
	}
	else {
		QToolTip::remove(gen_tooltips);
		QToolTip::remove(gen_loadnew);
		QToolTip::remove(gen_deletefax);
		QToolTip::remove(mod_device);
		QToolTip::remove(mod_speed);
		QToolTip::remove(prt_command);
		QToolTip::remove(prt_paper);
		QToolTip::remove(prt_lmargin);
		QToolTip::remove(prt_rmargin);
		QToolTip::remove(prt_tmargin);
		QToolTip::remove(prt_bmargin);
		QToolTip::remove(prt_level2);
		QToolTip::remove(prt_interpolate);
		QToolTip::remove(mail_server);
		QToolTip::remove(mail_port);
		QToolTip::remove(mail_name);
		QToolTip::remove(mail_org);
		QToolTip::remove(mail_email);
		QToolTip::remove(mail_reply);
	}
}	


void Preferences::reset()
{
	QString s;
	int speed;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	gen_tooltips->setChecked(config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS));
	gen_loadnew->setChecked(config->readBoolEntry(ID_PREF_GEN_LOADNEW, PREF_GEN_LOADNEW));
	gen_deletefax->setChecked(config->readBoolEntry(ID_PREF_GEN_DELETEFAX, PREF_GEN_DELETEFAX));

	config->setGroup(ID_PREF_GROUP_MODEM);
	mod_device->setText(config->readEntry(ID_PREF_MOD_DEVICE, PREF_MOD_DEVICE));
	speed = config->readNumEntry(ID_PREF_MOD_SPEED, PREF_MOD_SPEED);
	if ((speed < 0) || (speed > 4))
		speed = PREF_MOD_SPEED;
	mod_speed->setCurrentItem(speed);

	config->setGroup(ID_PREF_GROUP_PRINT);
	prt_command->setText(config->readEntry(ID_PREF_PRT_COMMAND, PREF_PRT_COMMAND));
	prt_paper->setButton(config->readNumEntry(ID_PREF_PRT_PAPER, PREF_PRT_PAPER));
	prt_lmargin->setValue(config->readNumEntry(ID_PREF_PRT_LMARGIN, PREF_PRT_LMARGIN));
	prt_rmargin->setValue(config->readNumEntry(ID_PREF_PRT_RMARGIN, PREF_PRT_RMARGIN));
	prt_tmargin->setValue(config->readNumEntry(ID_PREF_PRT_TMARGIN, PREF_PRT_TMARGIN));
	prt_bmargin->setValue(config->readNumEntry(ID_PREF_PRT_BMARGIN, PREF_PRT_BMARGIN));
	prt_level2->setChecked(config->readBoolEntry(ID_PREF_PRT_LEVEL2, PREF_PRT_LEVEL2));
	prt_interpolate->setChecked(config->readBoolEntry(ID_PREF_PRT_INTERPOL, PREF_PRT_INTERPOL));

	config->setGroup(ID_PREF_GROUP_MAIL);
	mail_server->setText(config->readEntry(ID_PREF_MAIL_SERVER, PREF_MAIL_SERVER));
	s.setNum(config->readNumEntry(ID_PREF_MAIL_PORT, PREF_MAIL_PORT));
	mail_port->setText(s);
	mail_name->setText(config->readEntry(ID_PREF_MAIL_NAME, PREF_MAIL_NAME));
	mail_org->setText(config->readEntry(ID_PREF_MAIL_ORG, PREF_MAIL_ORG));
	mail_email->setText(config->readEntry(ID_PREF_MAIL_EMAIL, PREF_MAIL_EMAIL));
	mail_reply->setText(config->readEntry(ID_PREF_MAIL_REPLY, PREF_MAIL_REPLY));
}


void Preferences::apply()
{
	int i;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	config->writeEntry(ID_PREF_GEN_TOOLTIPS, gen_tooltips->isChecked());
	config->writeEntry(ID_PREF_GEN_LOADNEW, gen_loadnew->isChecked());
	config->writeEntry(ID_PREF_GEN_DELETEFAX, gen_deletefax->isChecked());
	
	config->setGroup(ID_PREF_GROUP_MODEM);
	config->writeEntry(ID_PREF_MOD_DEVICE, mod_device->text());
	config->writeEntry(ID_PREF_MOD_SPEED, mod_speed->currentItem());

	config->setGroup(ID_PREF_GROUP_PRINT);
	config->writeEntry(ID_PREF_PRT_COMMAND, prt_command->text());
	for (i=0; i < 6; i++)
		if (((QRadioButton *)prt_paper->find(i))->isChecked()) {
			config->writeEntry(ID_PREF_PRT_PAPER, i);
			break;
		}
	config->writeEntry(ID_PREF_PRT_LMARGIN, prt_lmargin->value());
	config->writeEntry(ID_PREF_PRT_RMARGIN, prt_rmargin->value());
	config->writeEntry(ID_PREF_PRT_TMARGIN, prt_tmargin->value());
	config->writeEntry(ID_PREF_PRT_BMARGIN, prt_bmargin->value());
	config->writeEntry(ID_PREF_PRT_LEVEL2, prt_level2->isChecked());
	config->writeEntry(ID_PREF_PRT_INTERPOL, prt_interpolate->isChecked());

	config->setGroup(ID_PREF_GROUP_MAIL);
	config->writeEntry(ID_PREF_MAIL_SERVER, mail_server->text());
	config->writeEntry(ID_PREF_MAIL_PORT, atoi(mail_port->text()));
	config->writeEntry(ID_PREF_MAIL_NAME, mail_name->text());
	config->writeEntry(ID_PREF_MAIL_ORG, mail_org->text());
	config->writeEntry(ID_PREF_MAIL_EMAIL, mail_email->text());
	config->writeEntry(ID_PREF_MAIL_REPLY, mail_reply->text());

	config->sync();

	emit apply_clicked();
}
