/*
    KMLOFax

    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdlib.h>
#include <string.h>

#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qsize.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qframe.h>
#include <qstring.h>

#include <kapp.h>
#include <kiconloader.h>
#include <kmsgbox.h>
#include <kfiledialog.h>

#include "exportdlg.h"
#include "mlofile.h"
#include "filters.h"



ExportDlg *exportdlg = 0;



ExportDlg::ExportDlg(QWidget *parent, const char *name) : QDialog (parent, name, true)
{
	QBoxLayout *vbox, *hbox, *fvbox, *fhbox;
	QGroupBox *gb;
	QPushButton *ok, *cancel;
	QLabel *l1, *l2;
	QSize size;
	QFrame *frame;

	gotToolTips = false;

	vbox = new QVBoxLayout(this , 8);

	gb = new QGroupBox(i18n("Export to File:"), this);
	fvbox = new QVBoxLayout(gb, 12);
	fvbox->addSpacing(8);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	file = new QLineEdit(gb);
	file->setMinimumWidth(240);
	file->setFixedHeight(file->sizeHint().height());
	file->setText("facsimile.tif");
	fhbox->addWidget(file);

	browse = new QPushButton(gb);
	browse->setPixmap(Icon("fileopen.xpm"));
	browse->setFixedSize(browse->sizeHint());	
	fhbox->addWidget(browse);
	
	fvbox->activate();

	vbox->addWidget(gb);
		
	gb = new QGroupBox(i18n("Options"), this);
	fvbox = new QVBoxLayout(gb, 12);
	fvbox->addSpacing(8);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	l1 = new QLabel(i18n("&From page"), gb);
	fhbox->addWidget(l1);
	from_page = new QSpinBox(1, 999, 1, gb);
	l1->setBuddy(from_page);
	from_page->setValue(1);
	from_page->setFixedWidth(from_page->sizeHint().width());
	from_page->setFixedHeight(from_page->sizeHint().height());
	fhbox->addWidget(from_page);
	fhbox->addStretch(1);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	l2 = new QLabel(i18n("&To page"), gb);
	fhbox->addWidget(l2);
	to_page = new QSpinBox(1, 999, 1, gb);
	l2->setBuddy(to_page);
	to_page->setValue(999);
	to_page->setFixedWidth(to_page->sizeHint().width());
	to_page->setFixedHeight(to_page->sizeHint().height());
	fhbox->addWidget(to_page);
	fhbox->addStretch(1);
	
	size = l1->sizeHint().expandedTo(l2->sizeHint());
	l1->setMinimumSize(size);
	l2->setMinimumSize(size);

	frame = new QFrame(gb);
	frame->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	frame->setFixedHeight(6);
	fvbox->addWidget(frame, 1);
	
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	
	l1 = new QLabel(i18n("&Compression:"), gb);
	l1->setMinimumSize(l1->sizeHint());
	l1->setFixedHeight(l1->sizeHint().height());
	fhbox->addWidget(l1);

	compression = new QComboBox(false, gb);
	l1->setBuddy(compression);
	compression->insertItem(i18n(" None "), 0);
	compression->insertItem(i18n(" CCITT Group 3 "), 1);
	compression->insertItem(i18n(" LZW "), 2);
	compression->insertItem(i18n(" Pack Bits "), 3);
	compression->setMinimumSize(compression->sizeHint());
	fhbox->addWidget(compression);
	fhbox->addStretch(1);
	
	fvbox->activate();
	
	vbox->addWidget(gb);
	
	hbox = new QHBoxLayout(8);
	vbox->addLayout(hbox);

	hbox->addStretch(1);

	ok = new QPushButton(i18n("OK"), this);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("Cancel"), this);
	size = ok->sizeHint().expandedTo(cancel->sizeHint());

	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addWidget(ok);
	hbox->addWidget(cancel);

	vbox->activate();

	connect(browse, SIGNAL(clicked()), SLOT(browseClicked()));
	connect(ok, SIGNAL(clicked()), SLOT(okClicked()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	setCaption(i18n("KMLOFax Export"));

	resize(sizeHint());
}


ExportDlg::~ExportDlg()
{
}


void ExportDlg::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(file, i18n("Name of file to export to"));
		QToolTip::add(browse, i18n("Select file to export to"));
		QToolTip::add(from_page, i18n("First page to export"));
		QToolTip::add(to_page, i18n("Last page to export"));
		QToolTip::add(compression, i18n("Compression method for TIFF file"));
	}
	else {
		QToolTip::remove(file);
		QToolTip::remove(browse);
		QToolTip::remove(from_page);
		QToolTip::remove(to_page);
		QToolTip::remove(compression);
	}
}	


void ExportDlg::exportFax(const QString& name, const QString& alias)
{
	MLOFile mlofile;

	fname = name;
	salias = alias;

	mlofile.setName(name);

	if (!mlofile.open()) {
		KMsgBox::message(0, i18n("File Error"), i18n("Cannot open facsimile file."), KMsgBox::EXCLAMATION);
		return;
	}

	from_page->setRange(1, mlofile.pages());
	from_page->setValue(1);
	to_page->setRange(1, mlofile.pages());
	to_page->setValue(mlofile.pages());
	mlofile.close();

	if (isVisible())
		raise();
	else
		show();
}


void ExportDlg::browseClicked()
{
	QString s = KFileDialog::getSaveFileName(QDir::currentDirPath(), i18n("*.tif|TIFF Files (*.tif)\n*|All Files (*)"), this);
	if (!s.isEmpty())
		file->setText(s);
}


void ExportDlg::okClicked()
{
	MLO2TiffFilter filter;

	filter.setFile(fname);
	filter.setRange(from_page->value(), to_page->value());
	filter.setCompression(compression->currentItem());

	kapp->setOverrideCursor(waitCursor);

	filter.convertFile(file->text(), salias);

	kapp->restoreOverrideCursor();

	hide();
}
