/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "mlofile.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <string.h>

#include <qfile.h>

#include "global.h"


#ifndef CETX
#define CETX 03
#endif

#ifndef CDLE
#define CDLE 020
#endif


#define MODE_P    0
#define MODE_H    1
#define MODE_V0   2
#define MODE_VR_1 3
#define MODE_VR_2 4
#define MODE_VR_3 5
#define MODE_VL_1 6
#define MODE_VL_2 7
#define MODE_VL_3 8



static const hcode_t white_codes[] = {
	{0x07,  4,    2}, {0x08,  4,    3}, {0x0b,  4,    4}, {0x0c,  4,    5},
	{0x0e,  4,    6}, {0x0f,  4,    7}, {0x13,  5,    8}, {0x14,  5,    9},
	{0x07,  5,   10}, {0x08,  5,   11}, {0x1b,  5,   64}, {0x12,  5,  128},
	{0x17,  6,  192}, {0x18,  6, 1664}, {0x08,  6,   12}, {0x03,  6,   13},
	{0x34,  6,   14}, {0x35,  6,   15}, {0x2a,  6,   16}, {0x2b,  6,   17},
	{0x07,  6,    1}, {0x37,  7,  256}, {0x27,  7,   18}, {0x0c,  7,   19},
	{0x08,  7,   20}, {0x17,  7,   21}, {0x03,  7,   22}, {0x04,  7,   23},
	{0x28,  7,   24}, {0x2b,  7,   25}, {0x13,  7,   26}, {0x24,  7,   27},
	{0x18,  7,   28}, {0x02,  8,   29}, {0x03,  8,   30}, {0x1a,  8,   31},
	{0x1b,  8,   32}, {0x12,  8,   33}, {0x13,  8,   34}, {0x14,  8,   35},
	{0x15,  8,   36}, {0x16,  8,   37}, {0x17,  8,   38}, {0x28,  8,   39},
	{0x29,  8,   40}, {0x2a,  8,   41}, {0x2b,  8,   42}, {0x2c,  8,   43},
	{0x2d,  8,   44}, {0x04,  8,   45}, {0x05,  8,   46}, {0x0a,  8,   47},
	{0x0b,  8,   48}, {0x52,  8,   49}, {0x53,  8,   50}, {0x54,  8,   51},
	{0x55,  8,   52}, {0x24,  8,   53}, {0x25,  8,   54}, {0x58,  8,   55},
	{0x59,  8,   56}, {0x5a,  8,   57}, {0x5b,  8,   58}, {0x4a,  8,   59},
	{0x4b,  8,   60}, {0x32,  8,   61}, {0x33,  8,   62}, {0x34,  8,   63},
	{0x35,  8,    0}, {0x36,  8,  320}, {0x37,  8,  384}, {0x64,  8,  448},
	{0x65,  8,  512}, {0x68,  8,  576}, {0x67,  8,  640}, {0xcc,  9,  704},
	{0xcd,  9,  768}, {0xd2,  9,  832}, {0xd3,  9,  896}, {0xd4,  9,  960},
	{0xd5,  9, 1024}, {0xd6,  9, 1088}, {0xd7,  9, 1152}, {0xd8,  9, 1216},
	{0xd9,  9, 1280}, {0xda,  9, 1344}, {0xdb,  9, 1408}, {0x98,  9, 1472},
	{0x99,  9, 1536}, {0x9a,  9, 1600}, {0x9b,  9, 1728},

	/* extended make up codes */
	{0x08, 11, 1792}, {0x0c, 11, 1856}, {0x0d, 11, 1920}, {0x12, 12, 1984},
	{0x13, 12, 2048}, {0x14, 12, 2112}, {0x15, 12, 2176}, {0x16, 12, 2240},
	{0x17, 12, 2304}, {0x1c, 12, 2368}, {0x1d, 12, 2432}, {0x1e, 12, 2496},
	{0x1f, 12, 2560}, {0x00,  0,    0}
};


static const hcode_t black_codes[] = {
	{0x03,  2,    2}, {0x02,  2,    3}, {0x03,  3,    4}, {0x02,  3,    1},
	{0x03,  4,    5}, {0x02,  4,    6}, {0x03,  5,    7}, {0x05,  6,    8},
	{0x04,  6,    9}, {0x04,  7,   10}, {0x05,  7,   11}, {0x07,  7,   12},
	{0x04,  8,   13}, {0x07,  8,   14}, {0x18,  9,   15}, {0x0f, 10,   64},
	{0x17, 10,   16}, {0x18, 10,   17}, {0x08, 10,   18}, {0x37, 10,    0},
	{0x67, 11,   19}, {0x68, 11,   20}, {0x6c, 11,   21}, {0x37, 11,   22},
	{0x28, 11,   23}, {0x17, 11,   24}, {0x18, 11,   25}, {0xca, 12,   26},
	{0xcb, 12,   27}, {0xcc, 12,   28}, {0xcd, 12,   29}, {0x68, 12,   30},
	{0x69, 12,   31}, {0x6a, 12,   32}, {0x6b, 12,   33}, {0xd2, 12,   34},
	{0xd3, 12,   35}, {0xd4, 12,   36}, {0xd5, 12,   37}, {0xd6, 12,   38},
	{0xd7, 12,   39}, {0x6c, 12,   40}, {0x6d, 12,   41}, {0xda, 12,   42},
	{0xdb, 12,   43}, {0x54, 12,   44}, {0x55, 12,   45}, {0x56, 12,   46},
	{0x57, 12,   47}, {0x64, 12,   48}, {0x65, 12,   49}, {0x52, 12,   50},
	{0x53, 12,   51}, {0x24, 12,   52}, {0x37, 12,   53}, {0x38, 12,   54},
	{0x27, 12,   55}, {0x28, 12,   56}, {0x58, 12,   57}, {0x59, 12,   58},
	{0x2b, 12,   59}, {0x2c, 12,   60}, {0x5a, 12,   61}, {0x66, 12,   62},
	{0x67, 12,   63}, {0xc8, 12,  128}, {0xc9, 12,  192}, {0x5b, 12,  256},
	{0x33, 12,  320}, {0x34, 12,  384}, {0x35, 12,  448}, {0x6c, 13,  512},
	{0x6d, 13,  576}, {0x4a, 13,  640}, {0x4b, 13,  704}, {0x4c, 13,  768},
	{0x4d, 13,  832}, {0x72, 13,  896}, {0x73, 13,  960}, {0x74, 13, 1024},
	{0x75, 13, 1088}, {0x76, 13, 1152}, {0x77, 13, 1216}, {0x52, 13, 1280},
	{0x53, 13, 1344}, {0x54, 13, 1408}, {0x55, 13, 1472}, {0x5a, 13, 1536},
	{0x5b, 13, 1600}, {0x64, 13, 1664}, {0x65, 13, 1728},

	/* extended make up codes */
	{0x08, 11, 1792}, {0x0c, 11, 1856}, {0x0d, 11, 1920}, {0x12, 12, 1984},
	{0x13, 12, 2048}, {0x14, 12, 2112}, {0x15, 12, 2176}, {0x16, 12, 2240},
	{0x17, 12, 2304}, {0x1c, 12, 2368}, {0x1d, 12, 2432}, {0x1e, 12, 2496},
	{0x1f, 12, 2560}, {0x00,  0,    0}
};


static const uchar white_makeup_lut[] = {
	 0, 10, 11, 12, 21, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
	80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 13, 90, 91, 92, 93, 94,
	95, 96, 97, 98, 99,100,101,102,103
};

static const uchar white_term_lut[] = {
	68, 20,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 14, 15, 16, 17,
	18, 19,	22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35,
	36, 37, 38, 39,	40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
	52, 53, 54, 55, 56, 57,	58, 59, 60, 61, 62, 63, 64, 65, 66, 67
};

static const uchar black_makeup_lut[] = {
	 0, 15, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78,
	79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94,
	95, 96, 97, 98, 99,100,101,102,103
};

static const uchar black_term_lut[] = {
	19,  3,  0,  1,  2,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
	16, 17, 18, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
	33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48,
	49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64
};



static const hcode_t mode_codes[] = {
	{0x01, 1, MODE_V0  },
	{0x01, 3, MODE_H   },
	{0x03, 3, MODE_VR_1},
	{0x02, 3, MODE_VL_1},
	{0x01, 4, MODE_P   },
	{0x03, 6, MODE_VR_2},
	{0x02, 6, MODE_VL_2},
	{0x03, 7, MODE_VR_3},
	{0x02, 7, MODE_VL_3},
	{0x00, 0, 0        }
};





MLOFile::MLOFile( const char * name ) : QFile( name )
{
	init();
}


MLOFile::MLOFile() : QFile()
{
	init();
}


MLOFile::~MLOFile()
{
}


bool MLOFile::open()
{
	return QFile::open( IO_ReadOnly );
}


void MLOFile::close()
{
	QFile::close();
	init();
}


int MLOFile::readLine( char * line, uint maxlen )
{
	uint i = 0;
	int c;

	while (data_mode)
		getDataCh();
	
	while ((c = getch()) != '\n')
		if ((c == -1) || data_mode) {
			line[0] = 0;
			return 0;
		}
		
	maxlen--;
	for (i=0; i < maxlen; i++) {
		c = getch();
		if ((c == '\n') || (c == -1)) {
			if (i && (line[i-1] == '\r'))
				i--;
			break;
		} else
			line[i] = (char)c;
	}
	line[i] = 0;

	if (!strncmp(line, "CONNECT", 7))
		data_mode = TRUE;

	return i;
}


void MLOFile::readRleLine( short line[] )
{
	decodeDataLine( line, FALSE );
}


void MLOFile::readImgLine( uchar line[] )
{
	short rle_buf[MAX_RLE_BUFF], pos = 0;
	int count = 0;

	readRleLine( rle_buf );

	memset( (void *)line, 0x00, ((page_width-1) >> 3) + 1);

	while ((pos < page_width) && (rle_buf[count] != -1)) {
		pos += rle_buf[count++];
		if ((pos >= page_width) || (rle_buf[count] == -1))
			break;
		fillDataLine( line, pos, rle_buf[count] );
		pos += rle_buf[count++];
	}
}


int MLOFile::readHuffLine( uchar line[] )
{
	short rle_buf[MAX_RLE_BUFF];

	readRleLine( rle_buf );

	return encodeDataLine( line );
}
	

const char * MLOFile::sender()
{
	char buff[80];

	if (fax_sender[0])
		return fax_sender;

	memset( fax_sender, ' ', 20 );
	fax_sender[20] = 0;

	at(0);
	init();

	while (!atEnd()) {
		readLine(buff, 80);
		if (!strncmp(buff, "+FTI:", 5)) {
			char * p, * q;

			if (!(p = strchr(buff, '\"')))
				return fax_sender;
			p++;
			if (!(q = strchr(p, '\"')))
				return fax_sender;
			*q = 0;
			strncpy(fax_sender, p, 20);

			return fax_sender;
		}
	}

	return fax_sender;
}


int MLOFile::pages()
{
	char buff[80];

	if (fax_pages)
		return fax_pages;

	init_data();

	at(0);
	while (!atEnd()) {
		readLine(buff, 80);
		if (!strncmp(buff, "CONNECT", 7))
			fax_pages++;
	}

	return fax_pages;
}


bool MLOFile::gotoPage( int page )
{
	char buff[80];

	if (fax_page != page) {
		page_width = 1728;
		page_height = 0;
		page_used_height = 0;
		page_length = PAGE_LEN_A4;
		page_lpi = 98;
		page_bit_rate = 2400;
		page_2d = FALSE;
	}

	init_data();

	at(0);
	while (!atEnd()) {
		readLine(buff, 80);
		if (!strncmp(buff, "+FCS:", 5)) {
			int vr = 0, br = 0, wd = 0, ln = 0, df = 0;
			static int lpis[2] = { 98, 196 };
			static int ratings[6] = { 2400, 4800, 7200, 9600, 12000, 14400 };
			static int widths[5] = { 1728, 2048, 2432, 1216, 864 };

			sscanf(buff, "+FCS:%d,%d,%d,%d,%d", &vr, &br, &wd, &ln, &df);
			if ((vr >= 0) && (vr < 2))
				page_lpi = lpis[vr];
			if ((br >= 0) && (br < 6))
				page_bit_rate = ratings[br];
			if ((wd >= 0) && (wd < 5))
				page_width = widths[wd];
			if ((ln >= 0) && (ln < 3))
				page_length = ln;
			page_2d = (df == 1);
		} else if (!strncmp(buff, "CONNECT", 7)) {
			if (!(--page)) {
				refline[0] = page_width;
				refline[1] = -1;
				fax_page = page;
				return TRUE;
			}
		}
	}

	return FALSE;
}


bool MLOFile::readPageInfo( int page, page_info_t * info )
{
	if (fax_page != page)
		if (!gotoPage( page ))
			return FALSE;

	info->lpi = page_lpi;
	info->bit_rate = page_bit_rate;
	info->width = page_width;
	info->length = page_length;
	info->comp_2d = page_2d;

	return TRUE;
}


int MLOFile::pageHeight( int page )
{
	int eols;
	
	if ((fax_page == page) && page_height)
		return page_height;

	if (!gotoPage( page ))
		return 0;

	while (skipEol()) {
		eols = 1;
		if (!getDataBits(12)) {
#ifdef MLOFILE_DEBUG
			fprintf(stderr, "MLOFile: Unexpected end of file in pageHeight( %d ).\n", page);
#endif
			return page_height;
		}
		while (codeMatches(0x01, 12)) {
			cutDataBits(12);
			if ((!getDataBits(12)) || (eols++ == 6))
				return page_height;
		} 
		page_height++;
	}

	return page_height;
}


int MLOFile::pageUsedHeight( int page )
{
	int lines;
	short rle_buf[MAX_RLE_BUFF];

	if ((fax_page == page) && page_used_height)
		return page_used_height;

	if (!(lines = pageHeight( page )))
		return 0;

	if (!gotoPage( page ))
		return 0;

	page_height = lines;

	for (lines=0; lines < page_height; lines++) {
		decodeDataLine( rle_buf, TRUE );
		if ((rle_buf[0] != -1) && (rle_buf[1] != -1))
			page_used_height = lines + 1;
	}

	return page_used_height;
}

	
void MLOFile::init()
{
	init_data();

	fax_page = 0;
	fax_pages = 0;

	fax_sender[0] = 0;

	page_width = 0;
	page_height = 0;
	page_used_height = 0;
	page_length = 0;
	page_lpi = 0;
	page_bit_rate = 0;
	page_2d = FALSE;

	refline[2432] = -1;
}


void MLOFile::init_data()
{
	data_mode = FALSE;

	data_buff = 0;
	data_width = 0;
}


int MLOFile::getDataCh()
{
	int c;

	while ((c = getch()) == CDLE) {
		if ((c = getch()) == CDLE)
			return c;
		if ((c == CETX) || (c == -1)) {
			data_mode = FALSE;
			return -1;
		}
	}

	if (c == -1)
		data_mode = FALSE;

	return c;
}
	

bool MLOFile::getDataBits( uchar len )
{
	int c;

	while (data_width < len) {
		if ((c = getDataCh()) == -1)
			return FALSE;
		data_buff = (data_buff << 8) | (ulong)xchg_endian[c];
		data_width += 8;
	}

	return TRUE;
}


void MLOFile::cutDataBits( uchar width )
{
	data_width -= width;
	data_buff &= ((1 << data_width) - 1);
}


bool MLOFile::codeMatches( const uchar code, const uchar width )
{
	return (ushort)(data_buff >> (data_width - width)) == (ushort)code;
}


short MLOFile::findMatch( const hcode_t table[] )
{
	int i;

	for (i=0; table[i].width; i++)
		if (codeMatches( table[i].code, table[i].width )) {
			cutDataBits(table[i].width);
			return table[i].len;
		}

	return -1;
}


bool MLOFile::skipEol()
{
	if (!getDataBits(12))
		return FALSE;

	while (data_mode && (!codeMatches(0x01, 12))) {
		if ((data_buff >> (data_width - 11)) & 0x01)
			cutDataBits(11);
		else
			cutDataBits(1);
		if (!getDataBits(12))
			return FALSE;
	}

	if (data_mode)
		cutDataBits(12);
		
	return data_mode;
}


void MLOFile::fillDataLine( uchar * line, short pos, short len )
{
	short end, a, b;
	uchar a_mask, b_mask;

	if (!len)
		return;

	end = pos + len - 1;

	a = pos >> 3;
	b = end >> 3;

	a_mask = (uchar)(0xff <<  (pos & 0x07));
	b_mask = (uchar)(0xff >> ((end & 0x07) ^ 0x07));

	if (a == b)
		line[a] |= (a_mask & b_mask);
	else {
		line[a++] |= a_mask;
		while (a < b)
			line[a++] = 0xff;
		line[b] |= b_mask;
	}
}


short MLOFile::getRunLength( const hcode_t table[], uchar max_width )
{
	short len1, len2;

	if (!getDataBits( max_width )) {
#ifdef MLOFILE_DEBUG
		fprintf(stderr, "MLOFile: Could not get bits for make-up/terminating code.\n");
#endif
		return -1;
	}

	if ((len1 = findMatch( table )) < 64) {
#ifdef MLOFILE_DEBUG
		if (len1 < 0)
			fprintf(stderr, "MLOFile: Could not find make-up/terminating code.\n");
#endif
		return len1;
	}

	if (!getDataBits( max_width )) {
#ifdef MLOFILE_DEBUG
		fprintf(stderr, "MLOFile: Could not get bits for terminating code.\n");
#endif
		return -1;
	}

	len2 = findMatch( table );

	if ((len2 == -1) || (len2 > 63)) {
#ifdef MLOFILE_DEBUG
		fprintf(stderr, "MLOFile: Could not find terminating code.\n");
#endif
		return -1;
	}

	return len1 | len2;
}


bool MLOFile::decode1DDataLine( short run_lengths[], bool probeonly )
{
	short pos = 0;
	int count = 0;
	bool retval = TRUE;

	while ((pos < page_width) && (count < MAX_RLE_BUFF)) {
		if ((run_lengths[count] = getRunLength( white_codes, 12 )) == -1) {
			retval = FALSE;
			break;
		}
		pos += run_lengths[count++];
		if ((pos >= page_width) || (count >= MAX_RLE_BUFF))
			break;
		if ((run_lengths[count] = getRunLength( black_codes, 13 )) == -1) {
			retval = FALSE;
			break;
		}
		if (pos + run_lengths[count] > page_width)
			run_lengths[count] = page_width - pos;
		pos += run_lengths[count++];

		if (probeonly)
			break;
	}

	if (!count)
		run_lengths[count++] = 0;

	run_lengths[count-1] += (page_width - pos);
	run_lengths[count] = -1;
		
	return retval;
}


bool MLOFile::decode2DDataLine( short run_lengths[], bool probeonly )
{
	short mode, a0 = 0, b1 = 0;
	int count = 0, ref_count = 0, len;

	memset(run_lengths, 0, MAX_RLE_BUFF * sizeof(short));

	while ((a0 < page_width) && (count < MAX_RLE_BUFF)) {
		if (!getDataBits(7)) {
			run_lengths[count] = -1;
			return FALSE;
		}
		mode = findMatch( mode_codes );

		while (((b1 <= a0) && (b1 < page_width)) || ((count & 1) == (ref_count & 1)))
			b1 += refline[ref_count++];

		switch (mode) {
			case MODE_P:
				b1 += refline[ref_count++];
				run_lengths[count] += b1-a0;
				a0 = b1;
				break;
			case MODE_H:
				if (count & 1) {
					len = getRunLength( black_codes, 13 );
					run_lengths[count++] += len;
					a0 += len;
					len = getRunLength( white_codes, 12 );
				}
				else {
					len = getRunLength( white_codes, 12 );
					run_lengths[count++] += len;
					a0 += len;
					len = getRunLength( black_codes, 13 );
				}
				run_lengths[count++] += len;
				a0 += len;
				break;
			case MODE_V0:
				run_lengths[count++] += b1-a0;
				a0 = b1;
				break;
			case MODE_VR_1:
				run_lengths[count++] += b1+1-a0;
				a0 = b1+1;
				break;
			case MODE_VR_2:
				run_lengths[count++] += b1+2-a0;
				a0 = b1+2;
				break;
			case MODE_VR_3:
				run_lengths[count++] += b1+3-a0;
				a0 = b1+3;
				break;
			case MODE_VL_1:
				run_lengths[count++] += b1-1-a0;
				a0 = b1-1;
				b1 -= refline[--ref_count];
				break;
			case MODE_VL_2:
				run_lengths[count++] += b1-2-a0;
				a0 = b1-2;
				b1 -= refline[--ref_count];
				break;
			case MODE_VL_3:
				run_lengths[count++] += b1-3-a0;
				a0 = b1-3;
				b1 -= refline[--ref_count];
				break;
			default:
				run_lengths[count] = -1;
				return FALSE;
		}

		if (probeonly && (count > 1))
			break;
	}

	run_lengths[count] += (page_width - a0);

	if (run_lengths[count])
		count++;

	run_lengths[count] = -1;

	return TRUE;
}


void MLOFile::decodeDataLine( short line[], bool probeonly )
{
	bool one_dim = TRUE, retval;

	if (!skipEol()) {
		memcpy(line, refline, sizeof(refline));
		return;
	}

	if (page_2d) {
		if (!getDataBits(1)) {
			memcpy(line, refline, sizeof(refline));
			return;
		}
		one_dim = codeMatches(0x01, 1);
		cutDataBits(1);
	}

	if (one_dim)
		retval = decode1DDataLine( line, probeonly );
	else
		retval = decode2DDataLine( line, probeonly );
	
	if (retval)
		memcpy(refline, line, sizeof(refline));
	else
		memcpy(line, refline, sizeof(refline));
}


int MLOFile::storeDataCode( uchar dest[], int pos, const hcode_t & code )
{
	ushort data_buff;
	uchar width, *d;
	int sh, ln;

	width = code.width;
	data_buff = (ushort)code.code << (16-width);
	d = &dest[pos >> 3];

	sh = pos & 0x07;
	ln = 8-sh;
	*d = (*d & (0xff << ln)) | (uchar)(data_buff >> (8+sh));

	if (ln > width)
		ln = width;
	data_buff <<= ln;
	width -= ln;

	while (width) {
		d++;
		*d = (uchar)(data_buff >> 8);
		
		if (width > 8) {
			width-=8;
			data_buff <<= 8;
		}
		else
			width = 0;
	}
	
	return pos + code.width;
}


int MLOFile::encodeDataLine( uchar dest[] )
{
	static const hcode_t eol = {0x01, 12, 0};
	static hcode_t fill = {0x00, 0, 0};
	int i = 0, pos = 0;
	short length;

	while (refline[i] != -1) {
		length = refline[i];
		
		if (i & 0x01) {
			if (length > 63)
				pos = storeDataCode( dest, pos, black_codes[black_makeup_lut[length >> 6]] );
			pos = storeDataCode( dest, pos, black_codes[black_term_lut[length & 0x3f]] );
		}
		else {
			if (length > 63)
				pos = storeDataCode( dest, pos, white_codes[white_makeup_lut[length >> 6]] );
			pos = storeDataCode( dest, pos, white_codes[white_term_lut[length & 0x3f]] );
		}

		i++;
	}

	
	if ((pos & 0x07) != 0x04) {
		fill.width = (0x0c - (pos & 0x07)) & 0x07;
		pos = storeDataCode( dest, pos, fill );
	}

	pos = storeDataCode( dest, pos, eol );

	return pos >> 3;
}
