/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "kmlofax.h"
#include "kmlofax.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <unistd.h>

#include <X11/Xlib.h>
#undef TrueColor	// Avoid Qt <-> X11 conflict

#include <qkeycode.h>
#include <qpopupmenu.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qglobal.h>

#include <kiconloader.h>
#include <ktopwidget.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kstatusbar.h>
#include <kmsgbox.h>
#include <kapp.h>

#include "faxview.h"
#include "preferences.h"
#include "senderaliases.h"
#include "preview.h"
#include "printdlg.h"
#include "maildlg.h"
#include "global.h"


#define TOOL_ID_DL   0
#define TOOL_ID_VIEW 2
#define TOOL_ID_PRT  3
#define TOOL_ID_MAIL 4
#define TOOL_ID_DEL  5



KMLOFaxApp::KMLOFaxApp( int & argc, char ** argv, const QString & rAppName ) : KApplication( argc, argv, rAppName )
{
}


bool KMLOFaxApp::x11EventFilter( XEvent * ev )
{
	if (KApplication::x11EventFilter( ev ))
		return TRUE;

	if (fw) {
		if (ev->xany.window == fw->window) {
			fw->x11EventFilter( ev );
		}
	}

	return FALSE;
}



KMLOFax::KMLOFax( const char * name ) : KTopLevelWidget( name )
{
	QPixmap pix;
	KIconLoader * kil = mykapp->getIconLoader();
	KsConfig = mykapp->getConfig();

	prefs = new Preferences( 0, "preferences" );
	CHECK_PTR( prefs );
	prefs->reset();

	view = new FaxView( this, "faxview" );
	CHECK_PTR( view );
	view->setMinimumSize( 0, 0 );
	view->loadDir();
	setView( view );

	downloaddlg = new DownloadDlg( 0, "downloaddlg" );
	CHECK_PTR( downloaddlg );
	preview = new Preview( 0, "preview" );
	CHECK_PTR( preview );
	printdlg = new PrintDlg( 0, "printdlg" );
	CHECK_PTR( printdlg );
	maildlg = new MailDlg( 0, "maildlg" );
	CHECK_PTR( maildlg );

	QPopupMenu * file_menu = new QPopupMenu( 0, "mainfilemenu" );
	CHECK_PTR( file_menu );
	menu_item_dl = file_menu->insertItem(i18n("&Download"), this, SLOT(downloadFax()), CTRL+Key_D );
	file_menu->insertSeparator();
	menu_item_view = file_menu->insertItem(i18n("&View..."), this, SLOT(viewFax()), CTRL+Key_V);
	menu_item_prt = file_menu->insertItem(i18n("&Print..."), this, SLOT(printFax()), CTRL+Key_P);
	menu_item_mail = file_menu->insertItem(i18n("&Mail..."), this, SLOT(mailFax()), CTRL+Key_M);
	menu_item_del = file_menu->insertItem(i18n("De&lete"), this, SLOT(deleteFax()), Key_Delete);
	menu_item_ali = file_menu->insertItem(i18n("Sender &Alias..."), this, SLOT(senderAlias()));
	file_menu->insertSeparator();
	file_menu->insertItem(i18n("P&urge Directory"), this, SLOT(purgeDir()));
	file_menu->insertItem(i18n("&Rebuild Directory"), this, SLOT(rebuildDir()));
	file_menu->insertSeparator();
	file_menu->insertItem(i18n("&Quit"), this, SLOT(quit()), CTRL+Key_Q);

	QPopupMenu * opt_menu = new QPopupMenu( 0, "mainoptionsmenu" );
	CHECK_PTR( opt_menu );
	opt_menu->insertItem(i18n("&Preferences..."), this, SLOT(showPrefs()));
	opt_menu->insertItem(i18n("Sender &Aliases..."), this, SLOT(showSenderList()));
	opt_menu->insertSeparator();
	menu_item_tool = opt_menu->insertItem(i18n("Show &Toolbar"), this, SLOT(toggleToolbar()), CTRL+Key_T );
	opt_menu->setItemChecked( menu_item_tool, TRUE );
	menu_item_status = opt_menu->insertItem(i18n("Show &Statusbar"), this, SLOT(toggleStatusbar()), CTRL+Key_S );
	opt_menu->setItemChecked( menu_item_status, TRUE );

	QPopupMenu * help_menu = new QPopupMenu( 0, "mainhelpmenu" );
	CHECK_PTR( help_menu );
	help_menu->insertItem(i18n("&Contents"), this, SLOT(invokeHelp()), Key_F1);
	help_menu->insertSeparator();
	help_menu->insertItem(i18n("&About kmlofax..."), this, SLOT(showAbout()));

	menuBar = new KMenuBar( this, "mainmenubar" );
	CHECK_PTR( menuBar );
	menuBar->insertItem(i18n("&File"), file_menu);
	menuBar->insertItem(i18n("&Options"), opt_menu);
	menuBar->insertSeparator();
	menuBar->insertItem(i18n("&Help"), help_menu);

	setMenu(menuBar);
	menuBar->show();

	toolBar = new KToolBar( this, "maintoolbar" );
	CHECK_PTR( toolBar );

	pix = kil->loadIcon("filednl.xpm");
	toolBar->insertButton(pix, TOOL_ID_DL, SIGNAL(clicked()), this, SLOT(downloadFax()), true, i18n("Download Facsimile"));
	toolBar->insertSeparator( 1 );
	pix = kil->loadIcon("print_preview.xpm");
	toolBar->insertButton(pix, TOOL_ID_VIEW, SIGNAL(clicked()), this, SLOT(viewFax()), true, i18n("View Facsimile"));
	pix = kil->loadIcon("fileprint.xpm");
	toolBar->insertButton(pix, TOOL_ID_PRT, SIGNAL(clicked()), this, SLOT(printFax()), true, i18n("Print Facsimile"));
	pix = kil->loadIcon("send.xpm");
	toolBar->insertButton(pix, TOOL_ID_MAIL, SIGNAL(clicked()), this, SLOT(mailFax()), true, i18n("Mail Facsimile"));
	pix = kil->loadIcon("delete.xpm");
	toolBar->insertButton(pix, TOOL_ID_DEL, SIGNAL(clicked()), this, SLOT(deleteFax()), true, i18n("Delete Facsimile"));
	toolBar->insertSeparator( 6 );
	pix = kil->loadIcon("tick.xpm");
	toolBar->insertButton(pix, 7, SIGNAL(clicked()), this, SLOT(showPrefs()), true, i18n("Preferences"));
	pix = kil->loadIcon("help.xpm");
	toolBar->insertButton(pix, 8, SIGNAL(clicked()), this, SLOT(invokeHelp()), true, i18n("Help Contents"));
	pix = kil->loadIcon("exit.xpm");
	toolBar->insertButton(pix, 9, SIGNAL(clicked()), this, SLOT(quit()), true, i18n("Quit"));
	toolBar->alignItemRight(9, true);

	addToolBar(toolBar);
	toolBar->show();
	toolBarOn = TRUE;

	statusBar = new KStatusBar( this, "mainstatusbar" );
	CHECK_PTR( statusBar );
	statusBar->insertItem( "", 0 );
	setStatusBar( statusBar );
	statusBar->show();
	statusBarOn = TRUE;

	newPrefs();

	gotSelection( FALSE );

	connect(prefs, SIGNAL(apply_clicked()), SLOT(newPrefs()));

	connect(downloaddlg, SIGNAL(message(const char *)), SLOT(fetchMessage(const char *)) );
	connect(downloaddlg, SIGNAL(gotFile(const char *, const char *, int)), SLOT(fetchFile(const char *, const char *, int)) );
	connect(downloaddlg, SIGNAL(done()), SLOT(fetchDone()) );

	connect( view, SIGNAL(hasSelection(bool)), SLOT(gotSelection(bool)) );
	connect( view, SIGNAL(viewRequested(FaxViewItem *)), SLOT(viewRequestedSlot(FaxViewItem *)) );
	connect( view, SIGNAL(contextView()), SLOT(viewFax()) );
	connect( view, SIGNAL(contextPrint()), SLOT(printFax()) );
	connect( view, SIGNAL(contextMail()), SLOT(mailFax()) );
	connect( view, SIGNAL(contextDelete()), SLOT(deleteFax()) );
	connect( view, SIGNAL(contextAlias()), SLOT(senderAlias()) );

	resize( 650, 400 );

	show();
}


KMLOFax::~KMLOFax()
{
	view->saveDir();

	delete menuBar;
	delete toolBar;
	delete statusBar;
}


void KMLOFax::toggleToolbar()
{
	toolBarOn = !toolBarOn;

	menuBar->setItemChecked( menu_item_tool, toolBarOn );
	if (toolBarOn)
		toolBar->show();
	else
		toolBar->hide();
	updateRects();
}


void KMLOFax::toggleStatusbar()
{
	statusBarOn = !statusBarOn;

	menuBar->setItemChecked( menu_item_status, statusBarOn );
	if (statusBarOn)
		statusBar->show();
	else
		statusBar->hide();
	updateRects();
}


void KMLOFax::downloadFax()
{
	if (downloaddlg->startDownload( prefs->modDevice(), prefs->modSpeed() )) {
		menuBar->setItemEnabled( menu_item_dl, FALSE );
		toolBar->setItemEnabled( TOOL_ID_DL, FALSE );
		statusBar->changeItem( i18n("Looking for new facsimile..."), 0 );
	}
}


void KMLOFax::viewFax()
{
	if (view->faxItemName()) {
		preview->showFax( view->faxItemName(), view->faxItemAlias() );
		view->setFaxItemNew( FALSE );
	}
}


void KMLOFax::printFax()
{
	if (view->faxItemName()) {
		printdlg->printFax( view->faxItemName(), view->faxItemAlias() );
		view->setFaxItemNew( FALSE );
	}
}


void KMLOFax::mailFax()
{
	if (view->faxItemName()) {
		maildlg->mailFax( view->faxItemName(), view->faxItemAlias() );
	}
}


void KMLOFax::deleteFax()
{
	if (view->faxItemName()) {
		if (view->faxItemNew()) {
			KMsgBox::message( 0, i18n("File Error"), i18n("Cannot delete new facsimile file.\nPlease view or print facsimile first."), KMsgBox::EXCLAMATION);
			return;
		}

		if (KMsgBox::yesNo( 0, i18n("Question"), i18n("Really delete file?"), KMsgBox::QUESTION ) == 1) {
			unlink(expand_path(view->faxItemName()));
			view->setFaxItemFile( FALSE );
			gotSelection( TRUE );
		}
	}
}


void KMLOFax::purgeDir()
{
	if (KMsgBox::yesNo( 0, i18n("Question"), i18n("Really remove entries with deleted files?"), KMsgBox::QUESTION ) == 1)
		view->purgeDir();
}


void KMLOFax::rebuildDir()
{
	if (KMsgBox::yesNo( 0, i18n("Question"), i18n("Really discard current directory and scan for existing facsimile files?"), KMsgBox::QUESTION ) == 1)
		view->rebuildDir();
}


void KMLOFax::senderAlias()
{
	if (view->faxItemSender())
		view->aliases->setAlias( view->faxItemSender() );
}


void KMLOFax::newPrefs()
{
	bool tooltips;

	tooltips = prefs->genTooltips();

	prefs->addToolTips( tooltips );
	view->aliases->addToolTips( tooltips );
	printdlg->addToolTips( tooltips );
	maildlg->addToolTips( tooltips );
}


void KMLOFax::showPrefs()
{
	if (prefs->isVisible())
		prefs->raise();
	else
		prefs->show();
}


void KMLOFax::showSenderList()
{
	if (view->aliases->isVisible())
		view->aliases->raise();
	else
		view->aliases->show();
}


void KMLOFax::invokeHelp()
{
	mykapp->invokeHTMLHelp("kmlofax/kmlofax.html", "");
}


void KMLOFax::showAbout()
{
	QString msg;

	msg.sprintf("KMLOFax %s\n", VERSION);
	msg += i18n("Copyright  1999-2000 Oliver Gantz <o.gantz@tu-bs.de>.\n\n" \
		"A utility to process facsimile received with the\n" \
		"ELSA MicroLink(tm) Office modem.\n\n" \
		"This program is free software and is distributed under the terms\n" \
		"of the GNU General Public License.\n\n" \
		"ELSA and MicroLink are trademarks of ELSA AG, Aachen.");
	KMsgBox::message(0, i18n("About KMLOFax"), msg);
}


void KMLOFax::gotSelection( bool on )
{
	menuBar->setItemEnabled( menu_item_ali, on );

	if (on)
		on = view->faxItemFile();

	menuBar->setItemEnabled( menu_item_view, on );
	menuBar->setItemEnabled( menu_item_prt, on );
	menuBar->setItemEnabled( menu_item_mail, on );
	menuBar->setItemEnabled( menu_item_del, on );
	toolBar->setItemEnabled( TOOL_ID_VIEW, on );
	toolBar->setItemEnabled( TOOL_ID_PRT, on );
	toolBar->setItemEnabled( TOOL_ID_MAIL, on );
	toolBar->setItemEnabled( TOOL_ID_DEL, on );
}


void KMLOFax::viewRequestedSlot( FaxViewItem * item )
{
	preview->showFax( view->faxItemName( item ), view->faxItemAlias( item ) );
	view->setFaxItemNew( item, FALSE );
}


void KMLOFax::quit()
{
	mykapp->quit();
}


void KMLOFax::fetchMessage( const char * msg )
{
	statusBar->changeItem( msg, 0 );
}


void KMLOFax::fetchFile( const char * name, const char * time, int size )
{
	char buff[40];

	sprintf(buff, i18n("Received file %s."), name);
	statusBar->changeItem( buff, 0 );
	view->addFaxItem( name, time, size );	
}


void KMLOFax::fetchDone()
{
	menuBar->setItemEnabled( menu_item_dl, TRUE );
	toolBar->setItemEnabled( TOOL_ID_DL, TRUE );
	view->saveDir();
}
