/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

#include "netware.h"
#include "inter.h"
#include "server.h"

#define i18n(s)	s


static int check_common(char *reply, char *errors, int retval)
{
	if (strstr(errors, "Could not open connection")) {
		iwrite_fail(i18n("Unable to connect to print server"));
		return 0;
	}
	if (retval) {
		fprintf(stderr, "klp: child exited with status %d\n", retval);
		iwrite_fail(i18n("Error executing child process"));
		return 0;
	}
	return 1;
}


/*
 *	parse_next_job
 *
 *	parse somthing like:
 *
 *	1     username   stupid filename of 27 bytes   Job Active  UNKNOWN   06ba2001
 *
 *      \_/   \______/   \_________________________/   \________/  \_____/   \______/
 *
 *      rank  user       jobname                       status      form      job id
 *
 *	Note that status and form are not used.
 *	Note that the size of the job is unknown.
 *	Note that we only count "Job Active" entries.
 *	returns 0 on fail, 1 on success.
 */
static struct queue_item *parse_next_job(const char *line)
{
	struct queue_item *item = NULL;
	char *line_cp;
	char *l;
	char *p;

	item = xmalloc(sizeof(struct queue_item));
	*item = empty_queue_item;
	while (isspace(*line))
		++line;
	line_cp = xstrdup(line);
	if (!(l = strstr(line_cp, "Job Active")))
		goto fail;
	*(l++) = '\0';
	p = l + strlen(l);
	while ((p - l > 0) && isspace(*(p - 1)))
		--p;
	*p = '\0';
	while ((p - l > 0) && isalnum(*(p - 1)))
		--p;
	if ((p == l) || !isspace(*(p - 1)) || !isalnum(*p))
		goto fail;
	item->job_id = xstrdup(p);
	l = line_cp;
	if (!isdigit(*l))
		goto fail;
	while (isdigit(*l))
		++l;
	if (!isspace(*l))
		goto fail;
	*(l++) = '\0';
	item->rank = atoi(line_cp);
	while (isspace(*l))
		++l;
	if (!isgraph(*l))
		goto fail;
	p = l;
	while(isgraph(*l))
		++l;
	if (!isspace(*l))
		goto fail;
	*(l++) = '\0';
	item->user = xstrdup(p);

	while (isspace(*l))
		++l;
	p = l + strlen(l);
	while ((p - l > 0) && isspace(*(p - 1)))
		--p;
	*p = '\0';
	item->job_name = xstrdup(l);
	free(line_cp);
	printf("rank = `%d'\n", item->rank);
	printf("user = `%s'\n", item->user);
	printf("name = `%s'\n", item->job_name);
	printf("id   = `%s'\n", item->job_id);
	return item;

fail:
	free(line_cp);
	free_queue_item(item);
	return NULL;
}


struct queue *netware_q_parse(char *reply, char *errors, int retval)
{
	struct queue *queue;
	char *this_line;
	char *next_line;
	struct queue_item *item;

	if (!check_common(reply, errors, retval))
		return NULL;
	queue = xmalloc(sizeof(struct queue));
	*queue = empty_queue;
	queue->no_queue = 1;
	next_line = reply;
	do {
		this_line = next_line;
		next_line = strchr(this_line, '\n');
		if (next_line)
			*(next_line++) = '\0';
		if ((item = parse_next_job(this_line))) {
			queue->items = xrealloc(queue->items, (queue->item_count + 1) * sizeof(struct queue_item *));
			queue->items[queue->item_count++] = item;
		} else if (strstr(this_line, "queue is ready"))
			queue->no_queue = 0;
	} while (next_line);
	return queue;
}


void netware_p_parse(const char * const files[], char *reply, char *errors, int retval)
{
	if (!check_common(reply, errors, retval))
		return;
	iwrite_char('p');
}


void netware_r_parse(const char * const jobids[], char *reply, char *errors, int retval)
{
	if (!check_common(reply, errors, retval))
		return;
	iwrite_char('r');
}



static char *printer = NULL;
static char *host = NULL;


char *netware_set_printer(const char *h, const char *p)
{
	if (!p) {
		xfree(host);
		xfree(printer);
		return NULL;
	}
	host = xstrdup(h ? h : fq_hostname);
	printer = xstrdup(p);
	return NULL;
}


int netware_q_main(void)
{
	drop_root();
	execlp("nwqstat", "nwqstat", "--server", host, "--queue", printer, (char *)0);
	perror("klp: execlp");
	return EXIT_FAILURE;
}


int netware_p_main(const char * const files[])
{
	char **argv;
	int argc = 0;

	drop_root();
	argv = xmalloc(8 * sizeof(char *));
	argv[argc++] = "nwprint";
	argv[argc++] = "--server";
	argv[argc++] = xstrdup(host);
	argv[argc++] = "--queue";
	argv[argc++] = xstrdup(printer);
	argv[argc++] = "--username";
	argv[argc++] = xstrdup(username);
	argv[argc] = NULL;
	while (*files) {
		argv = xrealloc(argv, (argc + 2) * sizeof(char *));
		argv[argc++] = xstrdup(*(files++));
		argv[argc] = NULL;
	}
	execvp("nwprint", argv);
	perror("klp: execlp");
	return EXIT_FAILURE;
}


int netware_r_main(const char * const jobids[])
{
	char **argv;
	int argc = 0;

	drop_root();
	argv = xmalloc(6 * sizeof(char *));
	argv[argc++] = "nwdelqjob";
	argv[argc++] = "--server";
	argv[argc++] = xstrdup(host);
	argv[argc++] = "--queue";
	argv[argc++] = xstrdup(printer);
	argv[argc] = NULL;
	while (*jobids) {
		argv = xrealloc(argv, (argc + 2) * sizeof(char *));
		argv[argc++] = xstrdup(*(jobids++));
		argv[argc] = NULL;
	}
	execvp("nwdelqjob", argv);
	perror("klp: execlp");
	return EXIT_FAILURE;
}
