//
// $Id: kload.cpp,v 1.9 1999/03/06 15:22:43 pi Exp pi $
//
// $Log: kload.cpp,v $
// Revision 1.9  1999/03/06 15:22:43  pi
// Intermediate Checkin
//
// Revision 1.8  1998/10/27 22:04:13  pi
// Bug Fixes for V 0.61
//
// Revision 1.7  1998/10/23 11:59:08  pi
// Checkin for V 0.6
//
// Revision 1.6  1998/10/04 15:57:56  pi
// After first version of Options dialog
//
// Revision 1.5  1998/08/23 16:05:09  pi
// Optionen-Dialog eingebaut
//
// Revision 1.4  1998/08/09 13:53:33  pi
// Menu bar hideable, options menu on right mouse button
//
// Revision 1.3  1998/07/27 19:26:43  pi
// panel visibility mgmt. terminated
//
// 
// Hauptprogramm fuer kload:
// Definition des Applikations-Widget
// 
// 


extern "C" {
    #include <sys/utsname.h>
}

#include <qaccel.h>
#include <kapp.h>
#include <kconfig.h>
#include <kmenubar.h>
#include <kmsgbox.h> 

#include "kload.h"


// PanelSet Constructor: allocates panel objects
PanelSet::PanelSet(QWidget *parent, const char *name )
        : QWidget(parent,name)
{
    alignment = A_VERT;
    //setMinimumSize( WIDTH,( HEIGHT);
 
    nPanels = 0;

    myPanel[0] = new CPUPanel(this);
    nPanels++;
 
    myPanel[1] = new LoadPanel(this);
    nPanels++;
    
    myPanel[2] = new RAMPanel(this);
    nPanels++;
    
    myPanel[3] = new PagePanel(this);
    nPanels++;
    
    for (int i = 0;i < nPanels;i++) {
        panelVis[i] = true;
        connect(myPanel[i],SIGNAL(rightMouseButtonPressed(QPoint)),
                this,SLOT(childRightMouse(QPoint)));
        description[i] = myPanel[i]->getDescription();
    }
    
    timerId = -1; // -1 seems to be impossible als return value of startTimer()
    setRedrawInterval(2); 
}


// Slot for right Mouse Click events in child panels
void PanelSet::childRightMouse(QPoint q)
{
    emit rightMouseButtonPressed(q);
}



// Mouse click event handler
void PanelSet::mousePressEvent(QMouseEvent *e)
{
    if (e->button() == RightButton) {
        emit rightMouseButtonPressed(mapToGlobal(e->pos()));
    }
}


// Set redraw Interval and restart timer
void PanelSet::setRedrawInterval(int sec) 
{
    redrawInterval = sec;
    
    if (timerId != -1) {    
        killTimer(timerId);
    }
    timerId = startTimer(sec * QSECOND);
}


// Triggered by The Timer, wake up all child panels
void PanelSet::timerEvent(QTimerEvent *t) 
{
    if (t->timerId() == timerId) {
        for (int i = 0;i < nPanels;i++) {
            myPanel[i]->wakeUp();
        } 
    }
}


// Return number of visible Panels
int PanelSet::numVisPanels(void) const
{
    int i, n;
    
    for (n = 0, i = 0;i < nPanels;i++) {
        n += (panelVis[i] ? 1 : 0);
    }
    
    return n;
}


// 
int PanelSet::setPanelVisibility(int i, bool vis)
{
    // out of range
    if (i >= nPanels)
        return -1;
    // forbid all Panels are hidden
    if (!vis && (numVisPanels() == 1) && panelVis[i])
        return -2;
        
    if (vis && !panelVis[i]) {
        myPanel[i]->show();
        panelVis[i] = true;
    }
    else if (!vis && panelVis[i]) {
        myPanel[i]->hide();
        panelVis[i] = false;
    }
    doResize();
    return 0;
}


void PanelSet::setAlignment(int align)
{
    alignment = align;
    doResize();
}


void PanelSet::resizeEvent(QResizeEvent *)
{
    doResize();
}


void PanelSet::doResize(void)
{
    int ph, pw, numVisiblePanels, i,offset;
    
    numVisiblePanels = numVisPanels();
    
    if (alignment == A_VERT) {
        pw = width();
        ph = (height() - (numVisiblePanels - 1) * BORDER) / numVisiblePanels;
        
        for (offset = 0, i = 0;i < nPanels;i++) {
            if (panelVis[i]) {
                myPanel[i]->setGeometry(0,
                                        offset,
                                        pw,
                                        ph);
                offset += ph + BORDER;
            }                       
            else {
                myPanel[i]->setSize(pw,ph);
            }
        }
    } 
    else { // alignment == A_HORIZ
        pw = (width() - (numVisiblePanels - 1) * BORDER) / numVisiblePanels;
        ph = height();
        
        for (offset = 0, i = 0;i < nPanels;i++) {
            if (panelVis[i]) {
                myPanel[i]->setGeometry(offset,
                                        0,
                                        pw,
                                        ph);
                offset += pw + BORDER;
            }
            else {
                myPanel[i]->setSize(pw,ph);
            }
        }
    }       
    
}


///////////////////////////////////////////////////////////////////////

static char *getHostname(void)
{
    static struct utsname u;
    
    if (!uname(&u)) {
        return u.nodename;
    }
    else  {
        return "???";
    }
}


///////////////////////////////////////////////////////////////////////
// KLoad Constructor: allocates menu and panel objects
KLoad::KLoad(const char *name )
        : KTopLevelWidget( name )
{
    panelSet = new PanelSet(this,"PanelSet");    
    setView(panelSet);
    
    connect(panelSet,SIGNAL(rightMouseButtonPressed(QPoint)),
            this,SLOT(popupMouseMenu(QPoint)));
    
    /////////////////////// Menu: File /////////////////////////////////
    mFile = new QPopupMenu;
    mFile->insertItem(i18n("E&xit"),qApp, SLOT(quit()),CTRL+Key_Q); 
 

    /////////////////////// Menu: Options /////////////////////////////////
    mOptions = new QPopupMenu;
    mOptions->insertItem(i18n("&Preferences"),this,SLOT(doOptions()));

    mOptions->insertSeparator();
    
    mHideShowMenu = mOptions->insertItem(i18n("&Hide Menu bar"),this, 
                                         SLOT(toggleMenuBar()));
    
    mOptions->insertSeparator();

    mOptions->insertItem(i18n("&Save Settings"),this,SLOT(writeConfig())); 

    /////////////////////// Menu: Help /////////////////////////////////
    mHelp = new QPopupMenu;
    mHelp->insertItem(i18n("&About"),this,SLOT(about()));
    
    /////////////////////// Main Menu //////////////////////////////////
    menu = new KMenuBar(this);
    CHECK_PTR(menu);
    menu->insertItem(i18n("&File"),mFile);
    menu->insertItem(i18n("&Options"),mOptions);
    menu->insertSeparator(); // help Item right justified
    menu->insertItem(i18n("&Help"),mHelp);
    menu->show();
    setMenu(menu);

    // Read Configuration saved to disk
    readConfig();
    
    // Generate caption
    char caption[100];
    sprintf(caption,"kload@%s",getHostname());    
    setCaption(caption);
    
    od = 0;
}


KLoad::~KLoad()
{
}


// Invoke or raise options dialog
void KLoad::doOptions()
{   
    struct OptionSet os;    
    
    os.nPanels = panelSet->getNPanels();
    
    os.redrawInterval = panelSet->getRedrawInterval();
    os.alignment = panelSet->getAlignment();
    
    for (int i = 0;i < os.nPanels;i++) {
        os.label[i] = panelSet->getDescription(i);
            
        os.panelVis[i] = panelSet->getPanelVisibility(i);
    
        os.dblClckAction[i] = panelSet->getDoubleClickCmd(i);
        
        os.lowAction[i]  = panelSet->getLowBoundCmd(i);
        os.highAction[i] = panelSet->getHighBoundCmd(i);
        os.lowWM[i]      = panelSet->getLowWM(i);
        os.highWM[i]     = panelSet->getHighWM(i);

    }
    
    if (0 == od) {        
        od = new OptDialog(os);
        connect(od,SIGNAL(destroyed()),SLOT(odDestroyed()));
        connect(od,SIGNAL(applyOptions(OptionSet)),
                   SLOT(applyOptions(OptionSet)));
    }

    
    od->set(os);
    od->show();
    od->raise();
}


void KLoad::odDestroyed()
{
    od = 0;
}


// Called upon applyOptions signal from OptionDialog
void KLoad::applyOptions(OptionSet os)
{
    panelSet->setRedrawInterval(os.redrawInterval);
    panelSet->setAlignment(os.alignment);
    
    for (int i = 0;i < os.nPanels;i++) {
        panelSet->setDoubleClickCmd(i,os.dblClckAction[i]);

        panelSet->setPanelVisibility(i,os.panelVis[i]);
        panelSet->setLowBoundCmd(i,os.lowAction[i]);
        panelSet->setHighBoundCmd(i,os.highAction[i]);
        panelSet->setLowWM(i,os.lowWM[i]);
        panelSet->setHighWM(i,os.highWM[i]);
    }
}


void KLoad::popupMouseMenu(QPoint p)
{
    mOptions->popup(p);
}


// Display a short message about the author
void KLoad::about()
{
    KMsgBox::message(0,i18n("About kload"),
                i18n("kload 0.71\nUwe Schneider <uwe@richard-schneider.de>\n\n"
                "For the KDE Project!"));
}                     


// Read configuration from standard config file
void KLoad::readConfig()
{
    QString help;
    bool tf;
    int w,h;
    
    KConfig *config;
    config = kapp->getConfig();
    
    config->setGroup("General");
    w = config->readNumEntry("Width",200);
    h = config->readNumEntry("Height",150);
    this->resize(w,h);

    int ri = config->readNumEntry("Sampling Interval",2);
    if (ri < 1) ri = 1;
    if (ri > 60) ri = 60;
    panelSet->setRedrawInterval(ri);
    
    help = config->readEntry("Alignment","Vertical");
    
    if (help == "Horizontal") {
        panelSet->setAlignment(A_HORIZ);
    } else {
        panelSet->setAlignment(A_VERT);
    }

    tf = config->readBoolEntry("Show Menu Bar",true);
    if (tf) 
        showMenuBar();
    else
        hideMenuBar();

    for (int i = 0;i < panelSet->getNPanels();i++) {
        config->setGroup(panelSet->getName(i));
        tf = config->readBoolEntry("Visible",true);
        panelSet->setPanelVisibility(i,tf) ;
        help = config->readEntry("Double Click Action","kvt -e top");
        panelSet->setDoubleClickCmd(i,help);
        help = config->readEntry("LWM Command","");
        panelSet->setLowBoundCmd(i,help);
        help = config->readEntry("HWM Command","");
        panelSet->setHighBoundCmd(i,help);
        help = config->readEntry("LWM","0");
        panelSet->setLowWM(i,atof(help));
        help = config->readEntry("HWM","0");
        panelSet->setHighWM(i,atof(help));
    }
}


// Write configuration to standard config file
void KLoad::writeConfig()
{
    int a;
    char help[80];
    KConfig *config;
    config = kapp->getConfig();
    
    config->setGroup("General");
    config->writeEntry("Width",this->width());
    config->writeEntry("Height",this->height());
    config->writeEntry("Sampling Interval",panelSet->getRedrawInterval());
    
    a = panelSet->getAlignment();
    config->writeEntry("Alignment",
                       (a == A_HORIZ) ? "Horizontal" : "Vertical");

    config->writeEntry("Show Menu Bar",menu->isVisible());
    
    for (int i = 0;i < panelSet->getNPanels();i++) {
        config->setGroup(panelSet->getName(i));
        config->writeEntry("Visible",panelSet->getPanelVisibility(i));
        config->writeEntry("Double Click Action",panelSet->getDoubleClickCmd(i));
        config->writeEntry("LWM Command",panelSet->getLowBoundCmd(i));
        config->writeEntry("HWM Command",panelSet->getHighBoundCmd(i));
        sprintf(help,"%.2f",panelSet->getLowWM(i));
        config->writeEntry("LWM",help);
        sprintf(help,"%.2f",panelSet->getHighWM(i));
        config->writeEntry("HWM",help);
    }
       
    config->sync();
}


void KLoad::toggleMenuBar() {
    if (menu->isVisible())
        hideMenuBar();
    else
        showMenuBar();
}


void KLoad::hideMenuBar()
{
    menu->hide();
    mOptions->changeItem(i18n("&Show Menu Bar"),mHideShowMenu);
    
    updateRects();
}


void KLoad::showMenuBar()
{
    menu->show();
    mOptions->changeItem(i18n("&Hide Menu Bar"),mHideShowMenu);

    updateRects();
}


int main( int argc, char **argv )
{
    KApplication *mykapp = new KApplication( argc, argv,"kload");
    KLoad w;
    w.show();

    mykapp->setMainWidget(&w);
    return mykapp->exec();
}
