/***************************************************************************
 *                               kfsstatus.cpp                             *
 *                            -------------------                          *
 *                     Source code for KFSStatus                           *
 *               -A program to analyse mounted filesystems                 *
 *                                                                         *
 *               KFSStatus is part of the KDE-Project                      *
 *                         http://www.kde.org                              *
 *                                                                         *
 *   Copyright (C) Feb - Dec 1997 Markus Wuebben                           *
 *                  E-Mail: markus.wuebben@kde.org                         *
 *                          wuebben@eure.de                                *
 *                     Web: http://math.cornell.edu/~wuebben/markus/       * 
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          * 
 *                                                                         *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             *
 *                                                                         *
 ***************************************************************************/

#include "kfsstatus.h"
#include <stdlib.h>

KFSMainWidget::KFSMainWidget(QWidget *parent, const char *name)
  : QDialog(parent, name)
{
  setMaximumSize(410,472);
  setMinimumSize(410,472);
  tabDialog = new QTabDialog(this,"main",FALSE,0);
  tabDialog->move(0,0);

  topLevel = new KFSStatus(tabDialog,"kfs");
  tabDialog->addTab(topLevel,"General");

  setupWidget = new KFSSetup(tabDialog,"setup");
  tabDialog->addTab(setupWidget,"Setup");

  aboutWidget = new KFSAbout(tabDialog,"about");
  tabDialog->addTab(aboutWidget,"About");

  connect(tabDialog,SIGNAL(applyButtonPressed()),qApp,SLOT(quit()) );
}
  

KFSSetup::KFSSetup(QWidget *parent=0, const char *name=0)
  : QDialog(parent, name)
{

  setMaximumSize(400,370);
  setMaximumSize(400,370);

  QString tempUsedColor;
  QString tempFreeColor;

  config = new KConfig();
  config = KApplication::getKApplication()->getConfig();

  config->setGroup("COLOR");

  tempUsedColor = config->readEntry("usedColor");
  tempFreeColor = config->readEntry("freeColor");

  if(!tempUsedColor.isNull())
    usedColor.setNamedColor(tempUsedColor);
  else
    usedColor = darkMagenta;
  
  if(!tempFreeColor.isNull())
    freeColor.setNamedColor(tempFreeColor);
  else
    freeColor = blue;

  box  = new QGroupBox(this,"C");
  box->setGeometry(5,10,390,350);

  usedColorOld = usedColor;
  freeColorOld = freeColor;

  QLabel *freeColorLabel;
  freeColorLabel = new QLabel(this,"fCL");
  freeColorLabel->setGeometry(52,50,160,20);
  freeColorLabel->setText("Color indicating free space:");

  QLabel *usedColorLabel;
  usedColorLabel = new QLabel(this,"uCL");
  usedColorLabel->setGeometry(52,115,160,20);
  usedColorLabel->setText("Color indicating used space:");

  QPushButton *okColorBt;
  okColorBt = new QPushButton("Apply",this,"oCB");
  okColorBt->setGeometry(35,320,70,30);
  connect(okColorBt,SIGNAL(clicked()),SLOT(okColorSlot()));

  QPushButton *cancelColorBt;
  cancelColorBt = new QPushButton("Recover",this,"cCB");
  cancelColorBt->setGeometry(287,320,70,30);
  connect(cancelColorBt,SIGNAL(clicked()),SLOT(recoverColorSlot()));

  freeFrame = new QFrame(this,"Free color frame");
  freeFrame->setGeometry(217,45,30,28);
  freeFrame->setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  freeFrame->setBackgroundColor(freeColor);
  freeFrame->setLineWidth(2);

  usedFrame = new QFrame(this,"Used color frame");
  usedFrame->setGeometry(217,115,30,28);
  usedFrame->setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  usedFrame->setBackgroundColor(usedColor);
  usedFrame->setLineWidth(2);
  
  QPushButton *configFree;
  configFree = new QPushButton("Change",this,"cF");
  configFree->setGeometry(267, 45, 70,30);
  connect(configFree,SIGNAL(clicked()),SLOT(configureColorFree()));

  QPushButton *configUsed;
  configUsed = new QPushButton("Change",this,"cU");
  configUsed->setGeometry(267,115,70,30);
  connect(configUsed,SIGNAL(clicked()),SLOT(configureColorUsed()));

}

void KFSSetup::paintEvent(QPaintEvent *)
{

  QPainter p;
  QBrush b;
  QBrush b1;
  p.begin(box);

  b.setColor(usedColor);
  b.setStyle(SolidPattern);
  p.setBrush(b);
  p.drawPie(130,190,120,120,1440,-3000);

  b1.setStyle(SolidPattern);
  b1.setColor(freeColor);
  p.setBrush(b1);
  p.drawPie(130,190,120,120,1440,2760);
  p.drawText(160,170,"Preview:");
  p.end();

}

void KFSSetup::configureColorFree()
{
  if(KColorDialog::getColor(freeColor) == QDialog::Rejected)
    return; // This has to be currentColor
  freeFrame->setBackgroundColor(freeColor);
}
void KFSSetup::configureColorUsed()
{
  if(KColorDialog::getColor(usedColor) == QDialog::Rejected)
    return; // This has to be currentColor
  usedFrame->setBackgroundColor(usedColor);
}

void KFSSetup::okColorSlot()
{
  QString tempUsed; 
  QString tempFree;

  tempUsed.sprintf("#%02x%02x%02x",usedColor.red(),usedColor.green(),
		   usedColor.blue());
  tempFree.sprintf("#%02x%02x%02x",freeColor.red(),freeColor.green(),
		   freeColor.blue());

  config->setGroup("COLOR");
  config->writeEntry("usedColor",tempUsed);
  config->writeEntry("freeColor",tempFree);
  config->sync();

  usedColorOld = usedColor;
  freeColorOld = freeColor;

  repaint();
}

void KFSSetup::recoverColorSlot()
{
  // let's recover the old colors.

  QString tempUsed; 
  QString tempFree;
 
  usedColor = usedColorOld;
  freeColor = freeColorOld;
  usedFrame->setBackgroundColor(usedColor);
  freeFrame->setBackgroundColor(freeColor);

  tempUsed.sprintf("#%02x%02x%02x",usedColor.red(),usedColor.green(),
		   usedColor.blue());
  tempFree.sprintf("#%02x%02x%02x",freeColor.red(),freeColor.green(),
		   freeColor.blue());

  config->setGroup("COLOR");
  config->writeEntry("usedColor",tempUsed);
  config->writeEntry("freeColor",tempFree);

  config->sync();
  repaint();

}


KFSAbout::KFSAbout(QWidget *parent=0,const char *name = 0)
  : QDialog(parent,name)
{
  setMaximumSize(400,370);
  setMinimumSize(400,370);
  
  box = new QGroupBox(this,"box");
  box->setGeometry(30,30,340,310);

  QPushButton *helpBt = new QPushButton("Help",this,"bt");
  helpBt->setGeometry(290,300,70,30);
  connect(helpBt,SIGNAL(clicked()),SLOT(startKdehelp()) );

}

void KFSAbout::startKdehelp()
{
  KApplication::getKApplication()->invokeHTMLHelp("","");
}

void KFSAbout::paintEvent(QPaintEvent *)
{
  QPainter p;
  p.begin(box);
  QString Temp;
  Temp = "K File System Control v0.1.4 Dec 7 1997\n";
  p.drawText(35,40,Temp);
  Temp = "This is a tool to analyse each mounted";
  p.drawText(35,80,Temp);
  Temp = "filesystem. It is part of the KDE-Project.";
  p.drawText(35,100,Temp);
  Temp = "KDE: http://www.kde.org/";
  p.drawText(35,120,Temp);
  Temp = "Author:";
  p.drawText(35,160,Temp);
  Temp = "Markus Wuebben";
  p.drawText(35,180,Temp);
  Temp = "markus.wuebben@kde.org";
  p.drawText(35,200,Temp);
  Temp = "http://math.cornell.edu/~wuebben/markus/";
  p.drawText(35,220,Temp);
  p.end();

}

 

  
KFSStatus::KFSStatus(QWidget *parent, const char *name)
  : QDialog(parent,name)
{
  
  if(!(fopen("/etc/mtab","r")))
     { QString str;
       str  = "I can't open /etc/mtab !!!\n\n";
       str += "You might want to check if you have such a file\n";
       str += "and if the permissons are set correctly!";
       QMessageBox::information(0,"Error!",str);
       exception();
     }


  theBox = new QGroupBox(this,"gp");
  theBox->setGeometry(5,10,390,360);
			  
  config = new KConfig();
  config = KApplication::getKApplication()->getConfig();
  config->setGroup("COLOR");

  tempUsedColor = config->readEntry("usedColor");
  tempFreeColor = config->readEntry("freeColor");


  if(!tempUsedColor.isNull())
    usedColor.setNamedColor(tempUsedColor);
  else
    usedColor = darkMagenta;
  
  if(!tempFreeColor.isNull())
    freeColor.setNamedColor(tempFreeColor);
  else
    freeColor = blue;

 
  usedAngle=0;
  availAngle=0;
  
  setMinimumSize(400,370);
  setMaximumSize(400,370);

  QLabel *FSystem = new QLabel(this,"The Filesystem");
  FSystem->setText("Device:");
  FSystem->setGeometry(65,53,80,15);

  device = new QComboBox(TRUE,this,"bezeich");
  device->setGeometry(190,50,130,25);


  theList = new QList<Device>;
  
  getFileSystems();

  connect(device,SIGNAL(activated(int)),SLOT(deviceChange(int)));

  frei_f = new QFrame(this,"Free color frame");
  frei_f->setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  frei_f->setGeometry(25,90,20,20);
  frei_f->setBackgroundColor(freeColor);
  frei_f->setLineWidth(2);

  belegt_f = new QFrame(this,"Used color frame");
  belegt_f->setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
  belegt_f->setGeometry(25,130,20,20);
  belegt_f->setBackgroundColor(usedColor);
  belegt_f->setLineWidth(2);

  QFrame *hline = new QFrame(this,"hori-line");
  hline->setFrameStyle(QFrame::HLine | QFrame::Sunken);
  hline->setGeometry(20,170,367,5);
 
  fstype = new QLabel(this,"fstype");
  fstype->setGeometry(95,265,70,15);

  fsmounted = new QLabel(this,"fsmounted");
  fsmounted->setGeometry(95,313,90,15);
  
  fsid = new QLabel(this,"fsid");
  fsid->setGeometry(95,289,70,15);

  QPushButton *updateBt = new QPushButton("Update",this,"bt");
  updateBt->setGeometry(320,325,60,30);
  connect(updateBt,SIGNAL(clicked()),SLOT(deviceChangeI()) );

  setFsInfos();
  getFsStat(mymntent->mnt_dir);

 
}


KFSStatus::~KFSStatus(){
}

void KFSStatus::paintEvent(QPaintEvent *)
{
  QPainter p;
  QBrush b;
  QBrush b1;
  QString temp;
  
  tempUsedColor = config->readEntry("usedColor");
  tempFreeColor = config->readEntry("freeColor");

  if(!tempUsedColor.isNull())
    {usedColor.setNamedColor(tempUsedColor);
    belegt_f->setBackgroundColor(usedColor);
    }
  else
    {usedColor = darkMagenta;
    belegt_f->setBackgroundColor(usedColor);
    }

  if(!tempFreeColor.isNull())
    {freeColor.setNamedColor(tempFreeColor);
    frei_f->setBackgroundColor(freeColor);
    }
  else
    {freeColor = blue;
    frei_f->setBackgroundColor(freeColor);
    }

  p.begin(theBox);

  b.setColor(usedColor);
  b.setStyle(SolidPattern);
  p.setBrush(b);
  p.drawPie(185,210,120,120,1440,-usedAngle);

  b1.setStyle(SolidPattern);
  b1.setColor(freeColor);
  p.setBrush(b1);
  p.drawPie(185,210,120,120,1440,availAngle);

  p.drawText(60,97,"Free Memory:");
  p.drawText(60,137,"Used Memory:");
  p.drawText(60,197,"Capacity:");
  p.drawText(60,240,"FS-Info:");
  p.drawText(18,265,"FS-Type:");
  p.drawText(18,290,"FS-ID:");
  p.drawText(18,315,"Mounted on:");

  p.drawText(315,265,"Free:");
  temp.sprintf("%i%%",availPerc);
  p.drawText(350,265,temp);
  p.drawText(315,290,"Used:");
  temp.sprintf("%i%%",usedPerc);
  p.drawText(350,290,temp);

  temp.sprintf("%.0lf", (double)mystat.f_bavail*(double)mystat.f_bsize*1.0);
  p.drawText(175,98,temp);

  temp.sprintf("%.0lf",((double)mystat.f_blocks - (double)mystat.f_bavail)*(double)mystat.f_bsize*1.0);  
  p.drawText(175,139,temp);

  temp.sprintf("%.0lf",(double)mystat.f_blocks*(double)mystat.f_bsize*1.0);
  p.drawText(175,198,temp);

  temp.sprintf("%.1f",(double)mystat.f_bavail*(double)mystat.f_bsize/(1024*1024.0)  );
  p.drawText(310,98,temp);

  temp.sprintf("%.1f",(  (double)mystat.f_blocks - (double)mystat.f_bavail)*(double)mystat.f_bsize / (1024*1024.0));  
  p.drawText(310,139,temp);

  temp.sprintf("%.1f",((double)mystat.f_blocks*(double)mystat.f_bsize)/(1024*1024.0));
  p.drawText(310,198,temp);

  p.drawText(258,98,"Bytes");
  p.drawText(258,139,"Bytes");
  p.drawText(258,198,"Bytes");
  p.drawText(350,98,"MB");
  p.drawText(350,139,"MB");
  p.drawText(350,198,"MB");
  
  p.end();

}


void KFSStatus::getFsStat(char *filesystem)
{
  QString Temp;
  
  // Let's get the filesystem stats
  statfs(filesystem,&mystat);
  
  // Calculate angels and percents
  availPerc=(mystat.f_bavail)/(mystat.f_blocks/100);
  availAngle=(int)57.6*availPerc;
  usedPerc=100-availPerc;
  usedAngle=5760-availAngle;
  
  Temp.sprintf("%i",mystat.f_fsid.__val);
  fsid->setText(Temp);

}

int KFSStatus::setFsInfos()
{
  QString Temp;
  FILE *fp;
  const char *dev;
  
  dev = device->text(device->currentItem());  // device to check
  
  if(!(fp = setmntent("/etc/mtab","r")))
    {printf("Error opening /etc/mtab\n");
    QMessageBox::message("KFSStatus Error",
			 "I have trouble opening /etc/mtab",0);
    return -1;
    }  
  
  if(!(mymntent = getmntent(fp)))
    {printf("Error reading /etc/mtab\n");
    QMessageBox::message("KFSStatus Error",
			 "No filesystem seems to be mounted!",0);    
    return -1;
    }
  
  
  while((strcmp(mymntent->mnt_fsname,dev)))
    { mymntent = getmntent(fp);
    if(!mymntent)
      {printf("No such filesystem in /etc/mtab\n");
      QMessageBox::message("Error!","No such filesystem mounted",0);   
      return -1;
      }
   }
  
  Temp.sprintf("%s",mymntent->mnt_type);
  fstype->setText(Temp);
  
  Temp.sprintf("%s (%s)",mymntent->mnt_dir,mymntent->mnt_opts);
  fsmounted->setText(Temp);
 
  return 1;
}

int KFSStatus::getFileSystems()
{
  QString Temp;
  FILE *fp;
  
  if(!(fp = setmntent("/etc/mtab","r")))
    {printf("Error opening /etc/mtab\n");
    QMessageBox::message("KFSStatus Error",
			 "I have trouble opening /etc/mtab",0);
    return -1;
    }  
  theList->clear();
  
  while((mymntent = getmntent(fp)) != 0)
    {
      if(strcmp(mymntent->mnt_fsname,"none"))
	{device->insertItem(mymntent->mnt_fsname);
	theList->append(new Device(mymntent->mnt_fsname));
	}
    }
  if(!mymntent)
   printf("Found all filesystems...(%i)\n",theList->count());
 
  return 1; 
}

/****************SLOTS*****************/

int KFSStatus::deviceChange(int i)
{

 if(!strcmp(device->text(i),""))
    {QMessageBox::message("Error!","You have to specify a filesystem",0);
    return -1;}

 if(setFsInfos() < 0 )
   { printf("setFsInfos returned -1\n");
   return -1;}

 getFsStat(mymntent->mnt_dir);
 theBox->repaint();
 this->repaint();
 return 1;
}

void KFSStatus::deviceChangeI()
{
  int x = 0;
  QString theString;
  int currItemInt;
  QString currItemString;

  Device *theTempDev = new Device(""); 

  currItemInt = device->currentItem();	
  currItemString = device->text(currItemInt);

  if(deviceChange(device->currentItem()) < 0)
     {printf("Device change returned neg val\n");
     return;
     }

  getFileSystems();
  
  device->clear();

  theList->setAutoDelete(FALSE);

  x=0;

  theTempDev = theList->first();

  while(theTempDev != NULL)
      {theString = theTempDev->text();
      device->insertItem(theString);
      theTempDev = theList->next();
     }

  for(x=0;x <= device->count()-1;x++)
     {if(!strcmp(device->text(x),currItemString))
       device->setCurrentItem(x);
     else
       if(x >=device->count()-1)
	 device->setCurrentItem(device->count());
     }
}


void KFSStatus::exception()
{
  qApp->quit();
}


/************************MAIN**************/


#include "kfsstatus.moc"
main(int argc, char **argv)
{
  KApplication *fs_status = new KApplication(argc, argv,"kfsstatus");
  fs_status->setFont(QFont("Helvetica",12),true);
   
  KFSMainWidget *w = new KFSMainWidget(0,"main");
  fs_status->setMainWidget(w);
  w->setCaption("K File System Control");
  w->show();
  return fs_status->exec();
}












