/***********************************************************************
**
**   lineelement.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include "lineelement.h"

#include <mapcalc.h>

#include <kapp.h>
#include <kiconloader.h>
#include <qdatastream.h>
#include <qfile.h>
#include <qtextstream.h>

void LineElement::drawMapElement(QPainter* targetPainter,
        double dX, double dY, long mapCenterLon)
{
  extern double _scale[];
  extern int _scaleBorder[];
  extern double _currentScale;
  double lat, lon, projX, projY, drawX, drawY;

  bool show = true;
  bool highwayShow = false;
  bool isClosed = false;

  QPointArray pointArray(locLength);

  switch(typeID) {
    case Highway:
      show = false;
      if(_currentScale < _scale[3]) {
        targetPainter->setPen(QPen(QColor(255,75,75), 5));
        highwayShow = true;
      } else if(_currentScale < _scale[4]){
        targetPainter->setPen(QPen(QColor(255,75,75), 3));
      } else if(_currentScale < _scale[5]){
        targetPainter->setPen(QPen(QColor(255,75,75), 2));
      } else {
        targetPainter->setPen(QPen(QColor(255,75,75), 1));
      }
      break;
    case MidRoad:
      show = false;
      targetPainter->setPen(QPen(QColor(200,75,75), 1));
      if(_currentScale <= _scale[_scaleBorder[14]]) {
        if(_currentScale <= _scale[3]) {
          show = true;
          targetPainter->setPen(QPen(QColor(200,75,75), 2));
        } else if(_currentScale <= _scale[4]) {
          show = true;
          targetPainter->setPen(QPen(QColor(200,75,75), 1));
        }
      }
      targetPainter->setBrush(NoBrush);
      break;
    case SmallRoad:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[15]]) {
        show = true;
        targetPainter->setPen(QPen(QColor(200,50,50), 1));
      }
      targetPainter->setBrush(NoBrush);
      break;
    case Railway:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[5]]) {
        show = true;
        targetPainter->setBrush(NoBrush);
        targetPainter->setPen(QPen(QColor(0,0,0), 2, SolidLine));
      }
      break;
    case AerialRailway:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[6]]) {
        show = true;
        targetPainter->setBrush(NoBrush);
        targetPainter->setPen(QPen(QColor(0,0,0), 2, DashDotLine));
      }
      break;
    case Coast:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[0]]) {
        show = true;
        targetPainter->setBrush(QBrush(QColor(50,200,255),SolidPattern));
        targetPainter->setPen(QPen(QColor(50,50,255), 1));
      }
      break;
    case BigLake:
      isClosed = true;
    case BigRiver:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[1]]) {
        show = true;
        targetPainter->setPen(QPen(QColor(50,50,255), 1));
        targetPainter->setBrush(QBrush(QColor(50,200,255),SolidPattern));
      }
      break;
    case MidLake:
      isClosed = true;
    case MidRiver:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[2]]) {
        show = true;
        targetPainter->setPen(QPen(QColor(50,50,255), 1));
        targetPainter->setBrush(QBrush(QColor(50,200,255),SolidPattern));
      }
      break;
    case SmallLake:
      isClosed = true;
    case SmallRiver:
      show = false;
      if(_currentScale <= _scale[_scaleBorder[26]]) {
        show = true;
        targetPainter->setPen(QPen(QColor(50,50,255), 1));
        targetPainter->setBrush(QBrush(QColor(50,200,255),SolidPattern));
      }
      break;
  }
  if(show) {
    for(unsigned int i = 0; i < locLength; i++) {
      lat = numToRad(latitude[i]);
      lon = numToRad(longitude[i] - mapCenterLon);
      projX = calc_X_Lambert(lat, lon);
      projY = calc_Y_Lambert(lat, lon);
      drawX = (projX * RADIUS / _currentScale) + dX;
      drawY = (projY * RADIUS / _currentScale) + dY;

      pointArray.setPoint(i, drawX, drawY);
    }
    if(isClosed) {
      targetPainter->drawPolygon(pointArray);
    } else {
      targetPainter->drawPolyline(pointArray);
      if(highwayShow) {
        targetPainter->setPen(QPen(QColor(255,255,255), 1));
        targetPainter->drawPolyline(pointArray);
      }
    }
  }
}

bool LineElement::writeElementBin(const char* fileName)
{
  QFile outputFile(fileName);
  if(!outputFile.open(IO_WriteOnly | IO_Append)) {
    return false;
  }
  QDataStream out(&outputFile);

  out << (Q_UINT8) typeID;
  // Bei welchen Elementen mu der Name abgespeichert werden???
  if((typeID == BaseMapElement::HugeCity) ||
     (typeID == BaseMapElement::BigCity) ||
     (typeID == BaseMapElement::MidCity) ||
     (typeID == BaseMapElement::SmallCity) ||
     (typeID == BaseMapElement::BigLake) ||
     (typeID == BaseMapElement::MidLake) ||
     (typeID == BaseMapElement::SmallLake)) {
    out << (const char*) name;
  }
  out << (Q_UINT16) locLength;
  for(unsigned int i = 0; i < locLength; i++) {
    out << (Q_INT32) latitude[i];
    out << (Q_INT32) longitude[i];
  }

  return true;
}

bool LineElement::writeElementAscii(const char* fileName)
{
  QFile mapFile(fileName);
  if(!mapFile.open(IO_ReadWrite)) {
    return false;
  }
  mapFile.at(mapFile.size());
  QTextStream mapOut(&mapFile);

  /*************************************************************************
   *                                                                       *
   *  NOTE:                                                                *
   *  In the ascii-file, the name is written even if it is not used in the *
   *  map. This is, because with the name it is pretty much easier to      *
   *  find something in the ascii-file.                                    *
   *                                                                       *
   *************************************************************************/

  mapOut << "[NEW]\n"
         << "TYPE=" << typeID << endl
         << "NAME=" << name << endl;
  for(unsigned int i = 0; i < locLength; i++) {
    mapOut << latitude[i] << " " << (Q_INT32) longitude[i] << endl;
  }

  mapFile.close();

  return true;
}

void LineElement::setLocList(long* latList, long* lonList)
{
  latitude = latList;
  longitude = lonList;
}

LineElement::LineElement(QString n, unsigned int t,
      unsigned int length, long* latList, long* lonList)
{
  name = n;
  typeID = t;
  latitude = new long[length];
  longitude = new long[length];
  locLength = length;
  latitude = latList;
  longitude = lonList;
}

LineElement::LineElement(QString n, unsigned int t)
{
  name = n;
  typeID = t;
}

LineElement::~LineElement()
{
  delete[] latitude;
  delete[] longitude;
}
