/***********************************************************************
**
**   airport.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include "airport.h"

#include <mapcalc.h>

#include <kapp.h>
#include <kiconloader.h>
#include <qdatastream.h>
#include <qfile.h>
#include <qtextstream.h>

bool Airport::writeElementBin(const char* fileName)
{
  QFile outputFile(fileName);
  if(!outputFile.open(IO_WriteOnly | IO_Append)) {
    return false;
  }
  QDataStream out(&outputFile);

  out << (Q_UINT8) typeID;
  out << (const char*) name;
  out << (const char*) alias;
  out << (Q_INT32) latitude;
  out << (Q_INT32) longitude;
  out << (Q_UINT8) isWaypoint;
  if(typeID != BaseMapElement::ClosedAirfield) {
    out << (Q_UINT16) elevation;
    out << (const char*) frequency;
    out << (Q_UINT8) vdf;
    if((typeID != BaseMapElement::CivHeliport) &&
       (typeID != BaseMapElement::MilHeliport) &&
       (typeID != BaseMapElement::AmbHeliport)) {
      out << (Q_UINT8) rwNum;
      for(unsigned int i = 0; i < rwNum; i++) {
        out << (Q_UINT8) rwData[i].direction;
        out << (Q_UINT16) rwData[i].length;
        out << (Q_UINT8) rwData[i].surface;
      }
    }
  }
  outputFile.close();

  return true;
}

bool Airport::writeElementAscii(const char* fileName)
{
  QFile mapFile(fileName);
  if(!mapFile.open(IO_ReadWrite)) {
    return false;
  }
  mapFile.at(mapFile.size());
  QTextStream mapOut(&mapFile);

  mapOut << "[NEW]\n"
         << "TYPE=" << typeID << endl
         << "NAME=" << name << endl
         << "ALIAS=" << alias << endl
         << "AT=" << latitude << " " << longitude << endl
         << "ELEVATION=" << elevation << endl
         << "FREQUENCY=" << frequency << endl
         << "WAYPOINT=" << (int) isWaypoint << endl;
  for(unsigned int i = 0; i < rwNum; i++) {
    mapOut << "RUNWAY=" << rwData[i].direction << " "
           << rwData[i].length << " "
           << rwData[i].surface << endl;
  }
  mapOut << "[END]\n";
  mapFile.close();

  return true;
}

void Airport::drawMapElement(QPainter* targetPainter,
          double dX, double dY, long mapCenterLon)
{
  extern double _currentScale;
  double lat, lon, projX, projY, drawX, drawY;

  lat = numToRad(latitude);
  lon = numToRad(longitude - mapCenterLon);

  projX = calc_X_Lambert(lat,lon);
  projY = calc_Y_Lambert(lat,lon);

  drawX = projX * RADIUS / _currentScale + dX;
  drawY = projY * RADIUS / _currentScale + dY;

  targetPainter->setPen(QPen(QColor(0,0,0), 2));

  QString tempText;
  switch (typeID) {
    case BaseMapElement::Airport:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/airport.xpm"));
      break;
    case MilAirport:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/milairport.xpm"));
      break;
    case CivMilAirport:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/civmilair.xpm"));
      break;
    case Airfield:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/airfield_c.xpm"));
      break;
    case ClosedAirfield:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/closed.xpm"));
      break;
    case CivHeliport:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/civheliport.xpm"));
      break;
    case MilHeliport:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/milheliport.xpm"));
      break;
    case AmbHeliport:
      targetPainter->drawPixmap(drawX - 16, drawY - 16,
        Icon(KApplication::kde_datadir() + "/kflog/map/ambheliport.xpm"));
      break;
  }
}

Airport::Airport(QString n, QString a, unsigned int t, long latPos,
            long lonPos, unsigned int elev, QString f, bool v = false,
            struct runway* rwDat = 0, unsigned int rwN = 0)
    : SinglePoint(n, t, latPos, lonPos)
{
  alias = a;
  elevation = elev;
  frequency = f;
  vdf = v;
  rwData = rwDat;
  rwNum = rwN;
}

Airport::~Airport()
{

}
