/***************************************************************************
                          mapobject.h  -  class for the objects in the map
                             -------------------
    begin                : Sun Jun 27 1999
    copyright            : (C) 1999 by Heiner Lamprecht
    email                : heiner@kijumfo.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MAPOBJECT_H
#define MAPOBJECT_H

#include <qlist.h>
#include <qobject.h>
#include <resource.h>

/**
  * Location is used to represent the Cartesian-coordinates.
  * The resolution used here is 1/10,000th of a minute
  * (approx 0.1852m on a great circle). We decided to use minutes
  * instead of seconds as base unit, because our GPS-Logger used
  * them likewise ;-)
  * Therefore: 1 second :=      ~167
  *            1 minute :=    10,000
  *            1 degree :=   600,000
  *     Earth's circumference (1 degree * 360) is 216,000,000, so we
  * can use 32 bit integer vars to represent these values!
  * Positive values mean North or East and negative values South
  * or West
  */
class Location
{
 public:
  /** creates an empty Location-object. */
  Location();
  /** creates a new Location-object with the given coordinates. */
  Location(long latitude, long longitude);
  /** destroys the object */
  ~Location();
  /** sets the latitude */
  void setLat(long latitude);
  /** sets the lonitude */
  void setLon(long longitude);
  /** returns the latitude */
  long getLat();
  /** returns the lonitude */
  long getLon();

 private:
  /** the latitude ;-) */
  long latitude;
  /** the londitude ;-) */
  long longitude;
};

/**
  * This class provides functions for creating and accessing mapobjects.
  * @author Heiner Lamprecht
  * @version 0.0
  */
class MapObject : public QObject  {
 public:
  /** Creates a new mapobject. */
  MapObject(const char* t = 0, const QString n = 0, int coordNum = 0);
  /** Destroys the object. */
  ~MapObject();
  void setLocLength(int coordNum);
  /** Sets the alias-name for the object. */
  void setAlias(QString new_alias);
  /** Sets the "AT"-Value for the object. It will set the values
    * of the AT-Location. */
  void setAt(long at_lat, long at_log);
  /** Sets the elevation of the object. */
  void setElev(QString elev = 0, int elevType = 0);
  /** Sets the frequency of the object. */
  void setFrequency(QString freq_string, bool vdf = false);
  /** Sets the list of coordinates. The number of coordinates must
    * be given to the constructor. */
  void setLocList(long* lat, long* lon);
  /**
    * Sets the information about the runways. The param "num" declares
    * the number of runways, the param "dir" is an int-array containing
    * the direction of each runway, "length" (int-array) contains the
    * length of each runway and "type" (QString-array) contains the
    * type of each runway.
    */
  void setRunway(int num = 0, int dir = 0, int length = 0, QString type = 0);
  /** */
  const long getAtLat();
  const long getAtLon();
  /** Returns a pointer to the list of the latitudes. */
  const long* getLatList();
  /** Returns a pointer to the list of the longitudes. */
  const long* getLonList();
  /** Returns the number of points. */
  const int getLocLength();
  /** Returns the name of the object */
  const QString getName();
  /** Returns the type of the object */
  const QString getType();

 private:
  /** The alias-name of the object. */
  QString alias;
  /** The At-Location. It is used for punktural map-objects. */
  Location at;
  /** The elevation of the object. */
  QString elevation;
  /** The frequency for airfields and -ports and nav-objects */
  QString frequency;
  /**
    * The coordinates for the object, given as an array in the
    * internal format. The size of the array is given during
    * construction of the object.
    *
    * This one is for the latitude.
    */
  long* latitude;
  /** And this for the longitude. */
  long* longitude;
  /** The name of the object. */
  QString name;
  /** The type of the object. */
  QString type;
  /** The number of points. */
  int locLength;
};

/**
  * This class provides a list of all mapobjects
  * @author Heiner Lamprecht
  * @version 0.0
  */
class MapObjectList {
 public:
  /** Creates a new mapobjectlist. */
  MapObjectList();
  /** Creates a new mapobjectlist and read the input data from the
    * given file. If the file doesn't exist, the list will be empty. */
  MapObjectList(QString fileName);
  /** Destroys the list and all items in it. */
  ~MapObjectList();
  /** Adds a new object to the list. */
  void addObject(MapObject* newObject);
  /** Returns the current length of the list. */
  unsigned int getListLength();
  /** Returns a pointer to the given object in the list. */
  MapObject* getObject(unsigned int index);
  /** Creates a MapObjectList from the given file. The list will
    * be destroyed before, so all current items will be lost.
    * If no filename is given, the function takes the default
    * $kde_dir/kflog/map.wld.
    * Returns the length of the list. */
  unsigned int createList(QString* fileName = 0);

 private:
  /** The list of mapobject. */
  QList<MapObject> objectList;
  /** The name of the file used to fill the list. */
  QString mapFile;
};

#endif



