/*
	postprocess.cpp - part of KDiskCat the KDE Disk Catalog.

	Copyright (c) 1998,1999 Balzs Ternyi <terenyi@freemail.c3.hu>

	KDiskCat is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
			
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
					
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <kmsgbox.h>
#include <kapp.h>
#include <qregexp.h>
#include <iostream.h>
#include <unistd.h>
#include <qdir.h>
#include <time.h>
#include "postprocess.h"
#include "util.h"

#define PLUGIN_TIMEOUT 20 //Max seconds to wait for plugin response
#define LOOP_DELAY 10000000 // How long to sleep in a response cicle (ns)

extern const char list_separator;

const char enum_separator = ',';

PluginClient::PluginClient(QString exec)
{
	start_command=exec;
	kdcp=new KProcess();
	connect(kdcp,SIGNAL(processExited(KProcess *)),this,SLOT(slotProcessExited(KProcess *)));
	connect(kdcp,SIGNAL(receivedStdout(KProcess*,char*,int)),this,SLOT(slotRecStdout(KProcess*,char*,int)));
	connect(kdcp,SIGNAL(wroteStdin(KProcess*)),this,SLOT(slotStdinWrote(KProcess*)));

	running=false;
}
	
PluginClient::~PluginClient()
{
	delete kdcp;
}

QString PluginClient::getDescription(QFileInfo* finfo)
{
	QString ret="";
	int remaining_time;
	timespec delay;

	QString str=finfo->absFilePath();
	str+="\n";
	
	const char* cstr=str;
	kdcp->writeStdin(const_cast<char*>(cstr),str.length());
 	
	delay.tv_sec=0;
	delay.tv_nsec=LOOP_DELAY;
  	remaining_time=PLUGIN_TIMEOUT*(1000000000 / LOOP_DELAY);
  	
  	while (message.right(6) != "|END|\n" && remaining_time > 0 && isRunning())
  	{
  		kapp->processEvents(1);
  		if (message.right(6) != "|END|\n")
  		{
	  		nanosleep(&delay,0);
  			remaining_time--;
  		}
  	}
  	if (message.right(6) == "|END|\n")
  	{
  		ret=message.left(message.length() - 6);
  	}
  	else // There is no response
  	{
  		cerr << "PluginClient::NO RESPONSE!" << endl;
  		stop(); // pluginDied() emited
  	}
	
  	message="";
  	
	return ret;
}

int PluginClient::start()
{
	int ret=1;
		
	prepareCommandLine(start_command);
	
	message="";
	if (!kdcp->start(KProcess::NotifyOnExit,KProcess::All))
	{
		ret= -1;
	}

	running=kdcp->isRunning();
		
	return ret;
}

int PluginClient::stop()
{
	int ret=1;
	
	if (!kdcp->kill())
	{
		ret= -1;
	}
	running=false;
	
	return ret;
}

QString PluginClient::info()
{
	QString str="";
	int remaining_time;
	QString tmp;
		
	prepareCommandLine(start_command + " --help");
	
	message="";
	if (kdcp->start(KProcess::NotifyOnExit,KProcess::All))
	{
		running=true;
		remaining_time=PLUGIN_TIMEOUT;
		while (isRunning() && remaining_time > 0)
		{
			kapp->processEvents(1000);
			if (isRunning())
			{
				sleep(1);
				remaining_time--;
			}
		}
	}

	if (!isRunning())
	{
		str=message;
	}
		
	stop();
	message="";
	
	return str;
}

QString PluginClient::defaults()
{
	QString str="";
	int remaining_time;
	QString tmp;
	
	prepareCommandLine(start_command + " --default");
	
	message="";
	if (kdcp->start(KProcess::NotifyOnExit,KProcess::All))
	{
		running=true;
		remaining_time=PLUGIN_TIMEOUT;
		while (isRunning() && remaining_time > 0)
		{
			kapp->processEvents(1000);
			if (isRunning())
			{
				sleep(1);
				remaining_time--;
			}
		}
	}

	if (!isRunning())
	{
		str=message;
	}
		
	stop();
	message="";
	
	return str;
}

bool PluginClient::isRunning()
{
	return running;
}

void PluginClient::slotProcessExited(KProcess* proc)
{
	running=false;
	if (proc->normalExit())
	{
		if (proc->exitStatus()==0)
		{
		}
		else
		{
			cerr << "PluginClient::PLUGIN DIED!" << endl;
			emit pluginDied();
		}
	}
}

void PluginClient::slotRecStdout(KProcess *proc,char *buffer,int buflen)
{
	QString str(buffer,buflen+1);
	
//	cerr << "RSTDOUT:" << str << endl;	
	
	message+=str;
}

void PluginClient::slotStdinWrote(KProcess *proc)
{
   is_written=true;
}

int PluginClient::prepareCommandLine(QString tmp)
{
	int i;
	
	while (tmp.length() > 0)
	{
		i=tmp.find(" ");
		if (i>0)
		{
			*kdcp << tmp.mid(0,i);
			tmp.remove(0,tmp.find(" ")+1);
		}
		else
		{
			*kdcp << tmp;
			tmp="";
		}
	}
}

// -----------------------------------------------

PostProcess::PostProcess()
{
	plugins.setAutoDelete(true);
	readConfig();
}

PostProcess::~PostProcess()
{
}

QString PostProcess::getDescription(QFileInfo* finfo,QDir* fdir)
{
	QString ret;
	bool found=false;
	uint i=0;
	
	while (i< plugins.count() && !found)
	{
		if (!plugins.at(i)->isDead())
		{
			found=plugins.at(i)->fetchDescription(finfo,fdir,&ret);
		}
		i++;
	}
	
	if (!found)
	{
		ret="";
	}
	
	return ret;
}

void PostProcess::readConfig()
{
	KConfig* config;
	uint i,to;
	QString tmp,str;
	PluginData* pd;
	
   config=kapp->getConfig();

   config->setGroup("ADE PLUGINS");
   to=config->readNumEntry("NumberOfPlugins",0);
   for (i=0;i<to;i++)
   {
   	str.setNum(i);
      tmp=config->readEntry("Plugin"+str);
		if (Util::cropFirstWord(&tmp,list_separator) == "ON")
		{
			pd=new PluginData(Util::cropFirstWord(&tmp,list_separator),
									Util::cropFirstWord(&tmp,list_separator),
									Util::cropFirstWord(&tmp,list_separator));
   	   plugins.append(pd);
   	}
   }
}

//-----------------------------------------

PluginData::PluginData(QString pcommand,QString passoc,QString pfire)
{
	dead=false;
	started=false;
	
	command=pcommand;
	while (passoc.length()>0)
	{
		associations.append(Util::cropFirstWord(&passoc,enum_separator));
	}
	while (pfire.length()>0)
	{
		fire.append(Util::cropFirstWord(&pfire,enum_separator));
	}
}

PluginData::~PluginData()
{
	if (started)
	{
		client->stop();
		delete client;
	}
}

bool PluginData::isDead()
{
	return dead;
}

void PluginData::slotPluginDied()
{
	dead=true;
	KApplication::restoreOverrideCursor();
	KMsgBox::message(0,i18n("Message"),i18n("The ADE plugin died!\nThe files associated with this plugin are excluded from the ADE process!"));
	KApplication::setOverrideCursor(waitCursor);
}

bool PluginData::fetchDescription(QFileInfo* finfo,QDir* fdir,QString* desc)
{
	uint i,j;
	QRegExp rexp;
	QString fname;
	bool found=false;
   const QFileInfoList *files=fdir->entryInfoList();
   QFileInfoListIterator files_it(*files);
   QFileInfo *ftmp;
	
	rexp.setWildcard(true);
	fname=finfo->fileName();
	
	// Checking associations
	i=0;
	while (i<associations.count() && !found)
	{
		rexp=(const char*) associations.at(i);
		if (rexp.match(fname) != -1)
		{
			if (fire.count()>0)
			{
				// Checking fire
				j=0;
				while (j<fire.count() && !found)
				{
   				rexp=(const char*) fire.at(j);
  				   while (((ftmp=files_it.current()) != 0) && !found)
               {
   					if (rexp.match(ftmp->fileName()) != -1)
   					{
   						found=true;
   					}
				      ++files_it;
   				}
   				j++;
				}
			}
			else
			{
				found=true;
   		}
 			if (found)
 			{
   			if (!started)
   			{
   				client=new PluginClient(command);
   				connect(client,SIGNAL(pluginDied()),this,SLOT(slotPluginDied()));
   				started=true;
   				client->start();
   			}
   			if (client->isRunning())
   			{
   				*desc=client->getDescription(finfo);
   			}
   			else
   			{
   				slotPluginDied();
   			}
			}
		}
		i++;
	}
	
	return found;
}