/*
	cattree.cpp - part of KDiskCat the KDE Disk Catalog.

	Copyright (c) 1998,1999 Balzs Ternyi <terenyi@freemail.c3.hu>

	KDiskCat is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
			
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
					
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <kapp.h>
#include <kmsgbox.h>
#include <qregexp.h>
#include <strstream.h>
#include <kfm.h>
#include <qfileinfo.h>
#include <kpopmenu.h>
#include <kfiledialog.h>
#include <qprogressdialog.h>

#include "cattree.h"
#include "searchprogress.h"
#include "PropertiesDlg.h"
#include "renamedialog.h"
#include "newcatalog.h"
#include "newcategorydlg.h"
#include "scanprogressdlg.h"

extern void micro(QString* the_text);
extern void setTitle(const char * the_text);
extern CategoryHandler* cathandler;
extern CatHandler* cat;
extern QString kdiskcat_data_dir;
extern const char list_separator;
extern Prefs* prefs;

CatView::CatView(QWidget *parent=0,const char *name=0,Orientation orient=Vertical,Units units=Percent,int pos=50)
    :KNewPanner(parent,name,orient,units,pos)
{
   cattree= new eKTreeList(this,"cattree");
   catlist= new KTabListBox(this,"catlist",4,0);
   icon_loader= KApplication::getKApplication()->getIconLoader();

//   visible_branch.setAutoDelete(true);
   
   connect(cattree,SIGNAL(singleSelected(int)), SLOT(slotTreeSelection(int)));
   connect(catlist,SIGNAL(selected(int,int)), SLOT(slotListClicked(int,int)));
   connect(catlist,SIGNAL(highlighted(int,int)), SLOT(slotListSelection(int,int)));
   connect(cattree,SIGNAL(rightClicked(int)),SLOT(slotTreePopup(int)));
	connect(catlist,SIGNAL(popupMenu(int,int)),SLOT(slotListPopup(int)));
}

CatView::~CatView()
{
	KConfig* conf;
	
	conf=kapp->getConfig();
	conf->setGroup(kapp->appName());
	
	conf->writeEntry("Column0",catlist->columnWidth(0));
	conf->writeEntry("Column1",catlist->columnWidth(1));
	conf->writeEntry("Column2",catlist->columnWidth(2));
	conf->writeEntry("Column3",catlist->columnWidth(3));
}

void CatView::init()
{
	QStrList strlist;
	KConfig* conf;
	
	conf=kapp->getConfig();
	conf->setGroup(kapp->appName());
	
   folder_xpm=icon_loader->loadMiniIcon("folder.xpm");
   catalog_xpm=icon_loader->loadMiniIcon("mini_cd.xpm");

   cattree->setFocusPolicy(QWidget::NoFocus);
   catlist->setFocusPolicy(QWidget::StrongFocus);

   activate(cattree,catlist);

   catlist->setColumn(0,i18n("Filename"),conf->readNumEntry("Column0",100),KTabListBox::TextColumn,KTabListBox::NoOrder,KTabListBox::Ascending,0);
   catlist->setColumn(1,i18n("Size"),conf->readNumEntry("Column1",70),KTabListBox::TextColumn,KTabListBox::NoOrder,KTabListBox::Ascending,0);
   catlist->setColumn(2,i18n("Date"),conf->readNumEntry("Column2",106),KTabListBox::TextColumn,KTabListBox::NoOrder,KTabListBox::Ascending,0);
   catlist->setColumn(3,i18n("Description"),conf->readNumEntry("Column3",100),KTabListBox::TextColumn,KTabListBox::NoOrder,KTabListBox::Ascending,0);
   
   catlist->setSeparator(list_separator);
   cathandler->init(kdiskcat_data_dir);
   strlist=cathandler->getCategoryNames();
   emit categoriesChanged(&strlist);
}

void CatView::refreshTree()
{
   KPath kp;
   QString *str;

   KApplication::setOverrideCursor(waitCursor);

   visible_branch.clear();
   setTitle("");
   
   cattree->setAutoUpdate(false);
   
   cattree->clear();

   cat->setCatalogLoading(false);

   if (cat->firstBranch() != -1)
	{
	   kp=cat->nextBranch();
	
	   while (kp.count() != 0)
	   {
	      if (kp.count()==1)
	      {
	         str=kp.pop();
	         cattree->insertItem((const char *) *str,&catalog_xpm,-1);
	      }
	      else
	      {
	         str=kp.pop();
	         cattree->addChildItem((const char *) *str,&folder_xpm,&kp);
	      }
	      kp=cat->nextBranch();
	   }
	}
	
   cat->setCatalogLoading(true);

   cattree->setAutoUpdate(true);
   cattree->repaint();

   kp.clear();
   refreshList(kp);

   KApplication::restoreOverrideCursor();
}

void CatView::refreshList(KPath selected_item)
{
   QString str, tmp, val;
   KPath kp;
   CatEntry* ce;
   QDate d;
   QTime t;

   catlist->setAutoUpdate(false);
	catlist->clear();

   if (selected_item.count() != 0)
   {
      visible_branch=selected_item;

      setTitle(cat->Path2Char(selected_item));

      if (selected_item.count()>1)
      {
         str="..";
         str+=list_separator;
         str+=" ";
         str+=list_separator;
         str+=" ";
         str+=list_separator;
         str+=i18n("UPPER DIRECTORY");
         
         catlist->insertItem((const char *) str);
         catlist->changeItemColor(blue);
      }

      cat->setCurrentBranch(selected_item);

      ce=cat->first();
      while (ce != 0)
      {
            str=(const char *) ce->filename;
            str+=list_separator;
            tmp=sizeToString(ce->f_size);
            str+=tmp;
            str+=list_separator;
            d=ce->mod_date.date();
            t=ce->mod_date.time();
            // QT v. 2 supports QString::arg, that's easier than the following:
            tmp=i18n("yyyy.mm.dd hh:MM");
				val.setNum(d.year())  ;tmp.replace("yyyy",val);
				val.setNum(d.month()) ;tmp.replace("mm"  ,("0"+val).right(2));
				val.setNum(d.day())   ;tmp.replace("dd"  ,("0"+val).right(2));
				val.setNum(t.hour())  ;tmp.replace("hh"  ,("0"+val).right(2));
				val.setNum(t.minute());tmp.replace("MM"  ,("0"+val).right(2));

            str+=tmp;
            str+=list_separator;
            if (ce->description.isEmpty())
            {
               str+=" ";
            }
            else
            {
               str+=ce->description;
            }
            catlist->insertItem((const char *) str);
            switch (ce->entry_type)
            {
               case CatEntry::E_DIR:
                  catlist->changeItemColor(blue);
                  break;
               case CatEntry::E_FILE+CatEntry::E_LINK:
                  catlist->changeItemColor(red);
                  break;
               case CatEntry::E_DIR+CatEntry::E_LINK:
                  catlist->changeItemColor(darkCyan);
                  break;
            }
         ce=cat->next();
      }
   }

   // It doesn't want to scroll up after dir change
	//catlist->setTopItem(0);

   catlist->setAutoUpdate(true);
   catlist->repaint();
}

QString CatView::sizeToString(uint size)
{
   QString ret;
   
   if (size>9999999)
   {
      ret.sprintf("%d Kb",size/1024);
   }
   else
   {
      ret.sprintf("%d",size);
   }
   return ret;
}

void CatView::gotoSelected(QString category,KPath branch,QString filename)
{
	int i,count;
	bool found=false;
	QString str;
	
	slotCategoryChoice(cathandler->getCategoryNames().find(category));
	refreshList(branch);
	count=catlist->count();
	for (i=0;i<count && !found;i++)
	{
		str=catlist->text(i,0);
		if (str==filename)
		{
			catlist->setCurrentItem(i);
			catlist->markItem(i);
			found=true;
		}
	}
}

//SLOTS

void CatView::slotSave()
{
   cat->synchronizeAll();
}

void CatView::slotExportCategory()
{
	QString filename,tmp,limiter;
	QString str=cathandler->getCurrentCategory();
   int current_item=0;
   CatEntry* ce;
   KPath current_branch;
   QDate d;
   QTime t;

	if (str != "")
	{
		filename=KFileDialog::getSaveFileName(str+".txt","*.txt",this,"export_fileselect");
		if (filename != 0)
		{
      	QFile f(filename);

	      if (f.open(IO_WriteOnly))
			{
   	      QTextStream s(&f);
   	
				QProgressDialog progress(i18n("Exporting category..."),i18n("Cancel"),cat->branchCount(),0,"progress",true);
				
				if (prefs->export_delimiter=="\\t")
				{
					limiter="\t";
				}
				else
				{
					limiter=prefs->export_delimiter;
				}

         	if (cat->firstBranch() == 1)
         	{
         		progress.setProgress(0);
         		
					current_branch=cat->nextBranch();
        			while (current_branch.count() > 0)
        			{
         			ce=cat->first();
         			while (ce != 0)
         			{
         				str="";
         				switch (ce->entry_type)
         				{
	         				case CatEntry::E_LINK+CatEntry::E_FILE:
	         					str="FL";
   	      					break;
   	      				case CatEntry::E_DIR:
   	      					str="D";
   	      					break;
   	      				case CatEntry::E_FILE:
   	      					str="F";
   	      					break;
   	      				case CatEntry::E_CATALOG:
   	      					str="C";
   	      					break;
   	      				case CatEntry::E_LINK+CatEntry::E_DIR:
   	      					str="DL";
   	      					break;
         				}
         				
         				str+=limiter;
         				str+=cat->Path2Char(current_branch);
         				str+="/";
         				str+=ce->filename;
         				str+=limiter;
         				tmp.sprintf("%d",ce->f_size);
         				str+=tmp;
         				str+=limiter;
         				d=ce->mod_date.date();
         				t=ce->mod_date.time();
         				tmp.sprintf("%04d-%02d-%02d %02d:%02d",d.year(),d.month(),d.day(),t.hour(),t.minute());
         				str+=tmp;
         				str+=limiter;
         				str+=ce->description.simplifyWhiteSpace();
         				str+="\n";
         				s << str;
         				ce=cat->next();
         			}
						
        				current_branch=cat->nextBranch();
        				current_item++;
						progress.setProgress(current_item);
						kapp->processEvents();
						
						if (progress.wasCancelled()) break;
        			}
            }
         		
				f.close();
         }
	      else
   	   {
   			KMsgBox::message(this,i18n("Message"),i18n("Unable to write to the specified file!"),KMsgBox::STOP);
	      }
	   }
   }
}

void CatView::slotTreeSelection(int index)
{
   KPath kp;

   kp=*cattree->itemPath(index);
   refreshList(kp);
}

void CatView::slotListClicked(int index,int)
{
   QString str;
   QString* new_str;
   KPath kp;
   KTreeListItem* tli;

   str=catlist->text(index,0);
   kp=visible_branch;
   if (str=="..")
   {
      kp.pop();
   }
   else
   {
      new_str=new QString(str);
      kp.push(new_str);
   }
   tli=cattree->itemAt(&kp);
   if (tli!=0) 
   {
      cattree->setCurrentItem(-1);
      refreshList(kp);
   }
}

void CatView::slotListSelection(int index,int)
{
   micro((QString *)&catlist->text(index,3));
   cattree->setCurrentItem(-1);
}

void CatView::slotCategoryChoice(int index)
{
   if (index >= 0)
   {
      KApplication::setOverrideCursor(waitCursor);

      cathandler->setActiveCategory(cathandler->getCategoryNames().at(index));
      refreshTree();

      KApplication::restoreOverrideCursor();
   }
}

void CatView::slotProperties()
{
	CatEntry* ce=0;
	CatEntry temp_ce;
	int current_tree_item_index;
	int current_list_item_index;
	KPath kp;
	QString str;
   PropertiesDlg* dia;
   KPath location;
	
	current_tree_item_index=cattree->currentItem();
   if (current_tree_item_index== -1)
   {
    	current_list_item_index=catlist->currentItem();
      if (current_list_item_index != -1)
      {
      	str=catlist->text(current_list_item_index,0);
         if (str != "..")
         {
            kp=visible_branch;
            kp.push(&str);
            if (cat->findEntry(kp) == 1)
            {
            	ce=cat->getCurrentEntry();
            	location=visible_branch;
            }
         }
      }
   }
   else
   {
   	kp=*cattree->itemPath(current_tree_item_index);
   	if (cat->findEntry(kp) == 1)
	   {
	      ce=cat->getCurrentEntry();
	      kp.pop();
	      location=kp;
	   }
   }

   if (ce!=0)
   {
   	temp_ce.filename=ce->filename;
   	temp_ce.entry_type=ce->entry_type;
   	temp_ce.mod_date=ce->mod_date;
   	temp_ce.f_size=ce->f_size;
   	temp_ce.description=ce->description;
   	temp_ce.mount_point=ce->mount_point;
      dia= new PropertiesDlg(this,"dia",&temp_ce,&location);
      if (dia->exec())
      {
			cat->setCurrentEntry(&temp_ce);
      }
      delete dia;
   }
}

void CatView::slotOpen()
{
	QString url="file:";
	QString str;
	QString tmp;
	KFM *kfm=new KFM();
	KPath location;
	int current_list_item_index;
	int current_tree_item_index;
	KPath kp;
	CatEntry* ce=0;

	current_tree_item_index=cattree->currentItem();
   if (current_tree_item_index == -1)
   {
    	current_list_item_index=catlist->currentItem();
      if (current_list_item_index != -1)
      {
      	str=catlist->text(current_list_item_index,0);
      	if (str != "..")
      	{
	         kp=visible_branch;
   	      kp.push(&str);
      	   if (cat->findEntry(kp) == 1)
         	{
	          	ce=cat->getCurrentEntry();
   	        	location=visible_branch;
      	   }
         }
      }
   }
   else
   {
   	kp=*cattree->itemPath(current_tree_item_index);
   	if (cat->findEntry(kp) == 1)
	   {
	      ce=cat->getCurrentEntry();
	      kp.pop();
	      location=kp;
	   }
   }

   if (ce!=0)
	{
     	kp=location;
    	if (location.count() == 0)
    	{
    		kp.push(&ce->filename);
    	}

   	str=ce->filename;
   	while (kp.count() > 1)
   	{
   		kp.pop();
   	}
   	cat->findEntry(kp);
   	ce=cat->getCurrentEntry();
   	tmp=ce->mount_point;
   	if (location.count() != 0)
   	{
	   	location.push(&str);
	   	if (tmp=="/") tmp="";
  			tmp+=cat->Path2Char(location,1);
  		}
	
		QFileInfo *fileInfo =
		new QFileInfo(tmp);

		url+=tmp;
		
		if (fileInfo->isDir())
		{
			kfm->openURL(url.data());
		}
		else
		{
			kfm->exec(url.data(),0L);
		}
   }

	delete kfm;
}

void CatView::refreshAll()
{
   refreshTree();
}

void CatView::slotTreePopup(int index)
{
	KPopupMenu* kpm;
	KPath kp;
	
	if (index == -1)
	{
		return;
	}
	kp=*cattree->itemPath(index);
		
	if (kp.count() == 1)
	{
		kpm=new KPopupMenu(i18n("Catalog"),0,"catalogpopup");
		kpm->insertItem(i18n("New catalog..."));
		kpm->insertItem(i18n("Move catalog..."));
		kpm->setItemEnabled(4,false);
		kpm->insertItem(i18n("Rename catalog..."));
		kpm->insertItem(i18n("Delete catalog"));
		kpm->insertSeparator();
		kpm->insertItem(i18n("Open"));
		kpm->insertItem(i18n("Properties..."));

		switch (kpm->exec(QCursor::pos()))
		{
			case 3:
				slotNew();
				break;
			case 5:
				slotRenameCatalog();
				break;
			case 6:
				slotDeleteCatalog();
				break;
			case 8:
				slotOpen();
				break;
			case 9:
				slotProperties();
				break;
		}
		delete kpm;
	}
	else
	{
		kpm=new KPopupMenu(i18n("Catalog entry"),0,"entpopup");
		kpm->insertItem(i18n("Open"));
		kpm->insertItem(i18n("Properties..."));

		switch (kpm->exec(QCursor::pos()))
		{
			case 3:
				slotOpen();
				break;
			case 4:
				slotProperties();
				break;
		}
		delete kpm;
	}
}

void CatView::slotListPopup(int index)
{
	KPopupMenu* kpm;
	QString str;
	
	catlist->setCurrentItem(index);
	slotListSelection(index,0);
	str=catlist->text(index,0);
	if (str != "..")
	{
		kpm=new KPopupMenu(i18n("Catalog entry"),0,"entpopup");
		kpm->insertItem(i18n("Open"));
		kpm->insertItem(i18n("Properties..."));

		switch (kpm->exec(QCursor::pos()))
		{
			case 3:
				slotOpen();
				break;
			case 4:
				slotProperties();
				break;
		}
		delete kpm;
	}
}

void CatView::slotDelCategory()
{
	QStrList strlist;
	
	if (cathandler->getCurrentCategory() != "")
	{
	   if (KMsgBox::yesNo(this,i18n("Message"),i18n("Do you want to delete the current category?"))==1)
   	{
	      cathandler->deleteCategory(cathandler->getCurrentCategory());
      	strlist=cathandler->getCategoryNames();
      	emit categoriesChanged(&strlist);
      	refreshAll();
   	}
	}
}

void CatView::slotRenameCategory()
{
	QString str=cathandler->getCurrentCategory();
	renameDialog dia(this,"dia",i18n("Rename category"),str);
	QStrList strlist;
	
	if (str != "")
	{
   	if (dia.exec())
   	{
   		if (cathandler->renameCategory(str,dia.new_name) == 1)
   		{
      		strlist=cathandler->getCategoryNames();
            emit categoriesChanged(&strlist);
            comboSelectionRequired(&dia.new_name);
            slotCategoryChoice(strlist.find(dia.new_name));
			}
			else
			{
				KMsgBox::message(this,i18n("Message"),
					i18n("Unable to rename the category!\nMaybe there is an already existing category with the new name!"),
					KMsgBox::STOP);
			}
   	}
   }
}

void CatView::slotDeleteCatalog()
{
	KPath kp;
	QString str;
	
   if (KMsgBox::yesNo(this,i18n("Message"),i18n("Do you want to delete the current catalog?"))==1)
  	{
   	kp=visible_branch;
   	while (kp.count() > 1)
   	{
   		kp.pop();
   	}
   	str=*kp.top();
   	
   	cathandler->deleteCatalog(str);
   	refreshAll();
   }
}

void CatView::slotRenameCatalog()
{
	KPath kp;
	QString old_name;
   bool ok=false;
   bool start=true;

	int current_tree_item_index=cattree->currentItem();   	
	
   if (current_tree_item_index != -1)
   {
   	
   	kp=*cattree->itemPath(current_tree_item_index);
   	old_name=cathandler->getCatalogName(kp);
      if (old_name != "")
   	{
			renameDialog dia(this,"dia",i18n("Rename catalog"),old_name);
   	  	while (!ok && start)
      	{
            if (dia.exec())
            {
            	kp.clear();
            	kp.push(&dia.new_name);
   				if (cat->findEntry(kp) == -1)
   				{
   					ok=true;
   				}
   				else
   				{
   					KMsgBox::message(this,i18n("Message"),i18n("The name is already in use!"),KMsgBox::STOP);
   				}
            }
            else
            {
            	start=false;
            }
   		}
   		if (start)
   		{
        		cathandler->renameCatalog(old_name,dia.new_name);
        		refreshAll();
     		}
   	}
   }
}

void CatView::slotMoveCatalog()
{
}

void CatView::slotMove()
{

}

void CatView::slotNew()
{
   NewCatalog dia(this,"dia",prefs->default_mountpoint);
   ScanProgressDlg* prog;
   QString str;
   bool ok=false;
   bool start=true;
   int status;
   KPath kp;

   if (cathandler->getCurrentCategory() != "")
   {
   	while (!ok && start)
   	{
         if (dia.exec())
         {
         	str=dia.sle_catalog_name->text();
         	kp.push(&str);
				if (cat->findEntry(kp) == -1)
				{
					ok=true;
				}
				else
				{
					KMsgBox::message(this,i18n("Message"),i18n("The name is already in use!"),KMsgBox::STOP);
				}
				kp.pop();
         }
         else
         {
         	start=false;
         }
   		if (start && ok)
   		{
            KApplication::setOverrideCursor(waitCursor);
            prog=new ScanProgressDlg(this,"prog");
            prog->show();

   			status=cathandler->newCatalog(dia.sle_catalog_name->text(),dia.sle_root_dir->text());

            delete prog;
            KApplication::restoreOverrideCursor();
   			
            if (status == -1)
            {
            	KMsgBox::message(this,i18n("Message"),i18n("The specified directory (mount point) is empty or invalid!"),KMsgBox::STOP);
   				ok=false;
            }
            else
            {
   				refreshTree();
   			}
     		}
		}
   }
   str=i18n("Ready");
   micro(&str);
}

void CatView::slotNewCategory()
{
   int i;
   NewCategoryDlg dia(this,"dia");
   QString str;
   QStrList strlist;

   if (dia.exec())
   {
      i=cathandler->newCategory(dia.sle_name->text());
      if (i != -1)
      {
      	strlist=cathandler->getCategoryNames();
         emit categoriesChanged(&strlist);
         str=dia.sle_name->text();
         emit comboSelectionRequired(&str);
         slotCategoryChoice(strlist.find((const char*)str));
      }
   }
}
