/*
 * pgp2config.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "pgp2config.h"

#include <qdir.h>
#include <kfiledialog.h>

char* Pgp2Config::FlagStr[] = { "TEXTMODE",
				"ARMOR",
				"KEEPBINARY",
				"COMPRESS",
				"SHOWPASS",
				"CLEARSIG",
				"ENCRYPTTOSELF",
				"INTERACTIVE" };

/** return path to the pgp directory **/
QString 
Pgp2Config::pgpPath() const
{
  return pgp_path;
};

/// stores values
void    Pgp2Config::apply()
{
  int flag;

  for (flag=0; flag<8; flag++)
    {
      if (flags[flag])
	pgp_config->writeEntry(FlagStr[flag], "on");
      else
	pgp_config->writeEntry(FlagStr[flag], "off");
    }

  pgp_config->writeEntry("MYNAME", myname);

  if (pubring_changed)
    {
      pgp_config->writeEntry("PUBRING", pubring);
    }

  pgp_config->writeEntry("SECRING", secring);
  pgp_config->writeEntry("RANDSEED", randseed);
  
  pgp_config->writeEntry("MARGINALS_NEEDED", marginals_needed);
  pgp_config->writeEntry("COMPLETES_NEEDED", completes_needed);
  pgp_config->writeEntry("CERT_DEPTH", cert_depth);

  pgp_config->writeEntry("VERBOSE", verbose);
  pgp_config->writeEntry("COMMENT", comment);

  pgp_config->sync();
};

/** restores default values **/
void    Pgp2Config::setDefault()
{
  QDir myDir;
	
  pubring = pgp_config->readEntry("PUBRING", "");
  if (pubring == "")
    { 
      if (myDir.exists(pgp_path + "/pubring.pgp"))
	pubring = pgp_path + "/pubring.pgp";
      else
	pubring = KFileDialog::getOpenFileName(pgp_path, "*.pgp");

      config->writeEntry("PUBRING", pubring);
      config->sync();
    }
      
  secring = pgp_config->readEntry("SECRING", "");
  if (secring == "")
    {
      if (myDir.exists(pgp_path + "/secring.pgp"))
	pubring = pgp_path + "/secring.pgp";
      else
	secring = KFileDialog::getOpenFileName(pgp_path, "*.pgp");

      config->writeEntry("SECRING", secring);
      config->sync();
    }

  randseed = pgp_config->readEntry("RANDSEED", "");
  if (randseed == "")
    {
      if (myDir.exists(pgp_path + "/randseed.bin"))
	randseed = pgp_path + "/randseed.bin";
      else
	randseed = KFileDialog::getOpenFileName(pgp_path, "*.bin");
	
      config->writeEntry("RANDSEED", randseed);
      config->sync();
    }

  comment = pgp_config->readEntry("COMMENT", "");
  
  myname = pgp_config->readEntry("MYNAME", "");
  
  if (pgp_config->readEntry("INTERACTIVE") == "on")
    flags[Interactive] = TRUE;
  else
    flags[Interactive] = FALSE;
      
  if (pgp_config->readEntry("COMPRESS") == "off")
    flags[Compress] = FALSE;
  else
    flags[Compress] = TRUE;
    
  if (pgp_config->readEntry("KEEPBINARY") == "on")
    flags[Keepbinary] = TRUE;
  else
    flags[Keepbinary] = FALSE;
      
  if (pgp_config->readEntry("ARMOR") == "on")
    flags[Armor] = TRUE;
  else
    flags[Armor] = FALSE;
      
  if (pgp_config->readEntry("TEXTMODE") == "on")
    flags[Textmode] = TRUE;
  else
    flags[Textmode] = FALSE;
      
  if (pgp_config->readEntry("CLEARSIG") == "off")
    flags[Clearsig] = FALSE;
  else
    flags[Clearsig] = TRUE;
      
  if (pgp_config->readEntry("SHOWPASS") == "on")
    flags[Showpass] = TRUE;
  else
    flags[Showpass] = FALSE;
      
  if (pgp_config->readEntry("ENCRYPTTOSELF") == "on")
    flags[EncryptToSelf] = TRUE;
  else
    flags[EncryptToSelf] = FALSE;
  
  verbose  = pgp_config->readNumEntry("VERBOSE");
  if (verbose == 0)
    {
      verbose = 1;
    }
      
  marginals_needed = pgp_config->readNumEntry("MARGINALS_NEEDED");
  if (marginals_needed == 0)
    {
      marginals_needed = 2;
    }
      
  completes_needed = pgp_config->readNumEntry("COMPLETES_NEEDED");
  if (completes_needed == 0)
    {
      completes_needed = 1;
    }
  
  cert_depth = pgp_config->readNumEntry("CERT_DEPTH");
  if (cert_depth == 0)
    {
      cert_depth = 4;
    }
};

/** set pgp flags **/
void    Pgp2Config::setFlag(Flag flag, bool value)
{
  flags[flag] = value;     
};

/** get pgp flags **/
bool    Pgp2Config::getFlag(Flag flag) const
{
  return flags[(int) flag];
};

/** make key the default private key **/
void    Pgp2Config::setDefaultKey(QString myname)
{
  this->myname = myname;
};

/** return the default key **/
QString Pgp2Config::getDefaultKey() const
{
  return myname;
};

void    Pgp2Config::setPubring(QString pubring)
{
  if (this->pubring != pubring)
    {
      this->pubring = pubring;
      pubring_changed = TRUE;
    }
};

QString Pgp2Config::getPubring() const
{
  return pubring;
};

void    Pgp2Config::setSecring(QString secring)
{
  this->secring = secring;
};

QString Pgp2Config::getSecring() const
{
  return secring;
};

void    Pgp2Config::setRandseed(QString randseed)
{
  this->randseed = randseed;
};

QString Pgp2Config::getRandseed() const
{
  return randseed;
};

/** set default pgp comment **/
void    Pgp2Config::setComment(QString comment)
{
  this->comment = comment;
};

/** get default pgp comment **/
QString Pgp2Config::getComment() const
{
  return comment;
};

/** number of marginaly trusted introducers needed **/
void    Pgp2Config::setMarginalsNeeded(int marginals = 2)
{
  marginals_needed = marginals;
};

int     Pgp2Config::getMarginalsNeeded() const
{
  return marginals_needed;
};

/** number of completely trusted introducers needed **/
void    Pgp2Config::setCompletesNeeded(int completes = 1)
{
  completes_needed = completes;
};

int     Pgp2Config::getCompletesNeeded() const
{
  return completes_needed;
};

/** select how deep introducers may be nested **/
void    Pgp2Config::setCertDepth(int depth = 4)
{
  cert_depth = depth;
};

int     Pgp2Config::getCertDepth() const
{
  return cert_depth;
};

void    Pgp2Config::setVerbose(int verbose)
{
  this->verbose = verbose;
};

int     Pgp2Config::getVerbose() const
{
  return verbose;
};


Pgp2Config::Pgp2Config() :
  PgpConfig(),
  pubring_changed(FALSE)
{
  QDir    myDir;

  config->setGroup("PGP2");

  pgp_path = config->readEntry("pgp_path", "");

  if (pgp_path == "")
    {
      if (myDir.exists(QDir::homeDirPath() + "/.pgp"))
	pgp_path = QDir::homeDirPath() + "/.pgp";
      else
	pgp_path = KFileBaseDialog::getDirectory(QDir::homeDirPath());

      if (pgp_path == "")
	exit(0);

      config->writeEntry("pgp_path", pgp_path);
      config->sync();
    }

  pgp_config = new KSimpleConfig(pgp_path + "/config.txt");

  pubring = pgp_config->readEntry("PUBRING", "");

  if (pubring == "")
    { 
      if (myDir.exists(pgp_path + "/pubring.pgp"))
	pubring = pgp_path + "/pubring.pgp";
      else
	pubring = KFileDialog::getOpenFileName(pgp_path, "*.pgp");

      config->writeEntry("PUBRING", pubring);
      config->sync();
      pubring_changed = TRUE;
    }
     
  debug("pubring = " + pubring);

  secring = pgp_config->readEntry("SECRING", "");
  if (secring == "")
    {
      if (myDir.exists(pgp_path + "/secring.pgp"))
	secring = pgp_path + "/secring.pgp";
      else
	secring = KFileDialog::getOpenFileName(pgp_path, "*.pgp");

      config->writeEntry("SECRING", secring);
      config->sync();
    }

  randseed = pgp_config->readEntry("RANDSEED", "");
  if (randseed == "")
    {
      if (myDir.exists(pgp_path + "/randseed.bin"))
	randseed = pgp_path + "/randseed.bin";
      else
	randseed = KFileDialog::getOpenFileName(pgp_path, "*.bin");
	
      config->writeEntry("RANDSEED", randseed);
      config->sync();
    }

  comment = pgp_config->readEntry("COMMENT", "");
  
  myname = pgp_config->readEntry("MYNAME", "");
  
  if (pgp_config->readEntry("INTERACTIVE") == "on")
    flags[Interactive] = TRUE;
  else
    flags[Interactive] = FALSE;
      
  if (pgp_config->readEntry("COMPRESS") == "off")
    flags[Compress] = FALSE;
  else
    flags[Compress] = TRUE;
    
  if (pgp_config->readEntry("KEEPBINARY") == "on")
    flags[Keepbinary] = TRUE;
  else
    flags[Keepbinary] = FALSE;
      
  if (pgp_config->readEntry("ARMOR") == "on")
    flags[Armor] = TRUE;
  else
    flags[Armor] = FALSE;
      
  if (pgp_config->readEntry("TEXTMODE") == "on")
    flags[Textmode] = TRUE;
  else
    flags[Textmode] = FALSE;
      
  if (pgp_config->readEntry("CLEARSIG") == "off")
    flags[Clearsig] = FALSE;
  else
    flags[Clearsig] = TRUE;
      
  if (pgp_config->readEntry("SHOWPASS") == "on")
    flags[Showpass] = TRUE;
  else
    flags[Showpass] = FALSE;
      
  if (pgp_config->readEntry("ENCRYPTTOSELF") == "on")
    flags[EncryptToSelf] = TRUE;
  else
    flags[EncryptToSelf] = FALSE;
  
  verbose  = pgp_config->readNumEntry("VERBOSE");
  if (verbose == 0)
    {
      verbose = 1;
    }
      
  marginals_needed = pgp_config->readNumEntry("MARGINALS_NEEDED");
  if (marginals_needed == 0)
    {
      marginals_needed = 2;
    }
      
  completes_needed = pgp_config->readNumEntry("COMPLETES_NEEDED");
  if (completes_needed == 0)
    {
      completes_needed = 1;
    }
  
  cert_depth = pgp_config->readNumEntry("CERT_DEPTH");
  if (cert_depth == 0)
    {
      cert_depth = 4;
    }
};

Pgp2Config::~Pgp2Config()
{
  delete pgp_config;
};
