/* 
 * keyringtab.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>

#include <qdir.h> 
#include <qfileinf.h>
#include <qfile.h>

#include <kmsgbox.h>
#include <kbuttonbox.h>
#include <ksimpleconfig.h>
#include <kfiledialog.h>

#include "kpgp.h"
#include "pgptreelist.h"
#include "pgp2key.h"
#include "pgp2userid.h"
#include "pgp2signature.h"

#include "keyringtab.h"
#include "keyringtab.moc"

// appearance dialog
KeyringView::~KeyringView()
{
  delete list;
}
    
KeyringView::KeyringView (QWidget * parent, const char *name): 
  KConfigWidget (parent, name),
  defaultKey(0),
  keyTrustDialog(new KTrustSelect("trustselect"))
{
  keyring = PgpKeyring::getPgpKeyring();
  
  // init keylist
  list = new PgpTreeList(this, "");
  connect(list, SIGNAL(leftMenuSelected(QPoint, int)), this, SLOT(slotLeftButtonSelected(QPoint, int)));
  connect(list, SIGNAL(rightMenuSelected(QPoint, int)), this, SLOT(slotRightButtonSelected(QPoint, int)));
  list->move(10, 10);

  // init menus
  infoMenu = new KPopupMenu("Info Menu", 0);
  
  keyMenu = new KPopupMenu("Key Action", 0);
  connect(keyMenu, SIGNAL(activated(int)), this, SLOT(slotKeyMenuSelected(int)));

  userIdMenu = new KPopupMenu("User-ID Action", 0);
  connect(userIdMenu, SIGNAL(activated(int)), this, SLOT(slotUserIdMenuSelected(int)));

  signatureMenu = new KPopupMenu("Signature Action", 0);
  connect(signatureMenu, SIGNAL(activated(int)), this, SLOT(slotSignatureMenuSelected(int)));

  key_list.clear();
  // get all keys from keyring
  key_list = keyring->findKey();
  
  loadSettings();
}
   
/// Loads the settings, usually from an rc-file.
void 
KeyringView::loadSettings()
{
  PgpKeyItem        *keyItem;
  PgpKey            *key;
  
  list->setAutoUpdate(FALSE);
  list->clear();

  // init key-list
  for(key = key_list.first(); key != 0; key = key_list.next())
    {
      keyItem = new PgpKeyItem(key);
      if (key->isDefault())
	defaultKey = keyItem;
	 
      list->insertItem(keyItem);
    }

  list->setAutoUpdate(TRUE);
};


/// Applies the settings.
void 
KeyringView::applySettings()
{
  keyring->write();

  PgpConfig::getPgpConfig()->apply();
};

/// Resets the default values
void 
KeyringView::defaultSettings() 
{ 
  keyring->reload();

  key_list.clear();
  // get all keys from keyring
  key_list = keyring->findKey();
  
  loadSettings();

  debug("KeyringView::defaultValues()"); 
};
 

// ------------------------- SLOTS ----------------------------

/// emittet if config file has changed
void 
KeyringView::slotPubringChanged()
{
  debug("KeyringView::slotConfigChanged()");

  key_list.clear();

  // keyring has to reload the keys
  keyring->reload();

  // get all keys from keyring
  key_list = keyring->findKey();
  
  loadSettings();
};


void 
KeyringView::slotLeftButtonSelected(QPoint pos, int branch)
{
  switch (branch)
    {
    case 0:
      ((PgpKeyItem*) list->getCurrentItem())->setLeftMenu(infoMenu);
      infoMenu->move(pos);
      infoMenu->show();
      break;
    default:
      ;
    }
};

void 
KeyringView::slotRightButtonSelected(QPoint pos, int branch)
{
  switch (branch)
    {
    case 0:
      ((PgpKeyItem*) list->getCurrentItem())->setRightMenu(keyMenu);
      keyMenu->move(pos);
      keyMenu->show();
      break;
    case 1:
      ((PgpUserIdItem*) list->getCurrentItem())->setRightMenu(userIdMenu);
      userIdMenu->move(pos);
      userIdMenu->show();
      break;
    case 2:
      ((PgpSignatureItem*) list->getCurrentItem())->setRightMenu(signatureMenu);
      signatureMenu->move(pos);
      signatureMenu->show();
      break;
    default:
      debug("default");
    }
};
  
void 
KeyringView::resizeEvent(QResizeEvent *)
{
  list->resize(width()-20, height()-20);
}


void 
KeyringView::slotKeyMenuSelected(int menu)
{
  PgpKeyItem *item = (PgpKeyItem *) list->getCurrentItem();
  PgpKey     *key;

  switch (menu)
    {
    case MENU_KEY_EXTRACT:
      {
	QString filename = KFileDialog::getSaveFileName(QDir::homeDirPath(), "*.asc");
	if (filename)
	  {
	    if (!QFile::exists(filename))
	      item->getKey()->extractKey(filename);
	    else
	      debug("FIXME: Can't extract key, file exists");
	  }
	break;
      }

      // public key actions
    case MENU_PUBKEY_EDIT:
      {
	UINT8 trust = item->getKey()->getTrust();

	keyTrustDialog->setTrust(trust);
	if (keyTrustDialog->exec())
	  {
	    if (trust != keyTrustDialog->getTrust())
	      {
		trust = keyTrustDialog->getTrust();
		item->setTrust(trust);
		// now reload the treelist, to update all signatures made with this key
		loadSettings();
	      }
	  }

	break;
      }
    case MENU_PUBKEY_PAUSE:
      item->pauseKey();
      list->update();
      break;
    case MENU_PUBKEY_DELETE:
      key = item->getKey();
      if (1 == KMsgBox::yesNo(0, "Deleting Key from Keyring", "Do you really want to delete key\n" + key->getUserId()->userId() + " ?"))
	{
	  list->removeItem(list->currentItem());
	  key_list.remove(key);
	  PgpKeyring::getPgpKeyring()->removeKey(key);
	}
      else 
	return;
      break;
    case MENU_PUBKEY_CERT:
      {
	PgpSignature *newSignature;

	if (defaultKey == 0)
	  {
	    if (1 == KMsgBox::yesNo(0, "Key Certification", "No default private key selected.\nSelect a private key?"))
	      {
		debug("FIXME: select a private key");
		return;
	      }
	    else
	      return;
	  }
	
	// check, if key is already signed by the default key
	int i;
	
	for (i=0; i<item->getKey()->getUserId()->count(); i++)
	  {
	    
	    if (item->getKey()->getUserId()->signature(i)->keyId() == defaultKey->getKey()->keyId())
	      {
		if (1 == KMsgBox::yesNo(0, "Key Certification", "User-ID already signed by this key:\n--- " + item->getKey()->getUserId()->userId() + " ---\nSelect another private key?"))
		  {
		    debug("FIXME: select a private key");
		    return;
		  }
		else
		  return;
	      }
	  }
	
	newSignature = defaultKey->getKey()->sign(item->getKey());
	
	if (newSignature != 0)
	  {
	    item->addSignature(newSignature);
	    list->update();
	  }
	else
	  KMsgBox::message(0, "Error!", "Error while signing a key.\nMaybe kpgp does not work correct.");
 
	break; 
      }
  
      // private key actions
    case MENU_PRIVKEY_EDIT:
      {
	debug("FIXME: a better dialog should be used here");
	LineDialog *ld = new LineDialog(this, "Add New User-ID");
	if (ld->exec())
	  if (item->newUserId(ld->getUserId()))
	    list->update();
	  else
	    debug("ERROR: error while signing a private key");
       
	delete ld;
	break;
      }
    case MENU_PRIVKEY_REVOKE:  
      item->revokeKey();
      break;
    case MENU_PRIVKEY_DELETE:
      key = item->getKey();
      if (1 == KMsgBox::yesNo(0, "Delete Private Key", "Do you really want to remove your private key\n" + key->getUserId()->userId() + "\nfrom keyring?"))
	  {
	    list->removeItem(list->currentItem());
	    key_list.remove(key);
	    PgpKeyring::getPgpKeyring()->removeKey(key);
	  }
      break;
    case MENU_PRIVKEY_PASS:
      break;
    case MENU_PRIVKEY_DEFAULT:
      if (defaultKey)
	{
	  defaultKey->setDefault(FALSE);
	  debug("default unset");
	}
      defaultKey = item;
      PgpConfig::getPgpConfig()->setDefaultKey(defaultKey->getKey()->getUserId()->userId());
      item->setDefault(TRUE);
      list->update();
      break;
    default:
      debug("WARNING: unknown key action selected");
    }
};

void KeyringView::slotUserIdMenuSelected(int menu)
{
  PgpUserIdItem *item = (PgpUserIdItem *) list->getCurrentItem();

 switch (menu)
    {
    case MENU_USERID_DELETE:
      {
	PgpKeyItem *keyItem = ((PgpKeyItem*) item->getParent());
	
	if (item != keyItem->getChild())
	  {
	    if (1 == KMsgBox::yesNo(0, "Delete User-ID", "Do you really want to delete this User-ID?:\n" + item->getUserId()->userId()))
	      { 
		// keyItem->getKey()->removeUserId(item->getUserId());
		keyItem->removeUserId(item);
		list->removeItem(list->currentItem());
		list->update();
	      }
	  }
	else
	  KMsgBox::message(0, "Message", "The default UserID can`t be deleted");
      break;
      }
    } 
};

void KeyringView::slotSignatureMenuSelected(int menu)
{
  PgpSignatureItem *item = (PgpSignatureItem *) list->getCurrentItem();

  switch (menu)
    {
    case MENU_SIGNATURE_DELETE:
      {
	PgpUserIdItem *uItem = ((PgpUserIdItem*) item->getParent());
	if (1 == KMsgBox::yesNo(0, "Delete Signature", "Do you really want to delete this Signature?:\n" + item->getSignature()->keyId()))
	  { 
	    uItem->getUserId()->removeSignature(item->getSignature());
	    list->removeItem(list->currentItem());
	    uItem->update();
	    list->update();
	  }
	else
	  return;
	break;
      }
    default:
      debug("Error: unknown menu selected");
    }  
};
  

/*
    case MENU_PRIVKEY_REVOKE:  
      item = (PgpTreeListItem*) list->getCurrentItem();
      key = (PgpKey*) item->getPacket();
      key->setRevoked();
      item->setPixmap(&getPixmap(key));
      list->update();
      break;
*/


// -----------------------------------------------------------------------

LineDialog::LineDialog(QWidget *parent, const char *name)
  : QDialog(parent, 0, TRUE)
{
  setCaption(name);
  setFixedSize(264,80);
  
  QLabel *text = new QLabel(i18n("Please enter the new User-ID"),this);
  text->move(30,4);
  text->setAutoResize(TRUE);
  
  lineedit = new QLineEdit(this);
  lineedit->move(30, 8+text->size().height());
  lineedit->resize(200, lineedit->size().height());
  lineedit->setFocus();

  connect(lineedit,SIGNAL(returnPressed()),this,SLOT(accept()) );  
}

LineDialog::~LineDialog()
{}

QString
LineDialog::getUserId()
{
  return lineedit->text();
}


