/* -*- C++ -*-
 * This file implements the main view of the kardinfo utility.
 * kardinfo is used to manage PCMCIA slots from the KDE desktop.
 * 
 * copyright:  (C) Mirko Sucker, 1999
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@unibw-hamburg.de>
 * requires:   recent C++-compiler, at least Qt 1.4
 * $Revision: 1.3 $
 */

#include "kardinfo.h"
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <kledlamp.h>
#include <kiconloader.h>
#include <kapp.h>

/* ATTENTION: Please uncomment the next two lines for KDE 2. */
// #include <klocale.h>
// #include <kglobal.h>
/* ^^^ Until here. */

extern "C" {
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <assert.h>
#include <sys/file.h>
  // ----- some headers from the pcmcia package:
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
}

// ----- typedefs taken from cardinfo.c:

typedef struct event_tag_t {
    event_t event;
    char *name;
} event_tag_t;

static event_tag_t event_tag[] = {
  { CS_EVENT_CARD_INSERTION, "card insertion" },
  { CS_EVENT_CARD_REMOVAL, "card removal" },
  { CS_EVENT_RESET_PHYSICAL, "prepare for reset" },
  { CS_EVENT_CARD_RESET, "card reset successful" },
  { CS_EVENT_RESET_COMPLETE, "reset request complete" },
  { CS_EVENT_EJECTION_REQUEST, "user eject request" },
  { CS_EVENT_INSERTION_REQUEST, "user insert request" },
  { CS_EVENT_PM_SUSPEND, "suspend card" },
  { CS_EVENT_PM_RESUME, "resume card" },
  { CS_EVENT_REQUEST_ATTENTION, "request attention" },
};

#define NTAGS (sizeof(event_tag)/sizeof(event_tag_t))

static char *stabfile = "/var/run/stab";

// -----

// #############################################################################
// MOC OUTPUT FILES:
#include "kardinfo.moc"
// #############################################################################

#define GRID 3

KITab::KITab(QWidget* parent, const char* name)
  : QWidget(parent, name),
    content(new QLabel(i18n("(empty)"), this)),
    line(new QFrame(this)),
    labelState(new QLabel(i18n("No card."), this)),
    cd(new KLedLamp(this)),
    labelCd(new QLabel(i18n("card detected"), this)),
    power(new KLedLamp(this)),
    labelPower(new QLabel(i18n("power up"), this)),
    pp(new KLedLamp(this)),
    labelPp(new QLabel(i18n("programing power"), this)),
    wp(new KLedLamp(this)),
    labelWp(new QLabel(i18n("write protected"), this)),
    details(new QPushButton(i18n("&Details"), this)),
    actions(new QPushButton(this)),
    menu(new QPopupMenu),
    state((s_state)-1) // this is no valid state
{
  // ############################################################################
  if(content==0 || line==0 || labelState==0 || cd==0 || labelCd==0 || power==0 ||
     labelPower==0 || pp==0 || labelPp==0 || wp==0 || labelWp==0 || details==0)
    {
      QMessageBox::critical
	(this, i18n("kardinfo: Critical error"),
	 i18n("Out of memory!\nkardinfo will exit now."));
      kapp->quit();
    }
  // -----
  content->setAlignment(AlignRight);
  line->setFrameStyle(QFrame::HLine | QFrame::Plain);
  labelState->setFrameStyle(QFrame::Box | QFrame::Sunken);
  labelState->setAlignment(AlignRight | AlignVCenter);
  cd->setState(KLedLamp::On);
  power->setState(KLedLamp::On);
  pp->setState(KLedLamp::Off);
  wp->setState(KLedLamp::Off);
  details->setEnabled(false);
  actions->setPixmap(Icon("mini/kcmmemory.xpm"));
  QToolTip::add(actions, i18n("Card actions."));
  QToolTip::add(details, i18n("Card details."));
  setState(S_EMPTY);
  // ----- create the popup menu items:
  menu->insertItem(Icon("reload.xpm"), i18n("Reset card."));
  menu->insertItem(Icon("down.xpm"), i18n("Suspend card."));  
  menu->insertItem(Icon("up.xpm"), i18n("Resume card."));
  menu->insertItem(Icon("delete.xpm"), i18n("Eject card."));
  menu->insertItem(Icon("tick.xpm"), i18n("Insert card."));
  // menu->insertSeparator();
  // -----
  connect(details, SIGNAL(clicked()), SLOT(detailsClickedSlot()));
  connect(actions, SIGNAL(clicked()), SLOT(actionsClickedSlot()));
  connect(menu, SIGNAL(activated(int)), SLOT(menuActivatedSlot(int)));
  // ############################################################################
}

  /** The destructor. */
KITab::~KITab()
{
  // ############################################################################
  // ############################################################################
}

QSize 
KITab::sizeHint()
{
  // ############################################################################
  int cx, cy, tempx;
  QSize size;
  // ----- the width of the headline:
  size=content->sizeHint();
  cx=size.width();
  cy=size.height();
  // ----- 4 LEDs with descriptions follow:
  size=labelCd->sizeHint();
  cy+=4*GRID+4*size.height();
  tempx=QMAX(size.width(), labelPower->sizeHint().width());
  tempx=QMAX(tempx, labelPp->sizeHint().width());
  tempx=QMAX(tempx, labelWp->sizeHint().width());
  tempx+=GRID+QMAX(8, (int)(2.0/3)*size.height()); // the lamps
  cx=QMAX(cx, tempx);
  // ----- the buttons:
  size=actions->sizeHint();
  cx=QMAX(cx, size.width());
  tempx=size.height();
  size=details->sizeHint();
  cy+=GRID+QMAX(size.height(), tempx);
  cx+=GRID+size.width();
  // ----- the state frame:
  size=labelState->sizeHint();
  cy+=GRID+size.height();
  cx=QMAX(cx, size.width());
  // -----
  return QSize(cx+2*GRID, cy+2*GRID);
  // ############################################################################
}

void 
KITab::resizeEvent(QResizeEvent*)
{
  // ############################################################################
  const int ChildWidth=width()-2*GRID;
  int cx=GRID, cy=GRID, tempx, tempy;
  // ----- place the description label:
  tempy=content->sizeHint().height();
  content->setGeometry(cx, cy, ChildWidth, tempy);
  cy+=tempy+GRID;
  // ----- place the horizontal line:
  tempy=line->sizeHint().height();
  line->setGeometry(cx, cy, ChildWidth, tempy);
  cy+=tempy+GRID;
  // ----- now place the status LEDs:
  tempy=labelCd->sizeHint().height(); // same height for all labels
  tempx=QMAX(8, (int)(2.0/3)*tempy);
  cd->setGeometry(cx, cy, tempx, tempy);
  labelCd->setGeometry(cx+tempx+GRID, cy, ChildWidth-cx-tempx-2*GRID, tempy);
  cy+=tempy+GRID;
  power->setGeometry(cx, cy, tempx, tempy);
  labelPower->setGeometry(cx+tempx+GRID, cy, ChildWidth-cx-tempx-2*GRID, tempy);
  cy+=tempy+GRID;
  pp->setGeometry(cx, cy, tempx, tempy);
  labelPp->setGeometry(cx+tempx+GRID, cy, ChildWidth-cx-tempx-2*GRID, tempy);
  cy+=tempy+GRID;
  wp->setGeometry(cx, cy, tempx, tempy);
  labelWp->setGeometry(cx+tempx+GRID, cy, ChildWidth-cx-tempx-2*GRID, tempy);
  // ----- now show the state on the bottom:
  tempy=labelState->sizeHint().height();
  labelState->setGeometry(cx, height()-GRID-tempy, ChildWidth, tempy);
  if(height()-cy<tempy+GRID)
    {
      debug("KITab::resizeEvent: size hint not respected, window too small.");
    }
  // ----- display the "Details" and the "Actions" button above the state frame:
  details->adjustSize();
  actions->adjustSize();
  tempx=QMAX(details->height(), actions->height());
  actions->resize(tempx, tempx);
  details->resize(details->width(), tempx);
  tempy=height()-2*GRID-labelState->height()-tempx;
  details->move(width()-GRID-details->width(), tempy);
  actions->move(GRID, tempy);
  // ############################################################################
}

void 
KITab::detailsClickedSlot()
{
  // ############################################################################
  QString text;
  // -----
  text.sprintf
    (i18n("%s\n"
	  "Interrupt: %s\nDevice(s): %s\nIO Ports: %s"),
     (const char*)card, (const char*)irq, (const char*)dev, (const char*)io);
  QMessageBox::information(this, i18n("kardinfo: card specific details"), text);
  // ############################################################################
}

void 
KITab::actionsClickedSlot()
{
  // ############################################################################
  menu->exec(actions->mapToGlobal(QPoint(4, actions->height())));
  // ############################################################################
}

void 
KITab::menuActivatedSlot(int i)
{
  // ############################################################################
  emit(menuActivated(this, i));
  // ############################################################################
}

void 
KITab::setCardDevice(const char* device)
{
  // ############################################################################
  dev= ( device==0 ? "" : device );
  if(!dev.isEmpty())
    {
      dev=(QString)"/dev/"+dev;
    }
  // ############################################################################
}

void 
KITab::setCardDescription(const char* desc)
{
  // ############################################################################
  card= ( desc==0 ? "" : desc );
  if(card=="empty")
    {
      content->setText(i18n("Empty."));
    } else {
      content->setText(card);
    }
  // ############################################################################
}

void 
KITab::setState(s_state s)
{
  // ############################################################################
  int count;
  // -----
  if(s==state)
    {
      return;
    }
  for(count=0; (unsigned)count<menu->count(); ++count)
    {
      menu->setItemEnabled(count, false);
    }
  state=s;
  switch(state)
    {
    case S_EMPTY:
      labelState->setText(i18n("Empty."));
      // the whole menu will disabled below...
      break;
    case S_PRESENT:
      labelState->setText(i18n("Card present."));
      menu->setItemEnabled(4, true);
      break;
    case S_READY:
      labelState->setText(i18n("Card ready."));
      menu->setItemEnabled(0, true);
      menu->setItemEnabled(1, true);
      menu->setItemEnabled(3, true);
      break;
    case S_BUSY:
      labelState->setText(i18n("Card busy."));
      menu->setItemEnabled(0, true);
      menu->setItemEnabled(1, true);
      menu->setItemEnabled(3, true);
      break;
    case S_SUSPEND:
      labelState->setText(i18n("Card suspended."));
      menu->setItemEnabled(2, true);
      menu->setItemEnabled(3, true);
      break;
    default:
      labelState->setText(i18n("No card found."));
    }
  // WORK_TO_DO: enable/disable actions menu items, see cardinfo.c
  if(state==S_EMPTY || state==S_BUSY) // card are "busy" after an eject request
    {
      details->setEnabled(false);
      actions->setEnabled(false);
      labelCd->setEnabled(false);
      labelPower->setEnabled(false);
      labelPp->setEnabled(false);
      labelWp->setEnabled(false);
    } else {
      details->setEnabled(true);
      actions->setEnabled(true);
      labelCd->setEnabled(true);
      labelPower->setEnabled(true);
      labelPp->setEnabled(true);
      labelWp->setEnabled(true);
    }
  // ############################################################################
}

s_state
KITab::getState()
{
  // ############################################################################
  return state;
  // ############################################################################
}

void 
KITab::setCardPorts(const char* ports)
{
  // ############################################################################
  io= ( ports==0 ? "" : ports );
  // ############################################################################
}

void 
KITab::setCardIRQ(const char* lines)
{
  // ############################################################################
  irq= ( lines==0 ? "" : lines );
  // ############################################################################
}

void
KITab::setCardFlags(bool cd_, bool power_, bool pp_, bool wp_)
{
  // ############################################################################
  cd->setState(cd_ ? KLedLamp::On : KLedLamp::Off);
  pp->setState(pp_ ? KLedLamp::On : KLedLamp::Off);
  wp->setState(wp_ ? KLedLamp::On : KLedLamp::Off);
  power->setState(power_ ? KLedLamp::On : KLedLamp::Off);
  // ############################################################################
}

KardInfo::KardInfo(QWidget* parent, const char* name)
 : QFrame(parent, name),
   ns(0),
   tabctl(new KTabCtl(this)),
   timer(new QTimer(this))
{
  // ############################################################################
  int major;
  servinfo_t serv;
  // ----- initialize the slots:
  // ----- get effective user id to access the sockets:
  //       !!! THIS IS DANGEROUS CODE !!!
  if(geteuid()!=0)
    {
      QMessageBox::critical
	(this, i18n("kardinfo: Critical error"),
	 i18n("You do not have sufficient permissions to use kardinfo.\n"
	      "Contact your system administrator and tell him that\n"
	      "kardinfo must have set the sticky bit."));
      ::exit(0);
    }
  debug("KardInfo::KardInfo: got effective uid 0.");
  // -----
  major=lookupDevice("pcmcia");
  for(ns=0; ns<MAX_SOCKETS; ns++)
    {
      fd[ns]=openDevice((major<<8)+ns);
      if(fd[ns]<0)  
	{
	  break;
	}
      debug("KardInfo::KardInfo: slot %i.", ns);
    }
  if(ns==0)
    {
      QMessageBox::critical
	(this, i18n("kardinfo: Critical error"),
	 i18n("No PCMCIA sockets found."));
      setuid(getuid());
      debug("KardInfo::KardInfo: got real user id.");      
      ::exit(0);
    }
  // ----- now check card services release:
  if(ioctl(fd[0], DS_GET_CARD_SERVICES_INFO, &serv)==0)
    { // WORK_TO_DO: TEST DISABLED
      if(serv.Revision!=CS_RELEASE_CODE && false)
	{
	  debug("KardInfo::KardInfo: CS_RELEASE_CODE is %x, queried revision"
		" is %x.", CS_RELEASE_CODE, serv.Revision);
	  QMessageBox::critical
	    (this, i18n("kardinfo: Critical error"),
	     i18n("The Card Services release in your computer does\n"
		  "not work with this software."));
	  setuid(getuid());
	  debug("KardInfo::KardInfo: got real user id.");      
	  ::exit(EXIT_FAILURE);
	}
    } else {
      QMessageBox::critical
	    (this, i18n("kardinfo: Critical error"),
	     i18n("Cannot query Card Services revision information."));
      /* Is this necessary? 
       * I think so, mostly to ensure safety in the code closing the 
       * application. */
      setuid(getuid()); 
      debug("KardInfo::KardInfo: got real user id.");      
      ::exit(EXIT_FAILURE);
    }
  debug("KardInfo::KardInfo: got CS revision information.");
  // ----- switch back to real user id:
#ifndef UNSAFE_TOOLS
  setuid(getuid());
  debug("KardInfo::KardInfo: got real user id.");
#else
  debug("KardInfo::KardInfo: careful, this tool runs as root!");
#endif
  //       !!! END OF POSSIBLY DANGEROUS CODE !!!
  // ----- 
  if(tabctl==0 || timer==0)
    {
      QMessageBox::critical
	(this, i18n("kardinfo: Critical error"),
	 i18n("Out of memory!\nkardinfo will exit now."));
      kapp->quit();
    }
  // -----
  setFrameStyle(QFrame::Box | QFrame::Sunken); // for now
  // ----- fill the tabctl with some examples:
  for(int i=0; i<ns; ++i)
    {
      KITab *tab;
      QString temp;
      // ----
      temp.sprintf(i18n("Slot %i"), i);
      tab=new KITab(tabctl, temp);
      CHECK_PTR(tab);
      connect(tab, SIGNAL(menuActivated(KITab*, int)), 
	      SLOT(menuActivatedSlot(KITab*, int)));
      tabctl->addTab(tab, temp);
      tabs[i]=tab;
    }
  // ----- misc stuff:
  connect(kapp, SIGNAL(appearanceChanged()), SLOT(initializeGeometry()));
  connect(timer, SIGNAL(timeout()), SLOT(updateCardData()));
  initializeGeometry();
  resize(minimumSize());
  // ----- start the timer to update the card information every 200 msec:
  timer->start(200, true); // single shot, restarted after each update
  // ############################################################################
}

int
KardInfo::lookupDevice(const char* name)
{
  // ############################################################################
  FILE *f;
  int n;
  char s[32], t[32];
  // -----
  f=fopen("/proc/devices", "r");
  if(f==0)
    {
      return -errno;
    }
  while(fgets(s, 32, f)!=0)
    {
      if(sscanf(s, "%d %s", &n, t)==2)
	{
	  if(strcmp(name, t)==0) break;
	}
    }
  fclose(f);
  if(strcmp(name, t)==0)
    {
      return n;
    } else {
      return -ENODEV;
    }
  // ############################################################################
}

int
KardInfo::openDevice(dev_t dev)
{
  // ############################################################################
  char *fn;
  int fd;
  // -----
  if((fn=tmpnam(0))==0)
    {
      return -1;
    }
  if(mknod(fn, (S_IFCHR | S_IREAD), dev)!=0)
    {
      return -1;
    }
  fd=open(fn, O_RDONLY);
  if(fd<0)
     {
       unlink(fn);
       return -1;
     }
  if(unlink(fn)!=0)
    {
      close(fd);
      return -1;
    }
  return fd;
  // ############################################################################
}

KardInfo::~KardInfo()
{
  // ############################################################################
  // ############################################################################
}

QSize 
KardInfo::sizeHint()
{
  // ############################################################################
  return QSize(0, 0);
  // ############################################################################
}

void 
KardInfo::initializeGeometry()
{
  // ############################################################################
  int count;
  QSize size, maxtab;
  // -----
  for(count=0; count<ns; ++count)
    {
      size=tabs[count]->sizeHint();
      maxtab=QSize(QMAX(maxtab.width(), size.width()), 
		   QMAX(maxtab.height(), size.height()));
    } 
  // ----- 
  setMinimumSize(maxtab.width()+64, maxtab.height()+64);
  resize(minimumSize()); // trigger a resize event
  // ############################################################################
}

void 
KardInfo::updateCardData()
{
  // ############################################################################
  // debug("KardInfo::updateCardData: called.");
  int i, j, event, ret, state_;
  fd_set fds;
  time_t now;
  static time_t last=0;
  struct tm *tm;
  struct timeval timeout;
  cs_status_t status;
  FILE *f;
  char s[80], d[80], io[20], irq[4];
  char *t;
  ioaddr_t stop;
  config_info_t cfg;
  struct stat buf;
  // -----
  FD_ZERO(&fds); // reset file descriptors
  for(i=0; i<ns; i++)
    {
      FD_SET(fd[i], &fds);
    }
  timeout.tv_sec=0;
  timeout.tv_usec=0;
  ret=select(MAX_SOCKETS+4, &fds, 0, 0, &timeout);
  now=time(0);
  tm=localtime(&now);
  // -----
  if(ret>0)
    {
      for(i=0; i<ns; i++)
	{
	  if(!FD_ISSET(fd[i], &fds))
	    {
	      continue;
	    }
	  ret=read(fd[i], &event, 4);
	  if(ret!=4)
	    {
	      continue;
	    }
	  for(j=0; (unsigned)j<NTAGS; j++)
	    {
	      if(event_tag[j].event==(unsigned)event)
		{
		  break;
		}
	    }
	  if(j==NTAGS)
	    {
	      sprintf(s, "%2d:%02d:%02d socket %d: unknown event 0x%x",
		      tm->tm_hour, tm->tm_min, tm->tm_sec, i, event);
	    } else {
	      sprintf(s, "%2d:%02d:%02d socket %d: %s", tm->tm_hour, 
		      tm->tm_min, tm->tm_sec, i, event_tag[j].name);
	    }
	  // debug("KardInfo::updateCardData: event: %s.", s);
	  // WORK_TO_DO: add s to the event log
	}
    }
  if((stat(stabfile, &buf)==0) && (buf.st_mtime>=last))
    {
      f=fopen(stabfile, "r");
      if(f==0)
	{
	  return;
	}
      if(flock(fileno(f), LOCK_SH)!=0)
	{
	  alert(i18n("kardinfo: Error"), 
		i18n("Could not open the socket table file."));
	  return;
	}
      last=now;
      fgetc(f);
      for(i=0; i<ns; i++)
	{
	  if(!fgets(s, 80, f))
	    {
	      break;
	    }
	  s[strlen(s)-1]=0;
	  // debug("KardInfo::updateCardData: socket %i: %s.", i, s+9);
	  tabs[i]->setCardDescription(s+9);
	  *d=0;
	  for(;;)
	    {
	      int c=fgetc(f);
	      if((c==EOF) || (c=='S'))
		{
		  break;
		} else {
		  fgets(s, 80, f);
		  for(t=s, j=0; j<4; j++)
		    {
		      t=strchr(t, '\t')+1;
		    }
		  t[strcspn(t, "\t\n")]=0;
		  if(*d==0)
		    {
		      strcpy(d, t);
		    } else {
		      strcat(d, ", ");
		      strcat(d, t);
		    }
		}
	      // debug("KardInfo::updateCardData: socket %i: device: %s.", i, d);
	      tabs[i]->setCardDevice(d);
	    }
	}
      flock(fileno(f), LOCK_UN);
      fclose(f);
    }
  // -----
  for(i=0; i<ns; i++)
    {
      state_=S_EMPTY;
      status.Function=0;
      ioctl(fd[i], DS_GET_STATUS, &status);
      if(tabs[i]->getState()==S_EMPTY)
	{
	  if(status.CardState & CS_EVENT_CARD_DETECT)
	    {
	      tabs[i]->setState(S_PRESENT);
	    }
	} else {
	  if(status.CardState & CS_EVENT_PM_SUSPEND)
	    {
	      tabs[i]->setState(S_SUSPEND);
	    } else {
	      if(status.CardState & CS_EVENT_READY_CHANGE)
		{
		  tabs[i]->setState(S_READY);
		} else {
		  tabs[i]->setState(S_BUSY);
		}
	    }
	}
      // -----
      strcpy(io, "");
      strcpy(irq, "");
      memset(&cfg, 0, sizeof(cfg));
      ret=ioctl(fd[i], DS_GET_CONFIGURATION_INFO, &cfg);
      if(cfg.Attributes & CONF_VALID_CLIENT)
	{
	  if(cfg.AssignedIRQ!=0)
	    {
	      sprintf(irq, "%d", cfg.AssignedIRQ);
	    }
	  // debug("KardInfo::updateCardData: socket %i: irq: %s.", i, irq);
	  tabs[i]->setCardIRQ(irq);
	  if(cfg.NumPorts1>0)
	    {
	      stop=cfg.BasePort1+cfg.NumPorts1;
	      if(cfg.NumPorts2>0)
		{
		  if(stop==cfg.BasePort2)
		    {
		      sprintf(io, "%#x-%#x", cfg.BasePort1, stop+cfg.NumPorts2-1);
		    } else {
		      sprintf(io, "%#x-%#x, %#x-%#x", cfg.BasePort1, stop-1,
			      cfg.BasePort2, cfg.BasePort2+cfg.NumPorts2-1);
		    }
		} else {
		  sprintf(io, "%#x-%#x", cfg.BasePort1, stop-1);
		}
	    }
	  // debug("KardInfo::updateCardData: socket %i: io: %s.", i, io);
	  tabs[i]->setCardPorts(io);
	}
      tabs[i]->setCardFlags
	(status.CardState & CS_EVENT_CARD_DETECT, // card detected
	 cfg.Vcc > 0, // power
	 cfg.Vpp1 > 0, // programming power
	 status.CardState & CS_EVENT_WRITE_PROTECT); // write protected
    }
  // -----
  timer->start(300, true);
  // ############################################################################
}

void 
KardInfo::resizeEvent(QResizeEvent*)
{
  // ############################################################################
  tabctl->setGeometry(GRID, GRID, width()-2*GRID, height()-2*GRID);
  // ############################################################################
}

void 
KardInfo::menuActivatedSlot(KITab* emitter, int item)
{
  // ############################################################################
  int index;
  int ret=-1;
  // -----
  for(index=0; index<ns; ++index)
    {
      if(tabs[index]==emitter) break;
    }
  assert(index!=ns); // only our tabs are allowed :-)
  debug("KardInfo::menuActivatedSlot: signal from tab %i.", index);
  switch(item)
    {
    case 0:
      debug("KardInfo::menuActivatedSlot: the reset item.");
      ret=ioctl(fd[index], DS_RESET_CARD); 
      break;
    case 1:
      debug("KardInfo::menuActivatedSlot: the suspend item.");
      ret=ioctl(fd[index], DS_SUSPEND_CARD);
      break;
    case 2:
      debug("KardInfo::menuActivatedSlot: the resume item.");
      ret=ioctl(fd[index], DS_RESUME_CARD); 
      break;
    case 3:
      debug("KardInfo::menuActivatedSlot: the eject item.");
      ret=ioctl(fd[index], DS_EJECT_CARD);
      break;
    case 4:
      debug("KardInfo::menuActivatedSlot: the insert item.");
      ret=ioctl(fd[index], DS_INSERT_CARD); 
      break;
    default:
      debug("KardInfo::menuActivatedSlot: unknown item (unknown action)!");
      qApp->beep();
    }
  if(ret!=0)
    {
      QString text;
      text=i18n("The requested operation failed.");
      text+="\n(";
      text+=strerror(errno);
      text+=")";
      alert(i18n("kardinfo: Error"), text);
    }
  // ############################################################################
}

void 
KardInfo::alert(const char* headline, const char* text)
{
  // ############################################################################
  QMessageBox::critical(this, headline, text);
  // ############################################################################
}

int main(int argc, char** argv)
{
  KApplication *myapp=new KApplication(argc, argv);
  KardInfo *ki=new KardInfo;
  // -----
  myapp->setMainWidget(ki);
  ki->show();
  return myapp->exec();
}
